/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.matomo;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.commons.lang.StringUtils;

import org.ametys.core.cocoon.JSonReader;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Get the page data from matomo
 */
public class GetPageDataAction extends ServiceableAction
{
    /** The Matomo data helper */
    protected MatomoDataHelper _matomoHelper;
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _matomoHelper = (MatomoDataHelper) smanager.lookup(MatomoDataHelper.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Map<String, Object> result = new HashMap<>();
        
        Request request = ObjectModelHelper.getRequest(objectModel);
        String pageUrl = request.getParameter("pageUrl");
        String siteName = WebHelper.getSiteName(request);
        if (StringUtils.isNotBlank(siteName))
        {
            Site site = _siteManager.getSite(siteName);
            result.put("nbLastDays", _matomoHelper.getMatomoSiteNbLastDays(site));
            
            try
            {
                Integer nbVisitsFromLastDays = _matomoHelper.getNbVisitsFromLastDays(site, pageUrl);
                Integer nbTotalVisits = _matomoHelper.getNbTotalVisits(site, pageUrl);
                
                result.put("nbVisits", nbTotalVisits);
                result.put("nbVisitsFromLastDays", nbVisitsFromLastDays);
            }
            catch (MatomoException e) 
            {
                getLogger().error("An error occurred getting matomo data from site '" + site.getTitle() + "' and page URL '" + pageUrl + "'", e);
            }
        }
        
        request.setAttribute(JSonReader.OBJECT_TO_READ, result);
        return EMPTY_MAP;
    }
}
