/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.tracking;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.quartz.JobExecutionContext;

import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.runtime.config.Config;
import org.ametys.web.WebConstants;
import org.ametys.web.analytics.WebAnalyticsHelper;
import org.ametys.web.analytics.WebAnalyticsProvider;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Schedulable to send tracking information
 */
public class WebAnalyticsTrackingSchedulable extends AbstractStaticSchedulable
{
    /** Configuration key to know if sending stats is enabled */
    public static final String SEND_STATS_CONFIG = "send-statistics-enable";
    
    /** The site manager */
    protected SiteManager _siteManager;
    
    /** The web analytics helper */
    protected WebAnalyticsHelper _webAnalyticsHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _webAnalyticsHelper = (WebAnalyticsHelper) manager.lookup(WebAnalyticsHelper.ROLE);
    }
    
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        // Send tracking information only is it's enable in the configuration
        if (Config.getInstance().getValue(SEND_STATS_CONFIG, true, false))
        {
            Request request = ContextHelper.getRequest(_context);
            for (Site site : _siteManager.getSites())
            {
                request.setAttribute(WebConstants.REQUEST_ATTR_SITE, site);
                WebAnalyticsProvider provider = _webAnalyticsHelper.getSelectedProvider(site);
                provider.sendTrackingInformation(site);
            }
        }
    }
}
