/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.matomo.tracking.impl;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ReactionableObject;
import org.ametys.cms.repository.ReactionableObject.ReactionType;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.BooleanExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.webanalytics.matomo.tracking.AbstractMatomoEventTrackingProvider;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * The matomo tracking provider for the {@link Content} like
 */
public class LikeMatomoTrackingProvider extends AbstractMatomoEventTrackingProvider implements Serviceable
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }

    @Override
    protected List<MatomoEvent> getEvents(Site site)
    {
        Map<Page, Integer> nbLikesByPage = new HashMap<>();

        Expression likesExpr = new BooleanExpression("likes", true);
        Expression siteExpr = new StringExpression("site", Operator.EQ, site.getName());
        Expression expr = new AndExpression(likesExpr, siteExpr);
        
        try (AmetysObjectIterable<Content> contents = _resolver.query(ContentQueryHelper.getContentXPathQuery(expr)))
        {
            for (Content content : contents)
            {
                if (content instanceof WebContent webContent && content instanceof ReactionableObject reactionableContent)
                {
                    // Take the first page because we do not want to count the number of like twice or more ...
                    Optional<Page> page = webContent.getReferencingPages()
                        .stream()
                        .findFirst();
                    if (page.isPresent())
                    {
                        nbLikesByPage.compute(page.get(), (k, v) -> 
                            (v == null ? 0 : v) + _getNbLike(reactionableContent)
                        );
                    }
                }
            }
        }
        
        return nbLikesByPage.keySet()
                .stream()
                .map(p -> new MatomoEvent(
                        p.getSitemapName() + "/" + p.getPathInSitemap(), 
                        nbLikesByPage.get(p)))
                .toList();
    }
    
    private int _getNbLike(ReactionableObject reactionableContent)
    {
        return reactionableContent.getReactionUsers(ReactionType.LIKE).size();
    }
}
