/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webanalytics.matomo;

import java.time.Duration;
import java.time.LocalDate;
import java.util.List;
import java.util.Map;
import java.util.Optional;

import org.apache.avalon.framework.activity.Disposable;
import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.hc.client5.http.classic.methods.HttpPost;
import org.apache.hc.client5.http.impl.classic.CloseableHttpClient;
import org.apache.hc.core5.http.io.entity.EntityUtils;
import org.apache.hc.core5.io.CloseMode;

import org.ametys.core.cache.AbstractCacheManager;
import org.ametys.core.cache.Cache;
import org.ametys.core.util.HttpUtils;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.core.util.LambdaUtils.LambdaException;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.core.impl.cache.AbstractCacheKey;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.site.Site;

/**
 * The matomo data helper
 */
public class MatomoDataHelper extends AbstractLogEnabled implements Component, Initializable, Serviceable, Disposable
{
    /** The component role. */
    public static final String ROLE = MatomoDataHelper.class.getName();
    
    /** The Matomo URL config parameter name */
    public static final String MATOMO_URL_CONFIG = "piwik-url";
    
    /** The Matomo token config parameter name */
    public static final String MATOMO_TOKEN_CONFIG = "matomo-token";

    /** The Matomo site id config parameter name */
    public static final String MATOMO_SITE_ID_SITE_CONFIG = "piwik-id";

    private static final String __MATOMO_CACHE_LIFETIME_CONFIG = "matomo-cache-lifetime";
    private static final String __MATOMO_SITE_LIVE_STATS_CONFIG = "matomo-live-stats";
    private static final String __MATOMO_LAST_DAYS_SITE_CONFIG = "matomo-last-days";
    private static final String __MATOMO_BEGIN_YEAR_SITE_CONFIG = "matomo-begin-year";
    
    private static final String __CACHE_COMPUTED_VISITS = "computed-visits";
    private static final String __CACHE_LIVE_VISITS = "live-visits";
    
    /** The cache manager */
    protected AbstractCacheManager _cacheManager;
    
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    
    private CloseableHttpClient _httpClient;
    private String _matomoServerURL;
    private String _matomoToken;
    
    /**
     * The computed matomo request type.
     */
    private enum RequestType
    {
        /** Computed visits from begin year */
        BEGIN_YEAR,
        /** Computed visits of the current day */
        CURRENT_DAY,
        /** Computed visits of from the last days */
        LAST_DAYS;
    }
    
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _cacheManager = (AbstractCacheManager) manager.lookup(AbstractCacheManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
    }
    
    @Override
    public void initialize() throws Exception
    {
        // The duration of 6 hours is an arbitrary value because computed data in Matomo are re-computed every 6 hours. But the cache need to be recalculate only once every day.
        // But if the value is put in the cache at 23pm, we don't want the cache to be recalculate at 23pm the next day ... 6 hours seems to be fine ...
        _cacheManager.createMemoryCache(__CACHE_COMPUTED_VISITS,
                new I18nizableText("plugin.web-analytics", "PLUGINS_WEB_ANALTICS_CACHE_MATOMO_COMPUTED_VISITS_LABEL"),
                new I18nizableText("plugin.web-analytics", "PLUGINS_WEB_ANALTICS_CACHE_MATOMO_COMPUTED_VISITS_DESC"),
                true,
                Duration.ofHours(6));
        
        // Live stats for pages will be computed every n minutes (n is a config parameter)
        Long cacheLifeTime = Config.getInstance().getValue(__MATOMO_CACHE_LIFETIME_CONFIG, true, 5L);
        _cacheManager.createMemoryCache(__CACHE_LIVE_VISITS,
                new I18nizableText("plugin.web-analytics", "PLUGINS_WEB_ANALTICS_CACHE_MATOMO_LIVE_VISITS_LABEL"),
                new I18nizableText("plugin.web-analytics", "PLUGINS_WEB_ANALTICS_CACHE_MATOMO_LIVE_VISITS_DESC"),
                true,
                Duration.ofMinutes(cacheLifeTime));
        
        _httpClient = HttpUtils.createHttpClient(5, 5, false); // Max 5 attempts simultany and 5 sec of timeout
        _matomoServerURL = Config.getInstance().getValue(MATOMO_URL_CONFIG);
        _matomoToken = Config.getInstance().getValue(MATOMO_TOKEN_CONFIG);
    }
    
    private void _checkMatomoConfig(Site site) throws MatomoException
    {
        if (!site.getValue(__MATOMO_SITE_LIVE_STATS_CONFIG, true, false))
        {
            throw new MatomoException("Can't request the matomo stats because the site does not enable it.");
        }
        
        String siteId = site.getValue(MATOMO_SITE_ID_SITE_CONFIG);
        if (StringUtils.isBlank(_matomoServerURL) || StringUtils.isBlank(_matomoToken) || StringUtils.isBlank(siteId))
        {
            throw new MatomoException("One or more of these following parameters are empty: the matamo server URL, the matomo token or the matomo site id. Can't access to matomo data without these parameters");
        }
    }
    
    /**
     * Get the number of days from now to computed the number of visits
     * @param site the site
     * @return the number of days
     */
    public long getMatomoSiteNbLastDays(Site site)
    {
        return site.getValue(__MATOMO_LAST_DAYS_SITE_CONFIG, true, 30L);
    }
    
    /**
     * Get the number of visits since the begin year for the given page and site
     * @param site the site
     * @param pageUrl the page URL
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    public int getNbTotalVisits(Site site, String pageUrl) throws MatomoException
    {
        _checkMatomoConfig(site);

        String siteId = site.getValue(MATOMO_SITE_ID_SITE_CONFIG);
        String beginYear = site.getValue(__MATOMO_BEGIN_YEAR_SITE_CONFIG);
        
        // Number of visits of the current day (Live from Matomo)
        Integer liveVisitsOfDay = _getLiveVisitsOfDay(siteId, pageUrl);
        
        // Number of visits of the current day (Computed from Matomo)
        Integer nbComputedVisitsToday = _getComputedVisitsOfCurrentDay(siteId, pageUrl);
        
        // Number of visits from the begin years but with the computed current day
        Integer nbComputedVisitsFromLastYears = _getComputedVisitsFromBeginYear(siteId, pageUrl, beginYear);
        
        return nbComputedVisitsFromLastYears - nbComputedVisitsToday + liveVisitsOfDay;
    }
    
    /**
     * Get the number of visits from the last days for the given page and site
     * @param site the site
     * @param pageUrl the page URL
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    public int getNbVisitsFromLastDays(Site site, String pageUrl) throws MatomoException
    {
        _checkMatomoConfig(site);
        
        String siteId = site.getValue(MATOMO_SITE_ID_SITE_CONFIG);
        long lastDays = getMatomoSiteNbLastDays(site);
        
        // Number of visits of the current day (Live from Matomo)
        Integer liveVisitsOfDay = _getLiveVisitsOfDay(siteId, pageUrl);

        // Visits from the last n days (starting from yesterday)
        Integer computedVisitsFromLastDays = _getComputedVisitsFromLastDays(siteId, pageUrl, lastDays);
        
        return computedVisitsFromLastDays + liveVisitsOfDay;
    }
    
    /**
     * Get the live number of visits of today for the given page and site
     * @param siteId the site identifier in Matomo
     * @param pageUrl the page URL
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    protected int _getLiveVisitsOfDay(String siteId, String pageUrl) throws MatomoException
    {
        StringBuilder url = new StringBuilder(_matomoServerURL);
        url.append("/?module=API&method=Live.getLastVisitsDetails");
        url.append("&format=JSON");
        url.append("&period=day");
        url.append("&date=today");
        url.append("&idSite=");
        url.append(siteId);
        url.append("&token_auth=");
        url.append(_matomoToken);
        
        try
        {
            List<Map<String, Object>> liveVisits = _getLiveVisitsCache().get(siteId, LambdaUtils.wrap(k -> _requestLiveVisitsOfDay(url.toString())));
            return _getTodayNbVisits(liveVisits, pageUrl);
        }
        catch (LambdaException e)
        {
            if (e.getCause() instanceof RuntimeException re)
            {
                throw re;
            }
            else if (e.getCause() instanceof MatomoException m)
            {
                throw m;
            }
            else
            {
                throw new RuntimeException(e.getCause());
            }
        }
    }
    
    @SuppressWarnings("unchecked")
    private List<Map<String, Object>> _requestLiveVisitsOfDay(String url) throws MatomoException
    {
        return (List<Map<String, Object>>) _executeMatomoRequest(url, "list");
    }
    
    /**
     * Get the computed number of visits from the last years starting from the begin year for the given page and site
     * @param siteId the site identifier in matomo
     * @param pageUrl the page URL
     * @param beginYear the begin year
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    protected int _getComputedVisitsFromBeginYear(String siteId, String pageUrl, String beginYear) throws MatomoException
    {
        String truncateURL = _truncateURL(pageUrl);

        LocalDate beginDate = LocalDate.of(Integer.valueOf(beginYear), 1, 1);
        LocalDate now = LocalDate.now();
        
        int nbYear = now.getYear() - beginDate.getYear() + 1;
        
        StringBuilder url = new StringBuilder(_matomoServerURL);
        url.append("/?module=API&method=Actions.getPageUrl");
        url.append("&format=JSON");
        url.append("&period=month");
        url.append("&date=last");
        url.append(nbYear * 12);
        url.append("&idSite=");
        url.append(siteId);
        url.append("&pageUrl=");
        url.append(URIUtils.encodeParameter(truncateURL));
        url.append("&token_auth=");
        url.append(_matomoToken);
        
        try
        {
            MatomoComputedCacheKey key = MatomoComputedCacheKey.of(RequestType.BEGIN_YEAR, truncateURL);
            return _getComputedVisitsCache().get(key, LambdaUtils.wrap(k -> _requestComputedVisitsFromBeginYear(url.toString())));
        }
        catch (LambdaException e)
        {
            if (e.getCause() instanceof RuntimeException re)
            {
                throw re;
            }
            else if (e.getCause() instanceof MatomoException m)
            {
                throw m;
            }
            else
            {
                throw new RuntimeException(e.getCause());
            }
        }
    }
    
    private Integer _requestComputedVisitsFromBeginYear(String url) throws MatomoException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> json = (Map<String, Object>) _executeMatomoRequest(url, "map");
        return _getNbVisitsFromMap(json);
    }
    
    /**
     * Get the computed number of visits of today for the given page and site
     * @param siteId the site identifier in matomo
     * @param pageUrl the page URL
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    protected int _getComputedVisitsOfCurrentDay(String siteId, String pageUrl) throws MatomoException
    {
        String truncateURL = _truncateURL(pageUrl);

        StringBuilder url = new StringBuilder(_matomoServerURL);
        url.append("/?module=API&method=Actions.getPageUrl");
        url.append("&format=JSON");
        url.append("&period=day");
        url.append("&date=today");
        url.append("&idSite=");
        url.append(siteId);
        url.append("&pageUrl=");
        url.append(URIUtils.encodeParameter(truncateURL));
        url.append("&token_auth=");
        url.append(_matomoToken);
        
        try
        {
            MatomoComputedCacheKey key = MatomoComputedCacheKey.of(RequestType.CURRENT_DAY, truncateURL);
            return _getComputedVisitsCache().get(key, LambdaUtils.wrap(k -> _requestComputedVisitsOfCurrentDay(url.toString())));
        }
        catch (LambdaException e)
        {
            if (e.getCause() instanceof RuntimeException re)
            {
                throw re;
            }
            else if (e.getCause() instanceof MatomoException m)
            {
                throw m;
            }
            else
            {
                throw new RuntimeException(e.getCause());
            }
        }
    }
    
    private Integer _requestComputedVisitsOfCurrentDay(String url) throws MatomoException
    {
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> json = (List<Map<String, Object>>) _executeMatomoRequest(url, "list");
        return _getNbVisitsFromList(json);
    }
    
    /**
     * Get the computed number of visits from the last n days (starting from yesterday) for the given page and site
     * @param siteId the site identifier in matomo
     * @param pageUrl the page URL
     * @param nbDay the number of day
     * @return the number of visits
     * @throws MatomoException if an error with Matomo occurred
     */
    protected int _getComputedVisitsFromLastDays(String siteId, String pageUrl, long nbDay) throws MatomoException
    {
        String truncateURL = _truncateURL(pageUrl);

        StringBuilder url = new StringBuilder(_matomoServerURL);
        url.append("/?module=API&method=Actions.getPageUrl");
        url.append("&format=JSON");
        url.append("&period=day");
        url.append("&date=previous");
        url.append(nbDay);
        url.append("&idSite=");
        url.append(siteId);
        url.append("&pageUrl=");
        url.append(URIUtils.encodeParameter(truncateURL));
        url.append("&token_auth=");
        url.append(_matomoToken);
        
        try
        {
            MatomoComputedCacheKey key = MatomoComputedCacheKey.of(RequestType.LAST_DAYS, truncateURL);
            return _getComputedVisitsCache().get(key, LambdaUtils.wrap(k -> _requestComputedVisitsFromLastDays(url.toString())));
        }
        catch (LambdaException e)
        {
            if (e.getCause() instanceof RuntimeException re)
            {
                throw re;
            }
            else if (e.getCause() instanceof MatomoException m)
            {
                throw m;
            }
            else
            {
                throw new RuntimeException(e.getCause());
            }
        }
    }
    
    private Integer _requestComputedVisitsFromLastDays(String url) throws MatomoException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> json = (Map<String, Object>) _executeMatomoRequest(url, "map");
        return _getNbVisitsFromMap(json);
    }
    
    private Object _executeMatomoRequest(String matomoUrl, String returnType) throws MatomoException
    {
        try
        {
            if (getLogger().isInfoEnabled())
            {
                getLogger().info("A request is send to Matomo with url '" + matomoUrl + "'");
            }
            
            HttpPost post = new HttpPost(matomoUrl);
            return _httpClient.execute(post, httpResponse -> {
                if (httpResponse.getCode() != 200)
                {
                    throw new IllegalStateException("Could not join matomo for URL " + matomoUrl + ". Error code " + httpResponse.getCode());
                }
                
                if ("map".equals(returnType))
                {
                    return _jsonUtils.convertJsonToMap(EntityUtils.toString(httpResponse.getEntity()));
                }
                else if ("list".equals(returnType))
                {
                    return _jsonUtils.convertJsonToList(EntityUtils.toString(httpResponse.getEntity()));
                }
                
                throw new IllegalArgumentException("Invalid return type for request with url '" + matomoUrl + "'");
            });
        }
        catch (Exception e)
        {
            throw new MatomoException("An error occurred requesting Matomo for url '" + matomoUrl + "'", e);
        }
    }
    
    @SuppressWarnings("unchecked")
    private int _getNbVisitsFromMap(Map<String, Object> jsonAsMap)
    {
        int nbVisits = 0;
        for (String key : jsonAsMap.keySet())
        {
            nbVisits += _getNbVisitsFromList((List<Map<String, Object>>) jsonAsMap.get(key));
        }
        
        return nbVisits;
    }
    
    private int _getNbVisitsFromList(List<Map<String, Object>> data)
    {
        if (!data.isEmpty())
        {
            return (Integer) data.get(0).getOrDefault("nb_visits", 0);
        }
        
        return 0;
    }
    
    private int _getTodayNbVisits(List<Map<String, Object>> jsonAsMap, String pageUrl)
    {
        int nbVisits = 0;
        if (!jsonAsMap.isEmpty())
        {
            String truncateURL = _truncateURL(pageUrl);
            
            LocalDate today = LocalDate.now();
            for (Map<String, Object> visitInfo : jsonAsMap)
            {
                String visitDateAsString = (String) visitInfo.get("serverDate");
                LocalDate visitDate = LocalDate.parse(visitDateAsString);
                if (visitDate.equals(today))
                {
                    @SuppressWarnings("unchecked")
                    List<Map<String, Object>> actions = (List<Map<String, Object>>) visitInfo.get("actionDetails");
                    Optional<String> pageVisits = actions.stream()
                            .map(a -> (String) a.get("url"))
                            .map(this::_truncateURL)
                            .filter(url -> truncateURL.equals(url))
                            .findAny();
                    
                    if (pageVisits.isPresent())
                    {
                        nbVisits++;
                    }
                }
            }
        }
        return nbVisits;
    }
    
    /**
     * Truncate parameters and fragments of the URL
     * @param pageUrl the page URL
     * @return the page URL without parameters and fragments
     */
    protected String _truncateURL(String pageUrl)
    {
        return StringUtils.substringBefore(StringUtils.substringBefore(pageUrl, "?"), "#");
    }
    
    private Cache<MatomoComputedCacheKey, Integer> _getComputedVisitsCache()
    {
        return _cacheManager.get(__CACHE_COMPUTED_VISITS);
    }
    
    private Cache<String, List<Map<String, Object>>> _getLiveVisitsCache()
    {
        return _cacheManager.get(__CACHE_LIVE_VISITS);
    }
    
    private static final class MatomoComputedCacheKey extends AbstractCacheKey
    {
        public MatomoComputedCacheKey(RequestType type, String pageUrl)
        {
            super(type, pageUrl);
        }
        
        public static MatomoComputedCacheKey of(RequestType type, String pageUrl)
        {
            return new MatomoComputedCacheKey(type, pageUrl);
        }
    }
    
    public void dispose()
    {
        _httpClient.close(CloseMode.GRACEFUL);
    }
}
