/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.stream.StreamSupport;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;

import org.apache.commons.lang3.StringUtils;
import org.apache.xpath.XPathAPI;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.DefaultContent;
import org.ametys.core.observation.Event;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.contentio.archive.Archivers;
import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.contentio.archive.ContentsArchiverHelper.ContentFiller;
import org.ametys.plugins.contentio.archive.ImportGlobalFailException;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.ResourcesAdditionalDataGetter;
import org.ametys.plugins.contentio.archive.UnitaryImporter;
import org.ametys.plugins.contentio.archive.ZipEntryHelper;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.collection.AmetysObjectCollection;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.runtime.model.View;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.ModifiableSiteAwareAmetysObject;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteType;

class SiteImporter
{
    final ImportReport _report = new ImportReport();
    private final SitesArchiver _siteArchiver;
    private final ModifiableTraversableAmetysObject _root;
    private final Path _zipArchivePath;
    private final Merger _merger;
    private final Logger _logger;
    private final DocumentBuilder _builder;
    
    SiteImporter(SitesArchiver siteArchiver, ModifiableTraversableAmetysObject root, Path zipArchivePath, Merger merger, Logger logger) throws ParserConfigurationException
    {
        _siteArchiver = siteArchiver;
        _root = root;
        _zipArchivePath = zipArchivePath;
        _merger = merger;
        _logger = logger;
        _builder = DocumentBuilderFactory.newInstance()
                .newDocumentBuilder();
    }
    
    private class UnitarySiteImporter implements UnitaryImporter<Site>
    {
        private String _siteName;
        
        UnitarySiteImporter(String siteName)
        {
            _siteName = siteName;
        }
        
        @Override
        public String objectNameForLogs()
        {
            return "Site";
        }

        @Override
        public Document getPropertiesXml(Path zipEntryPath) throws Exception
        {
            return _getSitePropertiesXml(zipEntryPath);
        }

        @Override
        public String retrieveId(Document propertiesXml) throws Exception
        {
            return Archivers.xpathEvalNonEmpty("site/@id", propertiesXml);
        }

        @Override
        public Site create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
        {
            return _createSite(zipEntryPath, id, _siteName, propertiesXml);
        }
        
        @Override
        public ImportReport getReport()
        {
            return _report;
        }
        
        @Override
        public void unitaryImportFinalize()
        {
            // clear cache and resolve the site to avoid retrieving the outdated node
            _siteArchiver._siteManager.clearCache();
            Site site = _siteArchiver._siteManager.getSite(_siteName);
            // Notify observers
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_SITE, site);
            _siteArchiver._observationManager.notify(new Event(ObservationConstants.EVENT_SITE_ADDED, _siteArchiver._currentUserProvider.getUser(), eventParams));
            
            UnitaryImporter.super.unitaryImportFinalize();
        }
    }
    
    void importSites() throws IOException
    {
        Collection<Path> sitePaths = _retrieveSites();
        for (Path sitePath : sitePaths)
        {
            String siteName = sitePath.getFileName().toString();
            _importSite(sitePath, siteName);
        }
    }
    
    void importSite(String siteName) throws IOException
    {
        String path = Archivers.getHashedPath(siteName);
        String prefix = SitesArchiver.__COMMON_PREFIX + path + "/" + siteName;
        Path sitePath = ZipEntryHelper.zipFileRoot(_zipArchivePath)
                .resolve(prefix);
        _importSite(sitePath, siteName);
    }
    
    private Collection<Path> _retrieveSites() throws IOException
    {
        Collection<Path> sitePaths = new SiteRetriever().getThreeDepthPaths();
        return sitePaths;
    }
    
    /**
     * Inner class to retrieve from:
     * 
     * sites
     * |_45
     * __|_7d
     * ____|_www
     * |_7b
     * __|_d7
     * ____|_site1
     * 
     * the result paths:
     * # /sites/45/7d/www
     * # /sites/7b/d7/site1
     * 
     */
    private final class SiteRetriever
    {
        Collection<Path> getThreeDepthPaths() throws IOException
        {
            return _getThreeDepthPaths(SitesArchiver.__COMMON_PREFIX, 0)
                    .collect(Collectors.toList());
        }
        
        private Stream<Path> _getThreeDepthPaths(String currentPath, int count) throws IOException
        {
            DirectoryStream<Path> nextLevels = ZipEntryHelper.children(
                _zipArchivePath,
                Optional.of(currentPath),
                Files::isDirectory);
            return _getNextLevelPathDirectories(nextLevels, count);
        }
        
        private Stream<Path> _getThreeDepthPaths(Path currentPath, int count) throws IOException
        {
            if (count == 3)
            {
                return Stream.of(currentPath);
            }
            else
            {
                return _getThreeDepthPaths(currentPath.toString(), count);
            }
        }
        
        private Stream<Path> _getNextLevelPathDirectories(DirectoryStream<Path> currentLevels, int count)
        {
            return StreamSupport.stream(currentLevels.spliterator(), false)
                    .map(LambdaUtils.wrap(currentLevel -> _getThreeDepthPaths(currentLevel, count + 1)))
                    .flatMap(Function.identity());
        }
    }
    
    private void _importSite(Path sitePath, String siteName) throws IOException
    {
        new UnitarySiteImporter(siteName)
                .unitaryImport(_zipArchivePath, sitePath, _merger, _logger);
    }
    
    private Document _getSitePropertiesXml(Path sitePath) throws IOException
    {
        String propertiesFileName = sitePath.getFileName() + ".xml";
        String zipEntryPath = sitePath.resolve(propertiesFileName).toString();
        try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, zipEntryPath))
        {
            Document doc = _builder.parse(stream);
            return doc;
        }
        catch (SAXException e)
        {
            throw new IOException(e);
        }
    }
    
    private Site _createSite(Path sitePath, String id, String siteName, Document propertiesXml) throws ImportGlobalFailException, AmetysObjectNotImportedException, RepositoryException, TransformerException, IOException, Exception
    {
        _logger.info("Importing site from '{}!{}' ...", _zipArchivePath, sitePath);
        Site site = _createChildSite(id, siteName);
        site.setType(Archivers.xpathEvalNonEmpty("site/@type", propertiesXml));
        
        XMLValuesExtractorAdditionalDataGetter additionalDataGetter = new ResourcesAdditionalDataGetter(_zipArchivePath, sitePath);
        _setSiteProperties(site, propertiesXml, additionalDataGetter);
        
        Archivers.unitarySave(site.getNode(), _logger);
        
        ImportReport importResourceReport = _importResources(site, sitePath);
        _report.addFrom(importResourceReport);
        ImportReport importContentReport = _importContents(site, sitePath, siteName);
        _report.addFrom(importContentReport);
        ImportReport importPluginReport = _importPlugins(site, sitePath);
        _report.addFrom(importPluginReport);
        ImportReport importSitemapReport = _importSitemaps(site, sitePath);
        _report.addFrom(importSitemapReport);
        
        return site;
    }
    
    private Site _createChildSite(String id, String siteName) throws RepositoryException
    {
        String uuid = StringUtils.substringAfter(id, "://");
        // Create a Node with JCR primary type "ametys:site"
        // But then call '_replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
        Node srcNode = ((Site) _root.createChild(siteName, "ametys:site")).getNode();
        Node siteNode = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
        return _siteArchiver._resolver.resolve(_root.getPath(), siteNode, null, false);
    }
    
    private void _setSiteProperties(Site site, Document propertiesXml, XMLValuesExtractorAdditionalDataGetter additionalDataGetter) throws TransformerException, Exception
    {
        Element siteElement = (Element) XPathAPI.selectSingleNode(propertiesXml, "site");
        SiteType siteType = _siteArchiver._siteTypeEP.getExtension(site.getType());
        View view = View.of(siteType);
        Map<String, Object> values = new ModelAwareXMLValuesExtractor(siteElement, additionalDataGetter, siteType)
                .extractValues(view);
        
        site.getDataHolder().synchronizeValues(view, values);
    }
    
    private ImportReport _importResources(Site site, Path sitePath) throws IOException
    {
        String baseImportResourcePath = sitePath.resolve("resources")
                .toString();
        if (ZipEntryHelper.zipEntryFolderExists(_zipArchivePath, baseImportResourcePath))
        {
            Node parentOfRootResources = site.getNode();
            return _siteArchiver._resourcesArchiverHelper.importCollection(baseImportResourcePath + "/", parentOfRootResources, _zipArchivePath, _merger);
        }
        else
        {
            _logger.info("No resource to be imported for Site '{}', the path '{}!{}' does not exist", site, _zipArchivePath, baseImportResourcePath);
            return new ImportReport();
        }
    }
    
    private ImportReport _importContents(Site site, Path sitePath, String siteName) throws IOException
    {
        AmetysObjectCollection rootContents = site.hasChild(SitesArchiver.__SITE_CONTENTS_JCR_NODE_NAME)
                ? site.getChild(SitesArchiver.__SITE_CONTENTS_JCR_NODE_NAME)
                : site.createChild(SitesArchiver.__SITE_CONTENTS_JCR_NODE_NAME, "ametys:collection");
        String baseImportContentPath = sitePath.resolve("contents")
                .toString();
        if (ZipEntryHelper.zipEntryFolderExists(_zipArchivePath, baseImportContentPath))
        {
            Collection<ContentFiller> contentFillers = Collections.singleton(content -> _setWebAttributes(content, siteName));
            return _siteArchiver._contentsArchiverHelper.importContents(baseImportContentPath + "/", rootContents, _zipArchivePath, _merger, contentFillers);
        }
        else
        {
            _logger.info("No content to be imported for Site '{}', the path '{}!{}' does not exist", site, _zipArchivePath, baseImportContentPath);
            return new ImportReport();
        }
    }
    
    private void _setWebAttributes(DefaultContent content, String siteName)
    {
        if (content instanceof ModifiableSiteAwareAmetysObject)
        {
            _logger.debug("Setting site name '{}' to freshly imported content {}", siteName, content);
            ((ModifiableSiteAwareAmetysObject) content).setSiteName(siteName);
        }
    }
    
    private ImportReport _importPlugins(Site site, Path sitePath) throws IOException, RepositoryException
    {
        String baseImportPluginPath = sitePath.resolve("plugins")
                .toString();
        if (!ZipEntryHelper.zipEntryFolderExists(_zipArchivePath, baseImportPluginPath))
        {
            _logger.info("No plugin to be imported for Site '{}', the path '{}!{}' does not exist", site, _zipArchivePath, baseImportPluginPath);
            return new ImportReport();
        }
        
        List<ImportReport> reports = new ArrayList<>();
        
        DirectoryStream<Path> childPlugins = ZipEntryHelper.children(
                _zipArchivePath,
                Optional.of(baseImportPluginPath),
                Files::isDirectory);
        
        try (childPlugins)
        {
            for (Path childPlugin : childPlugins)
            {
                ImportReport importPluginReport = _importPlugin(site, childPlugin);
                reports.add(importPluginReport);
            }
        }
        
        return ImportReport.union(reports);
    }
    
    private ImportReport _importPlugin(Site site, Path zipPluginEntryPath) throws IOException, RepositoryException
    {
        String pluginName = zipPluginEntryPath.getFileName().toString();
        SitePluginArchiver sitePluginArchiver = _siteArchiver._retrieveSitePluginArchiver(pluginName);
        Node allPluginsNode = SitesArchiver._getAllPluginsNode(site);
        _logger.info("Importing site plugin '{}' ({}!{}) from archive with archiver '{}'", pluginName, _zipArchivePath, zipPluginEntryPath, sitePluginArchiver.getClass().getName());
        String zipPluginEntryPathStr = zipPluginEntryPath.toString();
        return sitePluginArchiver.partialImport(site, pluginName, allPluginsNode, _zipArchivePath, zipPluginEntryPathStr, _merger);
    }
    
    private ImportReport _importSitemaps(Site site, Path sitePath) throws RepositoryException, IOException
    {
        String baseImportSitemapPath = sitePath.resolve("pages")
                .toString();
        if (!ZipEntryHelper.zipEntryFolderExists(_zipArchivePath, baseImportSitemapPath))
        {
            _logger.info("No sitemap to be imported for Site '{}', the path '{}!{}' does not exist", site, _zipArchivePath, baseImportSitemapPath);
            return new ImportReport();
        }
        
        List<ImportReport> reports = new ArrayList<>();
        
        DirectoryStream<Path> sitemaps = ZipEntryHelper.children(
                _zipArchivePath,
                Optional.of(baseImportSitemapPath),
                Files::isDirectory);
        
        try (sitemaps)
        {
            for (Path sitemap : sitemaps)
            {
                var sitemapImporter = new SitemapImporter(_siteArchiver, site, sitemap, _zipArchivePath, _merger, _logger, _builder);
                sitemapImporter.importSitemap();
                reports.add(sitemapImporter._report);
            }
        }
        
        return ImportReport.union(reports);
    }
}
