/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.docx;

import java.io.IOException;
import java.io.InputStream;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.excalibur.xml.dom.DOMSerializer;
import org.apache.excalibur.xml.sax.SAXParser;
import org.w3c.dom.Document;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import org.ametys.core.util.IgnoreRootHandler;

/**
 * Aggregate all relevants XML parts of a .docx file.
 */
public class DocxGenerator extends ServiceableGenerator
{
    private DOMSerializer _domSerializer;
    
    @Override
    public void service(ServiceManager sManager) throws ServiceException
    {
        super.service(sManager);
        _domSerializer = (DOMSerializer) sManager.lookup(DOMSerializer.ROLE);
    }
    
    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> context = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        
        Document document = (Document) context.get("document");
        Document relations = (Document) context.get("relations");
        
        ZipFile zipFile = (ZipFile) context.get("zipFile");

        contentHandler.startDocument();
        contentHandler.startElement("", "document", "document", new AttributesImpl());
        _domSerializer.serialize(document, new IgnoreRootHandler(contentHandler), lexicalHandler);
        _domSerializer.serialize(relations, new IgnoreRootHandler(contentHandler), lexicalHandler);
        
        ZipArchiveEntry zipEntry = zipFile.getEntry("word/styles.xml");
        
        SAXParser saxParser = null;
        try
        {
            saxParser = (SAXParser) manager.lookup(SAXParser.ROLE);
            try (InputStream is = zipFile.getInputStream(zipEntry))
            {
                saxParser.parse(new InputSource(is), new IgnoreRootHandler(contentHandler));
            }
            
            zipEntry = zipFile.getEntry("word/numbering.xml");
            if (zipEntry != null)
            {
                try (InputStream is = zipFile.getInputStream(zipEntry))
                {
                    saxParser.parse(new InputSource(is), new IgnoreRootHandler(contentHandler));
                }
            }
        }
        catch (ServiceException e)
        {
            throw new SAXException("Unable to get a SAX parser.", e);
        }
        finally
        {
            manager.release(saxParser);
        }
        
        contentHandler.endElement("", "document", "document");
        contentHandler.endDocument();
    }
}
