/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.docx;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.Map;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.transformation.AbstractTransformer;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.AttributesImpl;

import org.ametys.cms.data.NamedResource;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ModifiableContent;

/**
 * Cocoon Transformer for getting images from the docx file after transformation to docbook.
 */
public class DocxImagesTransformer extends AbstractTransformer implements Contextualizable
{
    private ZipFile _zipFile;
    private Content _content;
    private Context _context;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void setup(SourceResolver resolver, Map objectModel, String src, Parameters par) throws ProcessingException, SAXException, IOException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> parentContext = (Map<String, Object>) objectModel.get(ObjectModelHelper.PARENT_CONTEXT);
        _zipFile = (ZipFile) parentContext.get("zipFile");
        _content = (Content) parentContext.get("content");
    }
    
    @Override
    public void startElement(String uri, String localName, String raw, Attributes atts) throws SAXException
    {
        if ("imagedata".equals(localName) && "local".equals(atts.getValue("type")))
        {
            String fileName = atts.getValue("filename");
            String fileRef = atts.getValue("fileref");
            
            AttributesImpl atts2 = new AttributesImpl();
            for (int i = 0; i < atts.getLength(); i++)
            {
                String name = atts.getQName(i);
                
                if (!"filename".equals(name) && !"fileref".equals(name))
                {
                    atts2.addAttribute(atts.getURI(i), atts.getLocalName(i), name, atts.getType(i), atts.getValue(i));
                }
            }
            
            atts2.addAttribute("", "fileref", "fileref", "CDATA", fileName);
            
            ZipArchiveEntry entry = _zipFile.getEntry("word/" + fileRef);
            try (InputStream is = _zipFile.getInputStream(entry))
            {
                NamedResource file = new NamedResource();
                file.setLastModificationDate(ZonedDateTime.now());
                file.setInputStream(is);
                
                String mimeType = _context.getMimeType(fileName);
                file.setMimeType(mimeType != null ? mimeType : "application/unknown");
                
                
                RichText richText = _content.getValue("content");
                if (richText == null)
                {
                    richText = new RichText();
                }
                
                richText.addAttachment(file);
                
                ((ModifiableContent) _content).setValue("content", richText);
            }
            catch (IOException e)
            {
                throw new SAXException("Unable to get image in the docx file", e);
            }
            
            super.startElement(uri, localName, raw, atts2);
            return;
        }

        super.startElement(uri, localName, raw, atts);
    }
    
    @Override
    public void recycle()
    {
        super.recycle();
        
        _zipFile = null;
        _content = null;
    }
}
