/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio;

import java.io.File;
import java.io.IOException;
import java.util.Map;

import org.ametys.cms.repository.Content;
import org.ametys.web.repository.content.ModifiableWebContent;
import org.ametys.web.repository.page.ModifiablePage;

/**
 * Component responsible for populating a Content from an existing file.<br>
 * Contents are meant to be provided totally empty, 
 * so that it is up to the implementation to set every metadata on the Content, 
 * even the content type, the workflow id, ...
 */
public interface ContentImporter
{
    /**
     * Extracts file content and populates the given {@link Content}.
     * Typical usage is to provide a newly created empty Content. 
     * @param file the source.
     * @param content the {@link Content} to be populated.
     * @param params in/out params to set and get additional data to and from the importer.
     * @throws IOException if an error occurs processing the stream.
     */
    public void importContent(File file, ModifiableWebContent content, Map<String, String> params) throws IOException;
    
    /**
     * Returns all mime types handled by this importer.
     * @return all mime types handled by this importer.
     */
    public String[] getMimeTypes();
    
    /**
     * Post treatment after import process
     * @param page The created page
     * @param content The created content
     * @param file The imported file
     * @throws IOException if an error occurred
     */
    public void postTreatment(ModifiablePage page, Content content, File file) throws IOException;
}
