/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio;

import java.io.InputStream;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.core.authentication.token.AuthenticationTokenManager;
import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.sitemap.Sitemap;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.Constants;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Context;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.commons.lang.StringUtils;

/**
 * Action to import a content from an input stream received through a POST request.
 */
public class ImportContentAction extends ServiceableAction implements Contextualizable
{
    private AuthenticationTokenManager _authenticationManager;
    private ContentIOManager _contentIOManager;
    private SiteManager _siteManager;

    private Context _context;

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        _authenticationManager = (AuthenticationTokenManager) manager.lookup(AuthenticationTokenManager.ROLE);
        _contentIOManager = (ContentIOManager) manager.lookup(ContentIOManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        List<String> error = new ArrayList<>();
        Boolean success = true;
        
        if (!"POST".equals(request.getMethod()))
        {
            success = false;
            error.add("Only POST methods are allowed.");
            getLogger().warn("Only POST methods are allowed.");
        }
        
        Object content = request.get("content");
        if (content == null)
        {
            success = false;
            error.add("No content provided. The parameter 'content' is mandatory.");
            getLogger().warn("No content provided. The parameter 'content' is mandatory.");
        }
        
        String authenticationToken = (String) request.get("key");
        if (authenticationToken == null)
        {
            success = false;
            error.add("No API key provided. The parameter 'key' is mandatory.");
            getLogger().warn("No API key provided. The parameter 'key' is mandatory.");
        }
        
        UserIdentity user = _authenticationManager.validateToken(authenticationToken);
        if (user == null)
        {
            success = false;
            error.add("The API key provided is invalid, please check the value of the parameter 'key'.");
            getLogger().warn("The API key provided is invalid, please check the value of the parameter 'key'.");
        }
        
        String siteName = parameters.getParameter("site");
        String sitemapName = parameters.getParameter("sitemap");
        String path = parameters.getParameter("path");
        boolean extern = parameters.getParameterAsBoolean("extern", false);
        
        SitemapElement rootPage = getPageFromPath(siteName, sitemapName, path);
        
        if (rootPage == null)
        {
            success = false;
            error.add("Unable to retrieve the parent page, the specified path was not found.");
            getLogger().warn("Unable to retrieve the parent page, the specified path was not found.");
        }
        
        if (success && content instanceof Part)
        {
            Part part = (Part) content;
            
            InputStream is = part.getInputStream();
            String contentName = part.getUploadName();
            String mimeType = _context.getMimeType(contentName);
            
            try
            {
                _contentIOManager.importContent(is, mimeType, contentName, user, rootPage, extern);
            }
            catch (RightsException e)
            {
                success = false;
                error.add("Insufficient rights to create a content.");
                getLogger().warn("Insufficient rights to create a content.");
            }
        }
        
        Map<String, Object> mapResult = new HashMap<>();
        mapResult.put("success", success);
        if (!success)
        {
            mapResult.put("error", error);
        }
        
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, mapResult);
        
        return EMPTY_MAP;
    }
    
    private SitemapElement getPageFromPath(String siteName, String sitemapName, String path)
    {
        Site site = null;
        Sitemap sitemap = null;
        
        try
        {
            site = _siteManager.getSite(siteName);
            sitemap = site.getSitemap(sitemapName);
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().warn("Unable to retrieve the site or sitemap.", e);
            return null;
        }
        
        if (StringUtils.isEmpty(path) || "/".equals(path))
        {
            return sitemap;
        }
        
        String pathCleaned = path;
        if (pathCleaned.startsWith("/"))
        {
            pathCleaned = pathCleaned.substring(1);
        }
        if (pathCleaned.endsWith("/"))
        {
            pathCleaned = pathCleaned.substring(0, path.length() - 1);
        }

        String[] pagesName = pathCleaned.split("/");
        
        SitemapElement page = sitemap;
        for (int i = 0; i < pagesName.length; i++)
        {
            if (StringUtils.isEmpty(pagesName[i]))
            {
                return null;
            }
            
            page = page.getChild(pagesName[i]);
            
            if (page == null)
            {
                return null;
            }
        }
        
        return page;
        
    }
}
