/*
 *  Copyright 2011 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.webcontentio.docx;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.environment.Context;
import org.apache.commons.compress.archivers.zip.ZipArchiveEntry;
import org.apache.commons.compress.archivers.zip.ZipFile;
import org.apache.commons.lang.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.sax.SAXParser;
import org.apache.excalibur.xml.xpath.PrefixResolver;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.Document;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.helpers.DefaultHandler;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.plugins.webcontentio.ContentImporter;
import org.ametys.web.repository.content.ModifiableWebContent;
import org.ametys.web.repository.page.ModifiablePage;

/**
 * Imports Docx files.
 */
public class DocxContentImporter implements ContentImporter, Serviceable, Contextualizable
{
    /** The service manager */
    protected ServiceManager _manager;
    
    private SourceResolver _resolver;
    private DOMParser _domParser;
    private XPathProcessor _xPathProcessor;
    private Context _context;
    
    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _manager = manager;
        _resolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _domParser = (DOMParser) manager.lookup(DOMParser.ROLE);
        _xPathProcessor = (XPathProcessor) manager.lookup(XPathProcessor.ROLE);
    }
    
    @Override
    public void importContent(File file, ModifiableWebContent content, Map<String, String> params) throws IOException
    {
        ZipFile zipFile = new ZipFile(file);
        
        String template = null;
        String longTitle = null;
        String contentType = null;
        boolean section;
        boolean directAccess;
        boolean footer;
        boolean event;
        
        Document document = _getDocument(zipFile, "word/document.xml", file);
        Document relations = _getDocument(zipFile, "word/_rels/document.xml.rels", file);
        
        PrefixResolver resolver = new DocxPrefixResolver();
        
        template = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[1]/w:tr[1]/w:sdt/w:sdtPr[w:tag/@w:val='template']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        longTitle = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[1]/w:tr[2]/w:sdt/w:sdtPr[w:tag/@w:val='long_title']/w:text", resolver);
        contentType = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[1]/w:tr[3]/w:sdt/w:sdtPr[w:tag/@w:val='content-type']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        
        // tags
        String sectionStr = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[2]/w:tr[1]/w:sdt/w:sdtPr[w:tag/@w:val='SECTION']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        String directAccessStr = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[2]/w:tr[2]/w:sdt/w:sdtPr[w:tag/@w:val='ACCES_DIRECTS']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        String footerStr = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[2]/w:tr[3]/w:sdt/w:sdtPr[w:tag/@w:val='FOOTER_LINK']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        String eventStr = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:tbl[2]/w:tr[4]/w:sdt/w:sdtPr[w:tag/@w:val='EVENT']/w:comboBox/w:listItem[@w:displayText=../../../w:sdtContent/w:tc/w:p/w:r/w:t]/@w:value", resolver);
        
        section = "SECTION".equals(sectionStr);
        directAccess = "ACCES_DIRECTS".equals(directAccessStr);
        footer = "FOOTER_LINK".equals(footerStr);
        event = "EVENT".equals(eventStr);
        
        // abstract
        NodeList abstractList = _xPathProcessor.selectNodeList(document, "/w:document/w:body/w:sdt[w:sdtPr/w:tag/@w:val='abstract' and not(w:sdtPr/w:showingPlcHdr)]/w:sdtContent/w:p/w:r/w:t", resolver);
        StringBuilder abstr = new StringBuilder();
        
        for (int i = 0; i < abstractList.getLength(); i++)
        {
            if (i != 0)
            {
                abstr.append('\n');
            }
            
            abstr.append(abstractList.item(i).getTextContent());
        }
        
        // illustration
        String pictureId = _xPathProcessor.evaluateAsString(document, "/w:document/w:body/w:sdt[w:sdtPr/w:tag/@w:val='illustration' and not(w:sdtPr/w:showingPlcHdr)]/w:sdtContent/w:p/w:r/w:drawing/wp:inline/a:graphic/a:graphicData/pic:pic/pic:blipFill/a:blip/@r:embed", resolver);
        
        if (StringUtils.trimToNull(pictureId) != null)
        {
            String pictureEntryName = _xPathProcessor.evaluateAsString(relations, "/rel:Relationships/rel:Relationship[@Id='" + pictureId + "']/@Target", resolver);
            
            int i = pictureEntryName.lastIndexOf('/');
            String fileName = i == -1 ? pictureEntryName : pictureEntryName.substring(i + 1);
            
            ZipArchiveEntry entry = zipFile.getEntry("word/" + pictureEntryName);
            
            try (InputStream is = zipFile.getInputStream(entry))
            {
                Binary illustration = new Binary();
                illustration.setLastModificationDate(ZonedDateTime.now());
                illustration.setInputStream(is);
                illustration.setFilename(fileName);
                
                String mimeType = _context.getMimeType(fileName);
                if (mimeType != null)
                {
                    illustration.setMimeType(mimeType);
                }
                
                content.getComposite("illustration", true).setValue("image", illustration);
            }
        }
        
        // title
        NodeList titleList = _xPathProcessor.selectNodeList(document, "/w:document/w:body/w:p[w:pPr/w:pStyle/@w:val='Titre'][1]/w:r/w:t", resolver);
        StringBuilder titleBuilder = new StringBuilder();
        
        for (int j = 0; j < titleList.getLength(); j++)
        {
            titleBuilder.append(titleList.item(j).getTextContent());
        }
        
        params.put("page.template", StringUtils.trimToNull(template));
        params.put("page.longTitle", StringUtils.trimToNull(longTitle));
        
        String title = titleBuilder.toString();
        
        content.setTitle(StringUtils.trimToNull(title) != null ? title : content.getName());
        String cType = _getContentType(contentType);
        content.setTypes(new String[] {cType});
        
        if (abstr.length() > 0)
        {
            content.setValue("abstract", abstr.toString());
        }
        
        if (section)
        {
            content.tag("SECTION");
        }
        
        if (directAccess)
        {
            content.tag("ACCES_DIRECTS");
        }
        
        if (footer)
        {
            content.tag("FOOTER_LINK");
        }
        
        if (event)
        {
            content.tag("EVENT");
        }
        
        // actual content
        Map<String, Object> context = new HashMap<>();
        context.put("document", document);
        context.put("relations", relations);
        context.put("zipFile", zipFile);
        context.put("content", content);
        Source src = _resolver.resolveURI("cocoon:/docx2docbook", null, context);
        
        try (InputStream is = src.getInputStream())
        {
            RichText richText = new RichText();
            richText.setLastModificationDate(ZonedDateTime.now());
            richText.setMimeType("text/xml");
            richText.setInputStream(is);
            
            SAXParser saxParser = null;
            try (InputStream in = richText.getInputStream())
            {
                saxParser = (SAXParser) _manager.lookup(SAXParser.ROLE);
                saxParser.parse(new InputSource(in), new DefaultHandler());
                content.setValue("content", richText);
            }
            catch (SAXException e)
            {
                throw new IOException("Invalid resulting XML after transformation", e);
            }
            catch (ServiceException e)
            {
                throw new IOException("Unable to get a SAX parser.", e);
            }
            finally
            {
                _manager.release(saxParser);
            }
        }
        
        ZipFile.closeQuietly(zipFile);
    }
    
    private String _getContentType(String contentType)
    {
        return StringUtils.trimToNull(contentType) != null ? contentType : "org.ametys.web.default.Content.article";
    }

    @Override
    public String[] getMimeTypes()
    {
        // handles docx mime-type
        return new String[]{"application/vnd.openxmlformats-officedocument.wordprocessingml.document"};
    }
    
    @Override
    public void postTreatment(ModifiablePage page, Content content, File file) throws IOException
    {
        // Nothing to do
    }
    
    private Document _getDocument(ZipFile zipFile, String entryName, File file) throws IOException
    {
        ZipArchiveEntry entry = zipFile.getEntry(entryName);
        
        try (InputStream is = zipFile.getInputStream(entry))
        {
            return _domParser.parseDocument(new InputSource(is));
        }
        catch (SAXException e)
        {
            throw new IOException("Unable to read " + entryName + " in file " + file.getAbsolutePath(), e);
        }
    }
    
    private class DocxPrefixResolver implements PrefixResolver
    {
        private Map<String, String> _ns = new HashMap<>();
        
        public DocxPrefixResolver()
        {
            _ns.put("w", "http://schemas.openxmlformats.org/wordprocessingml/2006/main");
            _ns.put("r", "http://schemas.openxmlformats.org/officeDocument/2006/relationships");
            _ns.put("wp", "http://schemas.openxmlformats.org/drawingml/2006/wordprocessingDrawing");
            _ns.put("a", "http://schemas.openxmlformats.org/drawingml/2006/main");
            _ns.put("pic", "http://schemas.openxmlformats.org/drawingml/2006/picture");
            _ns.put("rel", "http://schemas.openxmlformats.org/package/2006/relationships");
        }
        
        @Override
        public String prefixToNamespace(String prefix)
        {
            return _ns.get(prefix);
        }
    }
}
