/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.archive;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Property;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.math3.util.IntegerSequence.Incrementor;
import org.xml.sax.Attributes;
import org.xml.sax.SAXException;

import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.contentio.archive.ArchiveHandler;
import org.ametys.plugins.contentio.archive.Archiver;
import org.ametys.plugins.contentio.archive.Archivers;
import org.ametys.plugins.contentio.archive.ContentsArchiverHelper;
import org.ametys.plugins.contentio.archive.DefaultPluginArchiver;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.ManifestReaderWriter;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.PartialImport;
import org.ametys.plugins.contentio.archive.ResourcesArchiverHelper;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObjectFactoryExtensionPoint;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.TraversableAmetysObject;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.jcr.NodeHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.data.type.ModelItemTypeExtensionPoint;
import org.ametys.web.parameters.view.ViewParametersManager;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.page.jcr.DefaultPageFactory;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.repository.site.SiteType;
import org.ametys.web.repository.site.SiteTypesExtensionPoint;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.repository.sitemap.SitemapFactory;
import org.ametys.web.service.ServiceExtensionPoint;

import com.google.common.collect.Streams;

/**
 * {@link Archiver} for all sites.
 */
public class SitesArchiver extends AbstractLogEnabled implements Archiver, Serviceable
{
    /** Archiver id. */
    public static final String ID = "sites";
    
    static final String __SITE_RESOURCES_JCR_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":resources";
    static final String __SITE_CONTENTS_JCR_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":contents";
    static final String __SITE_PLUGINS_JCR_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":plugins";
    
    static final String __COMMON_PREFIX = ID + "/";
    static final String __ACL_ZIP_ENTRY_FILENAME = "acl.xml";
    
    private static final String __PARTIAL_IMPORT_PREFIX = ID + "/";
    
    /** The site manager */
    protected SiteManager _siteManager;
    /** The helper for archiving contents */
    protected ContentsArchiverHelper _contentsArchiverHelper;
    /** The helper for archiving resources */
    protected ResourcesArchiverHelper _resourcesArchiverHelper;
    /** The extension point for {@link SitePluginArchiver}s */
    protected SitePluginArchiverExtensionPoint _pluginArchiverExtensionPoint;
    /** The extension point for {@link TagProvider}s */
    protected TagProviderExtensionPoint _tagProviderEP;
    /** The Ametys Object Resolver */
    protected AmetysObjectResolver _resolver;
    /** The default page factory */
    protected DefaultPageFactory _defaultPageFactory;
    /** The sitemap factory */
    protected SitemapFactory _sitemapFactory;
    /** The extension point for Services */
    protected ServiceExtensionPoint _serviceExtensionPoint;
    /** The extension point for {@link SiteType}s */
    protected SiteTypesExtensionPoint _siteTypeEP;
    /** The page data type extension point */
    protected ModelItemTypeExtensionPoint _pageDataTypeExtensionPoint;
    /** The observation manager */
    protected ObservationManager _observationManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private ManifestReaderWriter _manifestReaderWriter = new SitesArchiverManifestReaderWriter();
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _contentsArchiverHelper = (ContentsArchiverHelper) manager.lookup(ContentsArchiverHelper.ROLE);
        _resourcesArchiverHelper = (ResourcesArchiverHelper) manager.lookup(ResourcesArchiverHelper.ROLE);
        _pluginArchiverExtensionPoint = (SitePluginArchiverExtensionPoint) manager.lookup(SitePluginArchiverExtensionPoint.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) manager.lookup(TagProviderExtensionPoint.ROLE);
        AmetysObjectFactoryExtensionPoint ametysObjectFactoryEP = (AmetysObjectFactoryExtensionPoint) manager.lookup(AmetysObjectFactoryExtensionPoint.ROLE);
        _defaultPageFactory = (DefaultPageFactory) ametysObjectFactoryEP.getExtension(DefaultPageFactory.class.getName());
        _sitemapFactory = (SitemapFactory) ametysObjectFactoryEP.getExtension(SitemapFactory.class.getName());
        _serviceExtensionPoint = (ServiceExtensionPoint) manager.lookup(ServiceExtensionPoint.ROLE);
        _siteTypeEP = (SiteTypesExtensionPoint) manager.lookup(SiteTypesExtensionPoint.ROLE);
        _pageDataTypeExtensionPoint = (ModelItemTypeExtensionPoint) manager.lookup(ModelItemTypeExtensionPoint.ROLE_PAGE_DATA);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    @Override
    public ManifestReaderWriter getManifestReaderWriter()
    {
        return _manifestReaderWriter;
    }
    
    @Override
    public void export(ZipOutputStream zos) throws IOException
    {
        zos.putNextEntry(new ZipEntry(__COMMON_PREFIX)); // even if there is no site, at least export the sites folder
        
        try (AmetysObjectIterable<Site> sites = _siteManager.getSites())
        {
            for (Site site : sites)
            {
                getLogger().info("Processing site '{}' for archiving", site.getName());
                _exportSite(site, zos);
            }
        }
    }
    
    private void _exportSite(Site site, ZipOutputStream zos) throws IOException
    {
        String siteName = site.getName();
        String prefix = __COMMON_PREFIX + NodeHelper.getFullHashPath(siteName);
        
        // export site's own data
        ZipEntry siteEntry = new ZipEntry(prefix + "/" + siteName + ".xml");
        zos.putNextEntry(siteEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            Attributes attrs = _getSiteAttributes(site);
            XMLUtils.startElement(contentHandler, "site", attrs);
            site.dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "site");
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX site '" + site.getPath() + "' for archiving", e);
        }
        
        // export site's resources
        ResourceCollection rootResources = site.getChild(__SITE_RESOURCES_JCR_NODE_NAME);
        _resourcesArchiverHelper.exportCollection(rootResources, zos, prefix + "/resources/");
        
        // export site's binaries
        Archivers.exportBinaries(site, zos, prefix + "/");
        
        // export all site's contents
        TraversableAmetysObject rootContents = site.getChild(__SITE_CONTENTS_JCR_NODE_NAME);
        _contentsArchiverHelper.exportContents(prefix + "/contents/", rootContents, zos);

        // export site's plugins data
        try
        {
            _getPluginNodes(site)
                .forEachRemaining(LambdaUtils.wrapConsumer(pluginNode -> _exportPlugin(site, pluginNode, zos, prefix)));
        }
        catch (RepositoryException e)
        {
            throw new IllegalArgumentException("Unable to archive plugins for site " + siteName, e);
        }
        
        _exportSitemaps(site, zos, prefix, siteName);
    }
    
    private Attributes _getSiteAttributes(Site site)
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("id", site.getId());
        attrs.addCDATAAttribute("type", site.getType());
        return attrs;
    }
    
    static Node _getAllPluginsNode(Site site) throws RepositoryException
    {
        return site.getNode()
                .getNode(__SITE_PLUGINS_JCR_NODE_NAME);
    }
    
    private static Iterator<Node> _getPluginNodes(Site site) throws RepositoryException
    {
        return _getAllPluginsNode(site)
                .getNodes();
    }
    
    SitePluginArchiver _retrieveSitePluginArchiver(String pluginName)
    {
        SitePluginArchiver sitePluginArchiver = _pluginArchiverExtensionPoint.getExtension(pluginName);
        if (sitePluginArchiver == null)
        {
            // there's no specific exporter for this plugin, let's fallback to the default export
            sitePluginArchiver = _pluginArchiverExtensionPoint.getExtension(DefaultPluginArchiver.EXTENSION_ID);
            
            if (sitePluginArchiver == null)
            {
                throw new IllegalStateException("There sould be a '__default' extension to SitePluginArchiverExtensionPoint. Please check your excluded features.");
            }
        }
        
        return sitePluginArchiver;
    }
    
    private void _exportPlugin(Site site, Node pluginNode, ZipOutputStream zos, String prefix) throws Exception
    {
        String pluginName = pluginNode.getName();
        SitePluginArchiver sitePluginArchiver = _retrieveSitePluginArchiver(pluginName);
        getLogger().info("Processing plugin '{}' for site '{}' for archiving at {} with archiver '{}'", pluginName, site.getName(), pluginNode, sitePluginArchiver.getClass().getName());
        sitePluginArchiver.export(site, pluginName, pluginNode, zos, prefix + "/plugins/" + pluginName);
    }
    
    private void _exportSitemaps(Site site, ZipOutputStream zos, String prefix, String siteName) throws IOException
    {
        // export all stored pages
        try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
        {
            for (Sitemap sitemap : sitemaps)
            {
                _exportSitemap(sitemap, zos, prefix);
            }
        }
        catch (RepositoryException e)
        {
            throw new IllegalArgumentException("Unable to archive pages for site " + siteName, e);
        }
    }
    
    private void _exportSitemap(Sitemap sitemap, ZipOutputStream zos, String prefix) throws IOException, RepositoryException
    {
        String sitemapPrefix = prefix + "/pages/" + sitemap.getName();
        zos.putNextEntry(new ZipEntry(sitemapPrefix + "/")); // even if there is no page, at least export the sitemap folder
        _exportSitemapData(sitemap, zos, sitemapPrefix);
        
        Stream<Node> nodes = Streams.stream(_getSitemapChildrenNodes(sitemap));
        Incrementor orderIncrementor = Incrementor.create()
                .withStart(0)
                .withMaximalCount(Integer.MAX_VALUE);
        nodes.filter(LambdaUtils.wrapPredicate(node -> node.isNodeType("ametys:page")))
                .forEachOrdered(LambdaUtils.wrapConsumer(pageNode ->
                {
                    _exportPage(pageNode, zos, sitemapPrefix, orderIncrementor.getCount());
                    orderIncrementor.increment();
                }));
    }
    
    private void _exportSitemapData(Sitemap sitemap, ZipOutputStream zos, String sitemapPrefix) throws RepositoryException, IOException
    {
        ZipEntry siteEntry = new ZipEntry(sitemapPrefix + ".xml");
        zos.putNextEntry(siteEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            Attributes attrs = _getSitemapAttributes(sitemap);
            XMLUtils.startElement(contentHandler, "sitemap", attrs);
            
            // internal properties (virtual, ...)
            _saxAmetysInternalProperties(sitemap, contentHandler);

            XMLUtils.startElement(contentHandler, "attributes");
            sitemap.dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "attributes");
            XMLUtils.endElement(contentHandler, "sitemap");
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX sitemap '" + sitemap.getPath() + "' for archiving", e);
        }
        
        Archivers.exportAcl(sitemap.getNode(), zos, ArchiveHandler.METADATA_PREFIX + sitemapPrefix + "/" + __ACL_ZIP_ENTRY_FILENAME);
    }
    
    private Attributes _getSitemapAttributes(Sitemap sitemap)
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("name", sitemap.getName());
        attrs.addCDATAAttribute("id", sitemap.getId());
        return attrs;
    }
    
    private static Iterator<Node> _getSitemapChildrenNodes(Sitemap sitemap) throws RepositoryException
    {
        return sitemap.getNode().getNodes();
    }
    
    private void _exportPage(Node pageNode, ZipOutputStream zos, String prefix, int order) throws RepositoryException, IOException
    {
        String pageName = pageNode.getName();
        Page page = _resolver.resolve(pageNode, false);
        
        String pagePrefix = prefix + "/" + pageName;
        
        ZipEntry siteEntry = new ZipEntry(pagePrefix + ".xml");
        zos.putNextEntry(siteEntry);
        
        try
        {
            TransformerHandler contentHandler = Archivers.newTransformerHandler();
            contentHandler.setResult(new StreamResult(zos));
            
            contentHandler.startDocument();
            Attributes attrs = _getPageAttributes(page, order);
            
            XMLUtils.startElement(contentHandler, "page", attrs);
            
            // internal properties (virtual, ...)
            _saxAmetysInternalProperties(page, contentHandler);

            XMLUtils.startElement(contentHandler, "attributes");
            page.dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "attributes");

            // Tags
            _saxTags(page, contentHandler);
            
            XMLUtils.startElement(contentHandler, "templateParameters");
            page.getTemplateParametersHolder().dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "templateParameters");
            
            XMLUtils.startElement(contentHandler, "pageContents");
            
            _saxZones(page, contentHandler);
            
            XMLUtils.endElement(contentHandler, "pageContents");
            
            XMLUtils.endElement(contentHandler, "page");
            contentHandler.endDocument();
        }
        catch (SAXException | TransformerConfigurationException e)
        {
            throw new RuntimeException("Unable to SAX page '" + page.getPath() + "' for archiving", e);
        }
        catch (RuntimeException e)
        {
            throw new RuntimeException("Unable to process Page for archiving: " + page.getId(), e);
        }
        
        _saxAttachments(page, pagePrefix, zos);

        Archivers.exportAcl(pageNode, zos, ArchiveHandler.METADATA_PREFIX + pagePrefix + "/" + __ACL_ZIP_ENTRY_FILENAME);
        
        Stream<Node> nodes = Streams.stream(_getPageChildrenNodes(pageNode));
        Incrementor orderIncrementor = Incrementor.create()
                .withStart(0)
                .withMaximalCount(Integer.MAX_VALUE);
        nodes.filter(LambdaUtils.wrapPredicate(node -> node.isNodeType("ametys:page")))
                .forEachOrdered(LambdaUtils.wrapConsumer(node ->
                {
                    _exportPage(node, zos, pagePrefix, orderIncrementor.getCount());
                    orderIncrementor.increment();
                }));
    }

    private Attributes _getPageAttributes(Page page, int order)
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("title", page.getTitle());
        attrs.addCDATAAttribute("long-title", page.getLongTitle());
        attrs.addCDATAAttribute("id", page.getId());
        attrs.addCDATAAttribute("type", page.getType().toString());
        attrs.addCDATAAttribute("order", Integer.toString(order));
        
        if (page.getType() == PageType.LINK)
        {
            attrs.addCDATAAttribute("url", page.getURL());
            attrs.addCDATAAttribute("urlType", page.getURLType().toString());
        }
        else if (page.getType() == PageType.CONTAINER)
        {
            attrs.addCDATAAttribute("template", page.getTemplate());
        }
        
        return attrs;
    }
    
    private void _saxAmetysInternalProperties(SitemapElement sitemapElement, TransformerHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "internal");
        if (sitemapElement instanceof JCRAmetysObject)
        {
            Node node = ((JCRAmetysObject) sitemapElement).getNode();
            try
            {
                _saxInternalPropeties(node, contentHandler, AmetysObjectResolver.VIRTUAL_PROPERTY);
            }
            catch (RepositoryException e)
            {
                throw new SAXException(e);
            }
        }
        XMLUtils.endElement(contentHandler, "internal");
    }
    
    private void _saxInternalPropeties(Node node, TransformerHandler contentHandler, String... properties) throws RepositoryException, SAXException
    {
        for (String propertyName : properties)
        {
            if (node.hasProperty(propertyName))
            {
                Property property = node.getProperty(propertyName);
                for (Value value : property.getValues())
                {
                    String propertyValue = value.getString();
                    XMLUtils.createElement(contentHandler, propertyName, propertyValue);
                }
            }
        }
    }
    
    private void _saxTags(Page page, TransformerHandler contentHandler) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "tags");
        Set<String> tags = page.getTags();
        for (String tagName : tags)
        {
            Map<String, Object> contextParameters = new HashMap<>();
            contextParameters.put("siteName", page.getSiteName());
            
            Tag tag = _tagProviderEP.getTag(tagName, contextParameters);

            if (tag != null)
            {
                // tag may be null if it has been registered on the page and then removed from the application
                XMLUtils.createElement(contentHandler, tagName);
            }
        }
        XMLUtils.endElement(contentHandler, "tags");
    }
    
    private void _saxAttachments(Page page, String pagePrefix, ZipOutputStream zos) throws IOException
    {
        // Put page attachments under /_metadata
        // It means that metadata of the attachments will be under /_metadata/_metadata
        String prefix = ArchiveHandler.METADATA_PREFIX + pagePrefix + "/_attachments/";
        _resourcesArchiverHelper.exportCollection(page.getRootAttachments(), zos, prefix);
    }
    
    private void _saxZones(Page page, TransformerHandler contentHandler) throws SAXException
    {
        for (Zone zone : page.getZones())
        {
            _saxZone(zone, contentHandler);
        }
    }
    
    private void _saxZone(Zone zone, TransformerHandler contentHandler) throws SAXException
    {
        try
        {
            String zoneName = zone.getName();
            
            AttributesImpl zoneAttrs = new AttributesImpl();
            zoneAttrs.addCDATAAttribute("name", zoneName);
            XMLUtils.startElement(contentHandler, "zone", zoneAttrs);
            
            XMLUtils.startElement(contentHandler, "attributes");
            zone.dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "attributes");
            
            XMLUtils.startElement(contentHandler, "zoneParameters");
            zone.getZoneParametersHolder().dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "zoneParameters");

            AmetysObjectIterable<? extends ZoneItem> zoneItems = zone.getZoneItems();
            _saxZoneItems(zoneItems, contentHandler);

            XMLUtils.endElement(contentHandler, "zone");
        }
        catch (RuntimeException e)
        {
            throw new RuntimeException("Unable to process Zone for archiving: " + zone.getId(), e);
        }
    }
    
    private void _saxZoneItems(AmetysObjectIterable<? extends ZoneItem> zoneItems, TransformerHandler contentHandler) throws SAXException
    {
        for (ZoneItem zoneItem : zoneItems)
        {
            try
            {
                _saxZoneItem(zoneItem, contentHandler);
            }
            catch (RuntimeException e)
            {
                throw new RuntimeException("Unable to process ZoneItem for archiving: " + zoneItem.getId(), e);
            }
        }
    }
    
    private void _saxZoneItem(ZoneItem zoneItem, TransformerHandler contentHandler) throws SAXException
    {
        ZoneType zoneType = zoneItem.getType();
        
        Attributes zoneItemAttrs = _getZoneItemAttributes(zoneItem, zoneType);
        
        XMLUtils.startElement(contentHandler, "zoneItem", zoneItemAttrs);
        
        XMLUtils.startElement(contentHandler, "attributes");
        zoneItem.dataToSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "attributes");
        
        XMLUtils.startElement(contentHandler, "zoneItemParameters");
        zoneItem.getZoneItemParametersHolder().dataToSAX(contentHandler);
        XMLUtils.endElement(contentHandler, "zoneItemParameters");

        if (zoneType == ZoneType.SERVICE)
        {
            ModelAwareDataHolder serviceParameters = _getServiceParameters(zoneItem);
            if (serviceParameters != null)
            {
                XMLUtils.startElement(contentHandler, "serviceParameters");
                serviceParameters.dataToSAX(contentHandler);
                XMLUtils.endElement(contentHandler, "serviceParameters");
                
                XMLUtils.startElement(contentHandler, "serviceViewParameters");
                if (serviceParameters.hasDefinition(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME))
                {
                    String viewName = serviceParameters.getValue(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME);
                    if (StringUtils.isNotBlank(viewName))
                    {
                        ModelAwareDataHolder serviceViewParameters = zoneItem.getServiceViewParametersHolder(viewName);
                        if (serviceViewParameters != null)
                        {
                            serviceViewParameters.dataToSAX(contentHandler);
                        }
                    }
                }
                XMLUtils.endElement(contentHandler, "serviceViewParameters");
            }
        }
        else if (zoneType == ZoneType.CONTENT)
        {
            String viewName = StringUtils.defaultIfBlank(zoneItem.getViewName(), "main");
            ModelAwareDataHolder contentViewParameters = zoneItem.getContentViewParametersHolder(viewName);
            
            XMLUtils.startElement(contentHandler, "contentViewParameters");
            contentViewParameters.dataToSAX(contentHandler);
            XMLUtils.endElement(contentHandler, "contentViewParameters");
        }
        
        XMLUtils.endElement(contentHandler, "zoneItem");
    }
    
    private ModelAwareDataHolder _getServiceParameters(ZoneItem zoneItem)
    {
        try
        {
            return zoneItem.getServiceParameters();
        }
        catch (Exception e)
        {
            getLogger().error("Cannot get service parameters for ZoneItem \"{}\"", zoneItem, e);
            return null;
        }
    }
    
    private Attributes _getZoneItemAttributes(ZoneItem zoneItem, ZoneType zoneType)
    {
        AttributesImpl zoneItemAttrs = new AttributesImpl();
        zoneItemAttrs.addCDATAAttribute("type", zoneType.toString());
        
        if (zoneType == ZoneType.CONTENT)
        {
            zoneItemAttrs.addCDATAAttribute("contentId", zoneItem.getContent().getId());
            
            String viewName = zoneItem.getViewName();
            if (viewName != null)
            {
                zoneItemAttrs.addCDATAAttribute("viewName", viewName);
            }
        }
        else if (zoneType == ZoneType.SERVICE)
        {
            zoneItemAttrs.addCDATAAttribute("serviceId", zoneItem.getServiceId());
        }
        
        return zoneItemAttrs;
    }
    
    private static Iterator<Node> _getPageChildrenNodes(Node pageNode) throws RepositoryException
    {
        return pageNode.getNodes();
    }
    
    @Override
    public List<I18nizableText> additionalSuccessImportMail()
    {
        return List.of(
                new I18nizableText("plugin.web-contentio", "PLUGINS_WEB_CONTENTIO_ARCHIVE_IMPORT_SITE_ARCHIVER_MAIL_ADDITIONAL_BODY_REBUILD_LIVE"),
                new I18nizableText("plugin.web-contentio", "PLUGINS_WEB_CONTENTIO_ARCHIVE_IMPORT_SITE_ARCHIVER_MAIL_ADDITIONAL_BODY_SITE_TOOL")
        );
    }
    
    @Override
    public Collection<String> managedPartialImports(Collection<String> partialImports)
    {
        if (partialImports.contains(ID))
        {
            // All sites
            return Collections.singletonList(ID);
        }
        else
        {
            // Return a Collection (can be empty) of sites to be imported
            return partialImports.stream()
                    .filter(partialImport -> partialImport.startsWith(__PARTIAL_IMPORT_PREFIX))
                    .collect(Collectors.toList());
        }
    }
    
    @Override
    public ImportReport partialImport(Path zipPath, Collection<String> partialImports, Merger merger, boolean deleteBefore) throws IOException
    {
        ModifiableTraversableAmetysObject siteRoot = _siteManager.getRoot();
        if (deleteBefore && siteRoot instanceof JCRAmetysObject)
        {
            _deleteBeforePartialImport(siteRoot, partialImports);
        }
        
        ImportReport result = _partialImport(zipPath, partialImports, merger, siteRoot);
        _saveImported(siteRoot);
        return result;
    }
    
    private void _deleteBeforePartialImport(ModifiableTraversableAmetysObject siteRoot, Collection<String> partialImports) throws IOException
    {
        if (!(siteRoot instanceof JCRAmetysObject))
        {
            return;
        }
        
        JCRAmetysObject jcrSiteRoot = (JCRAmetysObject) siteRoot;
        if (partialImports.contains(ID))
        {
            _deleteSiteRootBeforePartialImport(jcrSiteRoot);
        }
        else
        {
            for (String siteName : _retrieveSiteNames(partialImports))
            {
                _deleteSiteBeforePartialImport(siteName);
            }
        }
    }
    
    private void _deleteSiteRootBeforePartialImport(JCRAmetysObject siteRoot) throws IOException
    {
        try
        {
            Node rootNode = siteRoot.getNode();
            NodeIterator rootChildren = rootNode.getNodes();
            while (rootChildren.hasNext())
            {
                rootChildren.nextNode().remove();
            }
            rootNode.getSession().save();
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }
    }
    
    private void _deleteSiteBeforePartialImport(String siteName) throws IOException
    {
        if (_siteManager.hasSite(siteName))
        {
            Node siteNode = _siteManager.getSite(siteName).getNode();
            try
            {
                Session parentNodeSession = siteNode.getParent().getSession();
                siteNode.remove();
                parentNodeSession.save();
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
    }
    
    private Collection<String> _retrieveSiteNames(Collection<String> partialImports)
    {
        return partialImports.stream()
                .map(partialImport -> StringUtils.substringAfter(partialImport, __PARTIAL_IMPORT_PREFIX))
                .collect(Collectors.toList());
    }
    
    private ImportReport _partialImport(Path zipPath, Collection<String> partialImports, Merger merger, ModifiableTraversableAmetysObject siteRoot) throws IOException
    {
        try
        {
            var importer = new SiteImporter(this, siteRoot, zipPath, merger, getLogger());
            if (partialImports.contains(ID))
            {
                importer.importSites();
            }
            else
            {
                for (String siteName : _retrieveSiteNames(partialImports))
                {
                    importer.importSite(siteName);
                }
            }
            return importer._report;
        }
        catch (ParserConfigurationException e)
        {
            throw new IOException(e);
        }
    }
    
    private void _saveImported(ModifiableAmetysObject siteRoot)
    {
        if (siteRoot.needsSave())
        {
            getLogger().warn(Archivers.WARN_MESSAGE_ROOT_HAS_PENDING_CHANGES, siteRoot);
            siteRoot.saveChanges();
        }
    }
    
    private final class SitesArchiverManifestReaderWriter implements ManifestReaderWriter
    {
        @Override
        public Object getData()
        {
            return _siteManager.getSiteNames();
        }
        
        @Override
        public Stream<PartialImport> toPartialImports(Object data)
        {
            return Stream.concat(
                    _allSitesToPartialImports(),
                    _sitesDataToPartialImports(data));
        }
        
        private Stream<PartialImport> _allSitesToPartialImports()
        {
            return Stream.of(PartialImport.of(ID, new I18nizableText("plugin.web-contentio", "PLUGINS_WEB_CONTENTIO_ARCHIVE_IMPORT_SITE_ARCHIVER_OPTION_ALL"))); // for importing all sites
        }
        
        @SuppressWarnings("synthetic-access")
        private Stream<PartialImport> _sitesDataToPartialImports(Object data)
        {
            return Optional.ofNullable(data)
                    .filter(Collection.class::isInstance)
                    .map(this::_castToCollection)
                    .orElseGet(() ->
                    {
                        getLogger().warn("Unexpected manifest data '{}', we would expect an array representing the site names. The ZIP archive probably comes from a different version of Ametys.", data);
                        return Collections.emptySet();
                    })
                    .stream()
                    .sorted() // sort by site name
                    .map(this::_toPartialImport);
        }
        
        private Collection<String> _castToCollection(Object data)
        {
            return Collection.class.cast(data);
        }
        
        private PartialImport _toPartialImport(String siteName)
        {
            String key = __PARTIAL_IMPORT_PREFIX + siteName;
            return PartialImport.of(key, _toPartialImportLabel(siteName));
        }
        
        private I18nizableText _toPartialImportLabel(String siteName)
        {
            // "Site 'wwww'"
            return new I18nizableText("plugin.web-contentio", "PLUGINS_WEB_CONTENTIO_ARCHIVE_IMPORT_SITE_ARCHIVER_OPTION_ONE_SITE", Map.of("siteName", new I18nizableText(siteName)));
        }
    }
}
