/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio;

import java.io.File;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.Constants;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.Context;
import org.apache.commons.io.FileUtils;
import org.apache.excalibur.source.SourceUtil;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.WorkflowAwareContent;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightsException;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableAmetysObject;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workflow.support.WorkflowProvider.AmetysObjectWorkflow;
import org.ametys.web.repository.content.ModifiableWebContent;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

import com.opensymphony.workflow.WorkflowException;

/**
 * Manager for importing contents
 */
public class ContentIOManager extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = ContentIOManager.class.getName();
    
    private Context _context;

    private ContentImporterExtensionPoint _contentImporterExtensionPoint;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;
    private RightManager _rightsManager;
    private WorkflowProvider _workflowProvider;

    private CurrentUserProvider _currentUserProvider;

    private AmetysObjectResolver _resolver;

    @Override
    public void contextualize(org.apache.avalon.framework.context.Context context) throws ContextException
    {
        _context = (Context) context.get(Constants.CONTEXT_ENVIRONMENT_CONTEXT);
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _contentImporterExtensionPoint = (ContentImporterExtensionPoint) manager.lookup(ContentImporterExtensionPoint.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _rightsManager = (RightManager) manager.lookup(RightManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    /**
     * Import recursively documents contained in a directory on the server
     * Created contents are of type 'article' and pages are also created upon successful content creation.
     * @param directoryPath The absolute path of directory on the server
     * @param rootPageId The id of parent page
     * @return An array containing the number of successful and unsuccessful content created.
     * @throws AmetysRepositoryException If an error occurred
     * @throws ProcessingException if an error occurs during processing.
     */
    @Callable(rights = "Plugins_ContentIO_ImportDirectory")
    public Map<String, Object> importContent (String directoryPath, String rootPageId) throws ProcessingException, AmetysRepositoryException
    {
        UserIdentity user = _currentUserProvider.getUser();
        
        SitemapElement rootPage = _resolver.resolveById(rootPageId);
        
        if (!(rootPage instanceof ModifiableAmetysObject))
        {
            throw new ProcessingException("The selected page '" + rootPage.getPath() + "' is not modifiable.");
        }
        
        int[] result = importContent(directoryPath, rootPage, user);
        
        Map<String, Object> mapResult = new HashMap<>();
        mapResult.put("success", result[0]);
        mapResult.put("error", result[1]);
        
        return mapResult;
    }
    
    /**
     * Import recursively documents contained in a directory on the server
     * Created contents are of type 'article' and pages are also created upon successful content creation.
     * @param path The absolute path of directory on the server
     * @param rootPage The parent page of the new contents
     * @param user The user responsible for contents import
     * @return An array containing the number of successful and unsuccessful content created.
     * @throws ProcessingException if an error occurs during processing.
     */
    public int[] importContent(String path, SitemapElement rootPage, UserIdentity user) throws ProcessingException
    {
        int[] result = new int[2];
        
        File file = new File(path);
        if (!file.exists())
        {
            throw new ProcessingException("The file '" + path + "' does not exist.");
        }
        
        if (file.isFile())
        {
            Page page = _importFromFile(file, rootPage, user, false);
            if (page != null)
            {
                result[0]++;
            }
            else
            {
                result[1]++;
            }
        }
        else
        {
            result =  _importFromDirectory(file, rootPage, user, false);
        }
        
        if (rootPage.getSite().needsSave())
        {
            rootPage.getSite().saveChanges();
        }
        
        return result;
    }
    
    /**
     * Import an InputStream as a content into a new page
     * @param is The input stream to import
     * @param mimeType The Mime type of the input stream
     * @param contentName The name associated with the input stream
     * @param user The creator
     * @param rootPage The parent page of the newly created page
     * @param isContextExtern True if the current context is not in a site
     * @return The created page or null if failed to create the page
     * @throws IOException if an error occurs with the temporary file.
     */
    public ModifiablePage importContent(InputStream is, String mimeType, String contentName, UserIdentity user, SitemapElement rootPage, boolean isContextExtern) throws IOException
    {
        File tmpFile = null;
        try
        {
            tmpFile = new File(System.getProperty("java.io.tmpdir") + File.separator + Long.toString(Math.round(Math.random() * 1000000.0)), contentName);
            
            tmpFile.getParentFile().mkdirs();
            tmpFile.createNewFile();
            
            try (FileOutputStream tmpFileOS = new FileOutputStream(tmpFile))
            {
                SourceUtil.copy(is, tmpFileOS);
            }
            
            ModifiablePage page = _importFromFile(tmpFile, rootPage, user, isContextExtern);
            
            if (rootPage.getSite().needsSave())
            {
                rootPage.getSite().saveChanges();
            }
            
            return page;
        }
        catch (FileNotFoundException e)
        {
            getLogger().warn("Unable to create a temporary file " + tmpFile.getAbsolutePath() + " to import.");
        }
        finally
        {
            FileUtils.deleteQuietly(tmpFile);
        }
        
        return null;
    }
    
    private int[] _importFromDirectory(File dir, SitemapElement rootPage, UserIdentity user, boolean isContextExtern)
    {
        int[] result = new int[2];
        
        File[] children = dir.listFiles(new FilenameFilter()
        {
            // Get all children except child the same directory name (if exists)
            @Override
            public boolean accept(File directory, String name)
            {
                return !name.startsWith(directory.getName() + '.');
            }
        });
        
        for (File child : children)
        {
            if (child.isFile())
            {
                Page page = _importFromFile(child, rootPage, user, isContextExtern);
                if (page != null)
                {
                    result[0]++;
                }
                else
                {
                    result[1]++;
                }
            }
            else if (child.isDirectory())
            {
                File[] subChildren = child.listFiles(new FilenameFilter()
                {
                    @Override
                    public boolean accept(File directory, String name)
                    {
                        return name.startsWith(directory.getName() + '.');
                    }
                });
                
                boolean contentImported = false;
                ModifiablePage childPage = null;
                
                if (subChildren.length > 0)
                {
                    childPage = _importFromFile(subChildren[0], rootPage, user, isContextExtern);
                    if (childPage != null)
                    {
                        result[0]++;
                    }
                    else
                    {
                        result[1]++;
                    }
                    
                    contentImported = childPage != null;
                }
                
                if (!contentImported)
                {
                    // no child content with the same name
                    
                    String pageTitle = child.getName();
                    
                    String originalPageName = NameHelper.filterName(pageTitle);
                    
                    String pageName = originalPageName;
                    int index = 2;
                    while (rootPage.hasChild(pageName))
                    {
                        pageName = originalPageName + "-" + (index++);
                    }
                    
                    childPage = ((ModifiableTraversableAmetysObject) rootPage).createChild(pageName, "ametys:defaultPage");
                    childPage.setType(PageType.NODE);
                    childPage.setSiteName(rootPage.getSiteName());
                    childPage.setSitemapName(rootPage.getSitemapName());
                    childPage.setTitle(child.getName());
                }
                
                int[] subResult = _importFromDirectory(child, childPage, user, isContextExtern);
                result[0] += subResult[0];
                result[1] += subResult[1];
            }
        }
        
        return result;
    }
    
    /**
     * Import a file into a new content and a new page.
     * @param file The file to import
     * @param rootPage The parent page of the newly created page
     * @param user The user
     * @param isContextExtern True if the current context is not in a site
     * @return True if the content and page was successfully created.
     */
    private ModifiablePage _importFromFile(File file, SitemapElement rootPage, UserIdentity user, boolean isContextExtern)
    {
        String mimeType = _context.getMimeType(file.getName());
        ContentImporter importer = _contentImporterExtensionPoint.getContentImporterForMimeType(mimeType);
        
        if (importer == null)
        {
            getLogger().warn("Unable to import file " + file.getAbsolutePath() + ": no importer found.");
            return null;
        }
        
        try
        {
            Map<String, String> params = new HashMap<>();
            // import content
            
            Content content = _convertFileToContent(importer, file, rootPage.getSite(), rootPage.getSitemap(), user, params);
            if (content != null)
            {
                if (!hasRight(content.getTypes()[0], rootPage, user, isContextExtern))
                {
                    throw new RightsException("insufficient rights to create a new page and content for user '" + UserIdentity.userIdentityToString(user) + "'.");
                }
                
                // content has been imported, create the page
                ModifiablePage page = _createPageAndSetContent(rootPage, content, rootPage.getSite(), rootPage.getSitemap(), params, file);
                importer.postTreatment(page, content, file);
                return page;
            }
        }
        catch (AmetysRepositoryException e)
        {
            getLogger().error("Unable to import content from file " + file.getAbsolutePath(), e);
        }
        catch (IOException e)
        {
            getLogger().error("Unable to import content from file " + file.getAbsolutePath(), e);
        }
        catch (WorkflowException e)
        {
            getLogger().error("Unable to import content from file " + file.getAbsolutePath(), e);
        }

        return null;
    }
    
    private Content _convertFileToContent(ContentImporter importer, File file, Site site, Sitemap sitemap, UserIdentity user, Map<String, String> params) throws WorkflowException, AmetysRepositoryException
    {
        // Creates a workflow entry
        AmetysObjectWorkflow workflow = _workflowProvider.getAmetysObjectWorkflow();
        long workflowId = workflow.initialize("content", 0, new HashMap<>());
        
        ModifiableTraversableAmetysObject rootContents = site.getRootContents();
        
        String fileName = file.getName();
        
        // find an available name and creates the content
        int i = fileName.lastIndexOf('.');
        String desiredContentName = i != -1 ? fileName.substring(0, i) : fileName;
        
        String originalContentName = NameHelper.filterName(desiredContentName);
        
        String contentName = originalContentName;
        int index = 2;
        while (rootContents.hasChild(contentName))
        {
            contentName = originalContentName + "-" + (index++);
        }

        ModifiableWebContent content = rootContents.createChild(contentName, RepositoryConstants.NAMESPACE_PREFIX + ":defaultWebContent");
        
        ((WorkflowAwareContent) content).setWorkflowId(workflowId);

        // Common metadata
        content.setCreator(user);
        content.setCreationDate(ZonedDateTime.now());
        content.setLastContributor(user);
        content.setLastModified(ZonedDateTime.now());
        content.setLanguage(sitemap.getName());
        content.setSiteName(site.getName());
        
        try
        {
            importer.importContent(file, content, params);
        }
        catch (IOException e)
        {
            getLogger().error("Unable to import content from file " + file.getAbsolutePath(), e);
            content.remove();
            return null;
        }
        
        return content;
    }
    
    private ModifiablePage _createPageAndSetContent(SitemapElement rootPage, Content content, Site site, Sitemap sitemap, Map<String, String> params, File file)
    {
        String template = params.get("page.template");
        
        String fileName = file.getName();
        int i = fileName.lastIndexOf('.');
        
        String pageTitle = i != -1 ? fileName.substring(0, i) : fileName;
        
        String originalPageName = NameHelper.filterName(pageTitle);
        
        String pageName = originalPageName;
        int index = 2;
        while (rootPage.hasChild(pageName))
        {
            pageName = originalPageName + "-" + (index++);
        }
        
        ModifiablePage page = ((ModifiableTraversableAmetysObject) rootPage).createChild(pageName, "ametys:defaultPage");
        page.setType(PageType.CONTAINER);
        page.setTemplate(template == null ? "page" : template);
        page.setSiteName(site.getName());
        page.setSitemapName(sitemap.getName());
        page.setTitle(pageTitle);
        
        String longTitle = params.get("page.longTitle");
        if (longTitle != null)
        {
            page.setLongTitle(longTitle);
        }
        
        ModifiableZone zone = page.createZone("default");
        ModifiableZoneItem zoneItem = zone.addZoneItem();
        zoneItem.setType(ZoneType.CONTENT);
        zoneItem.setContent(content);
        
        return page;
    }
    
    /**
     * Test if the current user has the right needed by the content type to create a content.
     * @param contentTypeId the content type ID.
     * @param page the current page.
     * @param user The user
     * @param isContextExtern True if the current context is not in a site
     * @return true if the user has the right needed, false otherwise.
     */
    protected boolean hasRight(String contentTypeId, SitemapElement page, UserIdentity user, boolean isContextExtern)
    {
        ContentType contentType = _contentTypeExtensionPoint.getExtension(contentTypeId);
        if (contentType != null)
        {
            String right = contentType.getRight();
            return right == null || _rightsManager.hasRight(user, right, page) == RightResult.RIGHT_ALLOW;
        }
        
        return false;
    }
}
