/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.IntStream;

import javax.jcr.Node;
import javax.jcr.RepositoryException;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.transform.TransformerException;

import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.apache.xpath.XPathAPI;
import org.apache.xpath.objects.XObject;
import org.slf4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import org.ametys.cms.repository.Content;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.contentio.archive.ArchiveHandler;
import org.ametys.plugins.contentio.archive.Archivers;
import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.contentio.archive.ImportGlobalFailException;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.ImportReport.ImportError;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.UnitaryImporter;
import org.ametys.plugins.contentio.archive.ZipEntryHelper;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.ModelLessXMLValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.XMLValuesExtractorAdditionalDataGetter;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.DefaultTraversableAmetysObject;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.runtime.model.View;
import org.ametys.web.parameters.view.ViewParametersManager;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page.LinkType;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.SitemapElement;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.page.jcr.DefaultPage;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.service.Service;

import com.google.common.io.MoreFiles;

class SitemapImporter
{
    final ImportReport _report = new ImportReport();
    private final SitesArchiver _siteArchiver;
    private final Site _site;
    private final Path _sitemapFolderPath;
    private final Path _sitemapFilePath;
    private final Path _zipArchivePath;
    private final Merger _merger;
    private final Logger _logger;
    private final DocumentBuilder _builder;
    private Sitemap _sitemap;
    private final UnitarySitemapImporter _unitarySitemapImporter = new UnitarySitemapImporter();
    
    SitemapImporter(SitesArchiver siteArchiver, Site site, Path sitemapPath, Path zipArchivePath, Merger merger, Logger logger, DocumentBuilder builder)
    {
        _siteArchiver = siteArchiver;
        _site = site;
        _sitemapFolderPath = sitemapPath;
        _sitemapFilePath = _sitemapFolderPath.resolveSibling(_sitemapFolderPath.getFileName() + ".xml");
        _zipArchivePath = zipArchivePath;
        _merger = merger;
        _logger = logger;
        _builder = builder;
    }
    
    private class UnitaryPageImporter implements UnitaryImporter<Node>
    {
        private Document _propertiesXml;
        
        UnitaryPageImporter(Document propertiesXml)
        {
            _propertiesXml = propertiesXml;
        }

        @Override
        public String objectNameForLogs()
        {
            return "Page";
        }

        @Override
        public Document getPropertiesXml(Path zipEntryPath) throws Exception
        {
            return _propertiesXml;
        }

        @Override
        public String retrieveId(Document propertiesXml) throws Exception
        {
            return Archivers.xpathEvalNonEmpty("page/@id", propertiesXml);
        }

        @Override
        public Node create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
        {
            Node pageNode = _createPage(zipEntryPath, id, propertiesXml);
            _createPageAcl(pageNode, zipEntryPath);
            Archivers.unitarySave(pageNode, _logger);
            return pageNode;
        }
        
        @Override
        public ImportReport getReport()
        {
            return _report;
        }
    }
    
    void importSitemap() throws RepositoryException, IOException
    {
        boolean sitemapImported = _fillSitemap();
        if (sitemapImported)
        {
            _importAllPages();
        }
    }
    
    private boolean _fillSitemap() throws RepositoryException, IOException
    {
        Optional<Sitemap> optionalSitemap = _unitarySitemapImporter.unitaryImport(_zipArchivePath, _sitemapFilePath, _merger, _logger);
        if (optionalSitemap.isEmpty())
        {
            // potentially, sitemap was not imported => it was already logged in error level
            // but fail, do not import pages
            return false;
        }
        
        _sitemap = optionalSitemap.get();
        _createSitemapAcl();
        try
        {
            Archivers.unitarySave(_sitemap.getNode(), _logger);
            return true;
        }
        catch (AmetysObjectNotImportedException e)
        {
            // potentially, sitemap was not imported => it was already logged in error level
            // but fail, do not import pages
            return false;
        }
    }
    
    private Sitemap _createChildSitemap(String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
    {
        String sitemapName = Archivers.xpathEvalNonEmpty("sitemap/@name", propertiesXml);
        _logger.info("Creating a Sitemap object for '{}' file (lang={})", _sitemapFilePath, sitemapName);
        String uuid = StringUtils.substringAfter(id, "://");
        Node sitemapNode = _createChildSitemap(uuid, sitemapName);
        Sitemap sitemap = _resolveSitemap(sitemapNode);
        _setSitemapAttributes(sitemap, propertiesXml);
        Archivers.unitarySave(sitemapNode, _logger);
        
        return sitemap;
    }
    
    private Node _createChildSitemap(String uuid, String sitemapName) throws RepositoryException
    {
        // Create a Node with JCR primary type "sitemap"
        // But then call 'replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
        Sitemap srcSitemap = _site.addSitemap(sitemapName);
        Node srcNode = srcSitemap.getNode();
        Node nodeWithDesiredUuid = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
        return nodeWithDesiredUuid;
    }
    
    private Sitemap _resolveSitemap(Node node)
    {
        return _siteArchiver._sitemapFactory.getAmetysObject(node, null);
    }
    
    private void _setSitemapAttributes(Sitemap sitemap, Document propertiesXml) throws TransformerException, RepositoryException
    {
        _setAmetysInternalProperties(sitemap, propertiesXml, "sitemap");
    }
    
    private void _createSitemapAcl() throws RepositoryException, IOException
    {
        Node node = _sitemap.getNode();
        String zipEntryPath = new StringBuilder(ArchiveHandler.METADATA_PREFIX)
                .append(StringUtils.strip(_sitemapFolderPath.toString(), "/"))
                .append("/")
                .append(SitesArchiver.__ACL_ZIP_ENTRY_FILENAME)
                .toString();
        _logger.debug("Trying to import ACL node for Sitemap '{}', from ACL XML file '{}', if it exists", node, zipEntryPath);
        Archivers.importAcl(node, _zipArchivePath, _merger, zipEntryPath, _logger);
    }
    
    private void _createPageAcl(Node pageNode, Path pagePath) throws RepositoryException, IOException
    {
        String zipEntryPath = new StringBuilder(ArchiveHandler.METADATA_PREFIX)
                .append(StringUtils.strip(pagePath.getParent().toString(), "/"))
                .append("/")
                .append(MoreFiles.getNameWithoutExtension(pagePath))
                .append("/")
                .append(SitesArchiver.__ACL_ZIP_ENTRY_FILENAME)
                .toString();
        _logger.debug("Trying to import ACL node for Page '{}', from ACL XML file '{}', if it exists", pageNode, zipEntryPath);
        Archivers.importAcl(pageNode, _zipArchivePath, _merger, zipEntryPath, _logger);
    }
    
    private void _importAllPages() throws IOException
    {
        Pair<Path, Document>[] pageXmlFiles = ZipEntryHelper.zipFileTree(
            _zipArchivePath,
            Optional.of(_sitemapFolderPath.toString()),
            (p, attrs) -> attrs.isRegularFile())
                .map(path -> Pair.of(path, _getNullablePagePropertiesXml(path)))
                // filter out XML properties files which are not present (likely due to an error reading them)
                .filter(pathAndXml -> pathAndXml.getRight() != null)
                .sorted(_pageComparator())
                .toArray(Pair[]::new);
        
        for (Pair<Path, Document> pagePathAndXml : pageXmlFiles)
        {
            Path pagePath = pagePathAndXml.getLeft();
            Document propertiesXml = pagePathAndXml.getRight();
            _importPage(pagePath, propertiesXml);
        }
    }
    
    private Comparator<Pair<Path, Document>> _pageComparator()
    {
        // Sort by path name count, i.e. /foo/bar.xml should be before /foo/bar/baz.xml
        // Thus, it will ensure parent pages are created when processing a specific XML page file
        Comparator<Path> firstComparator = Comparator.comparingInt(Path::getNameCount);
        
        // Sort by the complete string representation of the parent Path
        // Just for debug purposes
        // So as to export /foo/aaa/one then /foo/aaa/two and then /foo/zzz/one
        // Instead of /foo/aaa/one then /foo/zzz/one and then /foo/aaa/two
        Comparator<Path> secondComparator = Comparator.comparing(path -> path.getParent().toString());
        
        // Sort by order, to keep the order of sibling pages
        Comparator<Document> thirdComparator = Comparator.comparingInt(this::_getOrder);
        
        return Comparator.comparing(Pair<Path, Document>::getLeft, firstComparator)
                .thenComparing(Comparator.comparing(Pair<Path, Document>::getLeft, secondComparator))
                .thenComparing(Comparator.comparing(Pair<Path, Document>::getRight, thirdComparator));
    }
    
    private int _getOrder(Document propertiesXml)
    {
        try
        {
            return Integer.parseInt(Archivers.xpathEvalNonEmpty("page/@order", propertiesXml));
        }
        catch (NumberFormatException | AmetysObjectNotImportedException | TransformerException e)
        {
            throw new LambdaUtils.LambdaException(e);
        }
    }
    
    private void _importPage(Path page, Document propertiesXml) throws ImportGlobalFailException
    {
        new UnitaryPageImporter(propertiesXml)
                .unitaryImport(_zipArchivePath, page, _merger, _logger);
    }
    
    private Document _getNullablePagePropertiesXml(Path page)
    {
        try
        {
            return _getPropertiesXml(page);
        }
        catch (Exception e)
        {
            _logger.error("An unexpected exception occured when trying to import page for '{}!{}'.", _zipArchivePath, page, e);
            _report.addError(new ImportError(e));
            return null;
        }
    }
    
    private Document _getPropertiesXml(Path pageOrSitemap)
    {
        String zipEntryPath = pageOrSitemap.toString();
        try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipArchivePath, zipEntryPath))
        {
            Document doc = _builder.parse(stream);
            return doc;
        }
        catch (SAXException | IOException e)
        {
            throw new RuntimeException(String.format("The exported page/sitemap '%s' could not be read properly.", pageOrSitemap), e);
        }
    }
    
    private Node _createPage(Path pagePath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, Exception
    {
        Path parentPageOrSitemap = pagePath.getParent();
        String parentRelPath = _relativePath(parentPageOrSitemap);
        Node parentNode = _retrieveParentSitemapElement(parentRelPath).getNode();
        String uuid = StringUtils.substringAfter(id, "://");
        String pageName = MoreFiles.getNameWithoutExtension(pagePath);
        _logger.info("Creating a Page object for '{}' file (id={}, name={}, parent={})", pagePath, id, pageName, parentNode);
        
        Node pageNode = _createChildPage(parentNode, uuid, pageName);
        ModifiablePage createdPage = _resolvePage(pageNode);
        _setPageAttributes(createdPage, propertiesXml);
        _setTags(createdPage, propertiesXml);
        _setTemplateParameters(createdPage, propertiesXml);
        _setZones(createdPage, propertiesXml);
        Archivers.unitarySave(pageNode, _logger);
        
        ImportReport importAttachmentReport = _setPageAttachments(createdPage, pagePath);
        _report.addFrom(importAttachmentReport);
        
        return pageNode;
    }
    
    private String _relativePath(Path pageOrSitemap)
    {
        // for instance, _sitemapFolderPath="/sites/45/7d/www/pages/fr"
        // relPath=pageOrSitemap.toString()="/sites/45/7d/www/pages/fr/foo/bar"
        // it should return "foo/bar"
        
        // for instance, _sitemapFolderPath="/sites/45/7d/www/pages/fr"
        // relPath=pageOrSitemap.toString()=""/sites/45/7d/www/pages/fr"
        // it should return ""
        
        // Cannot use _sitemapFolderPath.relativize(pageOrSitemap) as the two ZipPaths do not have the same FileSystem reference :-(
        // (created with different calls to ZipEntryHelper)
        String relPath = StringUtils.substringAfter(pageOrSitemap.toString(), _sitemapFolderPath.toString());
        relPath = StringUtils.strip(relPath, "/");
        return relPath;
    }
    
    private DefaultTraversableAmetysObject _retrieveParentSitemapElement(String relPath)
    {
        return relPath.isEmpty()
                ? _sitemap
                : _siteArchiver._defaultPageFactory.getChild(_sitemap, relPath);
    }
    
    private Node _createChildPage(Node parentNode, String uuid, String pageName) throws RepositoryException
    {
        // Create a Node with JCR primary type "ametys:defaultPage"
        // But then call 'replaceNodeWithDesiredUuid' to have it with the desired UUID (srcNode will be removed)
        Node srcNode = parentNode.addNode(pageName, "ametys:defaultPage");
        Node nodeWithDesiredUuid = Archivers.replaceNodeWithDesiredUuid(srcNode, uuid);
        return nodeWithDesiredUuid;
    }
    
    private ModifiablePage _resolvePage(Node node)
    {
        return _siteArchiver._defaultPageFactory.getAmetysObject(node, null);
    }
    
    private void _setPageAttributes(ModifiablePage page, Document propertiesXml) throws AmetysObjectNotImportedException, TransformerException, Exception
    {
        String title = Archivers.xpathEvalNonEmpty("page/@title", propertiesXml);
        page.setTitle(title);
        
        String longTitle = Archivers.xpathEvalNonEmpty("page/@long-title", propertiesXml);
        page.setLongTitle(longTitle);
        
        PageType type = PageType.valueOf(Archivers.xpathEvalNonEmpty("page/@type", propertiesXml));
        page.setType(type);
        
        switch (type)
        {
            case LINK:
                String url = Archivers.xpathEvalNonEmpty("page/@url", propertiesXml);
                LinkType urlType = LinkType.valueOf(Archivers.xpathEvalNonEmpty("page/@urlType", propertiesXml));
                page.setURL(urlType, url);
                break;
            case CONTAINER:
                String template = Archivers.xpathEvalNonEmpty("page/@template", propertiesXml);
                page.setTemplate(template);
                break;
            default:
                break;
        }
        
        _setAmetysInternalProperties(page, propertiesXml, "page");
        
        Element pageAttributesElement = (Element) XPathAPI.selectSingleNode(propertiesXml, "page/attributes");
        Map<String, Object> values = new ModelLessXMLValuesExtractor(pageAttributesElement, (dataPath, dataType) -> Optional.empty(), _siteArchiver._pageDataTypeExtensionPoint)
                .extractValues();
        
        page.synchronizeValues(values);
    }
    
    private void _setAmetysInternalProperties(SitemapElement sitemapElement, Document propertiesXml, String rootTagName) throws TransformerException, RepositoryException
    {
        if (sitemapElement instanceof JCRAmetysObject)
        {
            Node sitemapElementNode = ((JCRAmetysObject) sitemapElement).getNode();
            String xpath = String.format("%s/internal/*", rootTagName);
            NodeList nodeList = XPathAPI.selectNodeList(propertiesXml, xpath);
            Function<org.w3c.dom.Node, String> getInternalPropertyName = org.w3c.dom.Node::getNodeName;
            Function<org.w3c.dom.Node, String> getInternalPropertyValue = org.w3c.dom.Node::getTextContent;
            
            Map<String, List<String>> internalPropertiesToSet = IntStream.range(0, nodeList.getLength())
                    .mapToObj(nodeList::item)
                    .collect(Collectors.groupingBy(
                            getInternalPropertyName,
                            Collectors.mapping(
                                getInternalPropertyValue,
                                Collectors.toList())));
            
            for (Map.Entry<String, List<String>> internalProperty : internalPropertiesToSet.entrySet())
            {
                String internalPropertyName = internalProperty.getKey();
                List<String> internalPropertyValue = internalProperty.getValue();
                _logger.debug("Adding to '{}' internal property '{}' with value '{}'", sitemapElement, internalPropertyName, internalPropertyValue);
                sitemapElementNode.setProperty(internalPropertyName, internalPropertyValue.toArray(String[]::new));
            }
        }
    }
    
    private void _setTags(ModifiablePage page, Document propertiesXml) throws TransformerException
    {
        NodeList nodeList = XPathAPI.selectNodeList(propertiesXml, "page/tags/*");
        String[] tags = IntStream.range(0, nodeList.getLength())
                .mapToObj(nodeList::item)
                .map(org.w3c.dom.Node::getNodeName)
                .toArray(String[]::new);
        for (String tag : tags)
        {
            _logger.debug("Tagging page '{}' with tag '{}'", page, tag);
            page.tag(tag);
        }
    }
    
    private void _setTemplateParameters(ModifiablePage page, Document propertiesXml) throws Exception
    {
        Element templateParamsElement = (Element) XPathAPI.selectSingleNode(propertiesXml, "page/templateParameters");
        if (templateParamsElement != null)
        {
            ModifiableModelAwareDataHolder templateParameters = page.getTemplateParametersHolder();
            
            Map<String, Object> values = new ModelAwareXMLValuesExtractor(templateParamsElement, templateParameters.getModel())
                    .extractValues();
            
            templateParameters.synchronizeValues(values);
        }
    }
    
    private ImportReport _setPageAttachments(ModifiablePage page, Path pagePath) throws IOException, RepositoryException
    {
        if (page instanceof DefaultPage)
        {
            Node pageNode = ((DefaultPage) page).getNode();
            
            // in case ametys-internal:attachments is already created
            if (pageNode.hasNode(DefaultPage.ATTACHMENTS_NODE_NAME))
            {
                pageNode.getNode(DefaultPage.ATTACHMENTS_NODE_NAME).remove();
            }
            
            String commonPrefix = new StringBuilder()
                    .append(ArchiveHandler.METADATA_PREFIX)
                    .append(StringUtils.strip(pagePath.getParent().toString(), "/"))
                    .append("/")
                    .append(MoreFiles.getNameWithoutExtension(pagePath))
                    .append("/")
                    .append("_attachments")
                    .append("/")
                    .toString();
            return _siteArchiver._resourcesArchiverHelper.importCollection(commonPrefix, pageNode, _zipArchivePath, _merger);
        }
        
        return new ImportReport();
    }
    
    private void _setZones(ModifiablePage page, Document propertiesXml) throws TransformerException, AmetysObjectNotImportedException, Exception
    {
        NodeList zoneNodeList = XPathAPI.selectNodeList(propertiesXml, "page/pageContents/zone");
        for (int i = 0; i < zoneNodeList.getLength(); i++)
        {
            org.w3c.dom.Node zoneNode = zoneNodeList.item(i);
            _setZone(page, zoneNode);
        }
    }
    
    private void _setZone(ModifiablePage page, org.w3c.dom.Node zoneNode) throws TransformerException, AmetysObjectNotImportedException, Exception
    {
        String zoneName = Archivers.xpathEvalNonEmpty("@name", zoneNode);
        ModifiableZone zone = page.createZone(zoneName);
        
        Element zoneAttributesElement = (Element) XPathAPI.selectSingleNode(zoneNode, "attributes");
        Map<String, Object> values = new ModelLessXMLValuesExtractor(zoneAttributesElement, (dataPath, dataType) -> Optional.empty(), _siteArchiver._pageDataTypeExtensionPoint)
                .extractValues();
        zone.synchronizeValues(values);
        
        _setZoneParameters(zone, zoneNode);
        
        NodeList zoneItemNodeList = XPathAPI.selectNodeList(zoneNode, "zoneItem");
        for (int i = 0; i < zoneItemNodeList.getLength(); i++)
        {
            org.w3c.dom.Node zoneItemNode = zoneItemNodeList.item(i);
            _setZoneItem(zone, zoneItemNode);
        }
    }
    
    private void _setZoneParameters(ModifiableZone zone, org.w3c.dom.Node zoneItemNode) throws TransformerException, Exception
    {
        Element zoneParamsElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "zoneParameters");
        if (zoneParamsElement != null)
        {
            ModifiableModelAwareDataHolder zoneParameters = zone.getZoneParametersHolder();
            Map<String, Object> values = new ModelAwareXMLValuesExtractor(zoneParamsElement, zoneParameters.getModel())
                    .extractValues();
            zoneParameters.synchronizeValues(values);
        }
    }
    
    private void _setZoneItem(ModifiableZone zone, org.w3c.dom.Node zoneItemNode) throws TransformerException, Exception
    {
        ModifiableZoneItem zoneItem = zone.addZoneItem();
        ZoneType zoneType = ZoneType.valueOf(Archivers.xpathEvalNonEmpty("@type", zoneItemNode));
        zoneItem.setType(zoneType);
        _setZoneItemAttributes(zoneItem, zoneItemNode);
        _setZoneItemParameters(zoneItem, zoneItemNode);
        switch (zoneType)
        {
            case CONTENT:
                _setZoneItemContent(zoneItem, zoneItemNode);
                _setZoneItemContentViewParameters(zoneItem, zoneItemNode);
                break;
            case SERVICE:
                String serviceId = Archivers.xpathEvalNonEmpty("@serviceId", zoneItemNode);
                zoneItem.setServiceId(serviceId);
                _setZoneItemServiceParams(zoneItem, zoneItemNode, serviceId);
                _setZoneItemServiceViewParameters(zoneItem, zoneItemNode, serviceId);
                break;
            default:
                break;
        }
    }
    
    private void _setZoneItemAttributes(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode) throws TransformerException, Exception
    {
        Element zoneItemAttributesElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "attributes");
        Map<String, Object> values = new ModelLessXMLValuesExtractor(zoneItemAttributesElement, (dataPath, type) -> Optional.empty(), _siteArchiver._pageDataTypeExtensionPoint)
                .extractValues();
        zoneItem.synchronizeValues(values);
    }
    
    private void _setZoneItemParameters(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode) throws TransformerException, Exception
    {
        Element zoneItemParamsElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "zoneItemParameters");
        if (zoneItemParamsElement != null)
        {
            ModifiableModelAwareDataHolder zoneItemParameters = zoneItem.getZoneItemParametersHolder();
            Map<String, Object> values = new ModelAwareXMLValuesExtractor(zoneItemParamsElement, zoneItemParameters.getModel())
                    .extractValues();
            zoneItemParameters.synchronizeValues(values);
        }
    }
    
    private void _setZoneItemContent(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode) throws TransformerException, AmetysObjectNotImportedException
    {
        String contentId = Archivers.xpathEvalNonEmpty("@contentId", zoneItemNode);
        if (_siteArchiver._resolver.hasAmetysObjectForId(contentId))
        {
            Content content = _siteArchiver._resolver.resolveById(contentId);
            zoneItem.setContent(content);
        }
        XObject viewNameXObject = XPathAPI.eval(zoneItemNode, "@viewName");
        if (viewNameXObject.getType() != XObject.CLASS_NULL)
        {
            String viewName = viewNameXObject.str();
            zoneItem.setViewName(viewName);
        }
    }
    
    private void _setZoneItemContentViewParameters(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode) throws TransformerException, Exception
    {
        Element contentViewParamsElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "contentViewParameters");
        if (contentViewParamsElement != null)
        {
            String viewName = StringUtils.defaultIfBlank(zoneItem.getViewName(), "main");
            ModifiableModelAwareDataHolder contentViewParameters = zoneItem.getContentViewParametersHolder(viewName);
            Map<String, Object> values = new ModelAwareXMLValuesExtractor(contentViewParamsElement, contentViewParameters.getModel())
                    .extractValues();
            contentViewParameters.synchronizeValues(values);
        }
    }
    
    private void _setZoneItemServiceParams(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode, String serviceId) throws Exception
    {
        Element serviceParamsElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "serviceParameters");
        Service service = _getService(serviceId);
        if (service == null)
        {
            String message = String.format("Service with id '%s' for ZoneItem \"%s\" does not exist", serviceId, zoneItem);
            _logger.error(message);
            _report.addError(new ImportError(new Exception(message)));
        }
        else
        {
            XMLValuesExtractorAdditionalDataGetter additionalDataGetter = (dataPath, type) -> Optional.empty();
            View view = View.of(service);
            Map<String, Object> values = new ModelAwareXMLValuesExtractor(serviceParamsElement, additionalDataGetter, service)
                    .extractValues(view);
            
            ModifiableModelAwareDataHolder serviceParametersDataHolder = zoneItem.getServiceParameters();
            serviceParametersDataHolder.synchronizeValues(view, values);
        }
    }
    
    private void _setZoneItemServiceViewParameters(ModifiableZoneItem zoneItem, org.w3c.dom.Node zoneItemNode, String serviceId) throws TransformerException, Exception
    {
        Element serviceViewParamsElement = (Element) XPathAPI.selectSingleNode(zoneItemNode, "serviceViewParameters");
        Service service = _getService(serviceId);
        // Ignore if the service doesn't exist. Error was already logged
        if (service != null && serviceViewParamsElement != null)
        {
            ModifiableModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
            if (serviceParameters.hasDefinition(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME))
            {
                String viewName = serviceParameters.getValue(ViewParametersManager.SERVICE_VIEW_DEFAULT_MODEL_ITEM_NAME);
                ModifiableModelAwareDataHolder serviceViewParameters = zoneItem.getServiceViewParametersHolder(viewName);
                Map<String, Object> values = new ModelAwareXMLValuesExtractor(serviceViewParamsElement, serviceViewParameters.getModel())
                        .extractValues();
                serviceViewParameters.synchronizeValues(values);
            }
            else
            {
                String message = String.format("Service with id '%s' for ZoneItem \"%s\" doesn't have view but has view parameters.", zoneItem.getServiceId(), zoneItem);
                _logger.error(message);
                _report.addError(new ImportError(new Exception(message)));
            }
        }
    }
    
    private Service _getService(String serviceId)
    {
        return _siteArchiver._serviceExtensionPoint.getExtension(serviceId);
    }
    
    private final class UnitarySitemapImporter implements UnitaryImporter<Sitemap>
    {
        @Override
        public String objectNameForLogs()
        {
            return "Sitemap";
        }

        @Override
        public Document getPropertiesXml(Path zipEntryPath) throws Exception
        {
            return _getPropertiesXml(zipEntryPath);
        }

        @Override
        public String retrieveId(Document propertiesXml) throws Exception
        {
            return Archivers.xpathEvalNonEmpty("sitemap/@id", propertiesXml);
        }

        @Override
        public Sitemap create(Path zipEntryPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, ImportGlobalFailException, Exception
        {
            return _createChildSitemap(id, propertiesXml);
        }

        @Override
        public ImportReport getReport()
        {
            return _report;
        }
    }
}
