/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webcontentio.xml;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.Reader;
import java.io.StringWriter;
import java.io.UnsupportedEncodingException;
import java.lang.reflect.Array;
import java.net.HttpURLConnection;
import java.net.URL;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Properties;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import javax.xml.transform.OutputKeys;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.xml.dom.DOMParser;
import org.apache.excalibur.xml.xpath.XPathProcessor;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.data.Binary;
import org.ametys.cms.data.Geocode;
import org.ametys.cms.data.RichText;
import org.ametys.cms.data.type.ModelItemTypeConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareComposite;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeater;
import org.ametys.plugins.repository.data.holder.group.ModifiableModelAwareRepeaterEntry;
import org.ametys.plugins.repository.model.CompositeDefinition;
import org.ametys.plugins.repository.model.RepeaterDefinition;
import org.ametys.plugins.webcontentio.ContentImporter;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.ModelItem;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.web.repository.content.ModifiableWebContent;
import org.ametys.web.repository.page.ModifiablePage;

/**
 * Default XML content importer
 */
public class XmlContentImporter extends AbstractLogEnabled implements ContentImporter, Serviceable
{
    private DOMParser _domParser;
    private XPathProcessor _xPathProcessor;
    private ContentTypeExtensionPoint _contentTypeExtensionPoint;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _domParser = (DOMParser) manager.lookup(DOMParser.ROLE);
        _xPathProcessor = (XPathProcessor) manager.lookup(XPathProcessor.ROLE);
        _contentTypeExtensionPoint = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
    }
    
    @Override
    public void importContent(File file, ModifiableWebContent content, Map<String, String> params) throws IOException
    {
        Document doc = getXmlDocFromFile(file);
        
        if (doc == null)
        {
            throw new IOException("Unable to retrieve the xml document from the file received.");
        }
        
        Node xmlContent = _xPathProcessor.selectSingleNode(doc, "/content");

        String contentTypeId = _xPathProcessor.evaluateAsString(xmlContent, "@type");
        
        if (StringUtils.isEmpty(contentTypeId))
        {
            throw new IOException("Invalid file content : no content type specified.");
        }
        
        if (!_contentTypeExtensionPoint.hasExtension(contentTypeId))
        {
            throw new IOException("Invalid file content : the specified content type does not exist.");
        }
        
        content.setTypes(new String[] {contentTypeId});
        
        Node title = _xPathProcessor.selectSingleNode(xmlContent, "title");
        
        if (title == null)
        {
            throw new IOException("Invalid file content : no title found, but it is mandatory.");
        }
        
        _importAttributes(content, xmlContent);
    }
    
    @Override
    public String[] getMimeTypes()
    {
        // handles xml mime-types
        return new String[] {"application/xml", "text/xml"};
    }
    
    @Override
    public void postTreatment(ModifiablePage page, Content content, File file) throws IOException
    {
        // Nothing to do
    }
    
    private Document getXmlDocFromFile(File file) throws FileNotFoundException, UnsupportedEncodingException, IOException
    {
        InputStream is = new FileInputStream(file);
        Reader reader = new InputStreamReader(is, "UTF-8");
        Document doc = null;
        try
        {
            doc = _domParser.parseDocument(new InputSource(reader));
        }
        catch (SAXException e)
        {
            getLogger().error("[IMPORT] Unable to parse imported file " + file.getName(), e);
        }
        return doc;
    }
    
    private void _importAttributes(ModifiableWebContent content, Node xmlContent) throws IOException
    {
        NodeList attributesNodes = xmlContent.getChildNodes();
        for (int i = 0; i < attributesNodes.getLength(); i++)
        {
            Node attributeNode = attributesNodes.item(i);
            
            if (attributeNode.getNodeType() == Node.ELEMENT_NODE && content.hasDefinition(attributeNode.getLocalName()))
            {
                ModelItem attributeDefinition = content.getDefinition(attributeNode.getLocalName());
                _importAttribute(content, attributeDefinition, attributeNode);
            }
        }
    }

    @SuppressWarnings("unchecked")
    private void _importAttribute(ModifiableModelAwareDataHolder dataHolder, ModelItem attributeDefinition, Node attributeNode) throws IOException
    {
        if (attributeDefinition != null)
        {
            if (attributeDefinition instanceof RepeaterDefinition)
            {
                _setRepeater(dataHolder, (RepeaterDefinition) attributeDefinition, attributeNode);
            }
            else if (attributeDefinition instanceof CompositeDefinition)
            {
                _setComposite(dataHolder, (CompositeDefinition) attributeDefinition, attributeNode);
            }
            else if (attributeDefinition instanceof ElementDefinition)
            {
                _setAttribute(dataHolder, (ElementDefinition) attributeDefinition, attributeNode);
            }
        }
    }
    
    private void _setRepeater(ModifiableModelAwareDataHolder dataHolder, RepeaterDefinition repeaterDefinition, Node repeaterNode) throws IOException
    {
        NodeList entryNodes = _xPathProcessor.selectNodeList(repeaterNode, "entry");
        if (entryNodes.getLength() > 0)
        {
            ModifiableModelAwareRepeater repeaterData = dataHolder.getRepeater(repeaterDefinition.getName(), true);
            for (int i = 0; i < entryNodes.getLength(); i++)
            {
                Node entryNode = entryNodes.item(i);
                ModifiableModelAwareRepeaterEntry entryData = repeaterData.addEntry();
    
                NodeList subDataNodes = entryNode.getChildNodes();
                for (int j = 0; j < subDataNodes.getLength(); j++)
                {
                    Node subDataNode = subDataNodes.item(j);
                    if (subDataNode.getNodeType() == Node.ELEMENT_NODE)
                    {
                        String subDataName = subDataNode.getLocalName();
                        ModelItem childDefinition = repeaterDefinition.getChild(subDataName);
                        
                        _importAttribute(entryData, childDefinition, subDataNode);
                    }
                }
            }
        }
    }
    
    private void _setComposite(ModifiableModelAwareDataHolder dataHolder, CompositeDefinition compositeDefinition, Node compositeNode) throws IOException
    {
        NodeList subDataNodes = compositeNode.getChildNodes();
        if (subDataNodes.getLength() > 0)
        {
            ModifiableModelAwareComposite compositeData = dataHolder.getComposite(compositeDefinition.getName(), true);
            for (int i = 0; i < subDataNodes.getLength(); i++)
            {
                Node subDataNode = subDataNodes.item(i);
                if (subDataNode.getNodeType() == Node.ELEMENT_NODE)
                {
                    String subDataName = subDataNode.getLocalName();
                    ModelItem childDefinition = compositeDefinition.getChild(subDataName);
                    
                    _importAttribute(compositeData, childDefinition, subDataNode);
                }
            }
        }
    }

    @SuppressWarnings("unchecked")
    private <T> void _setAttribute(ModifiableModelAwareDataHolder dataHolder, ElementDefinition<T> attributeDefinition, Node attributeNode) throws IOException
    {
        ElementType<T> type = attributeDefinition.getType();
        
        if (attributeDefinition.isMultiple())
        {
            NodeList valuesNodeList = _xPathProcessor.selectNodeList(attributeNode, "value");
            List<T> values = new ArrayList<>();
            for (int i = 0; i < valuesNodeList.getLength(); i++)
            {
                _getSingleAttributeValue(valuesNodeList.item(i), type)
                    .ifPresent(value -> values.add(value));
            }
            
            if (!values.isEmpty())
            {
                T[] valuesAsArray = (T[]) Array.newInstance(type.getManagedClass(), values.size());
                dataHolder.setValue(attributeDefinition.getName(), values.toArray(valuesAsArray));
            }
        }
        else
        {
            _getSingleAttributeValue(attributeNode, type)
                .ifPresent(value -> dataHolder.setValue(attributeDefinition.getName(), value));
        }
    }
    
    @SuppressWarnings("unchecked")
    private <T> Optional<T> _getSingleAttributeValue(Node valueNode, ElementType<T> type) throws IOException
    {
        String id = type.getId();
        if (ModelItemTypeConstants.BINARY_ELEMENT_TYPE_ID.equals(id) || ModelItemTypeConstants.FILE_ELEMENT_TYPE_ID.equals(id))
        {
            return (Optional<T>) _getSingleBinaryAttributeValue(valueNode);
        }
        else if (ModelItemTypeConstants.GEOCODE_ELEMENT_TYPE_ID.equals(id))
        {
            return (Optional<T>) _getSingleGeocodeAttributeValue(valueNode);
        }
        else if (ModelItemTypeConstants.RICH_TEXT_ELEMENT_TYPE_ID.equals(id))
        {
            return (Optional<T>) _getSingleRichTextAttributeValue(valueNode);
        }
        else
        {
            return _getSingleDefaultAttributeValue(valueNode, type);
        }
    }

    private Optional<Geocode> _getSingleGeocodeAttributeValue(Node geocodeNode)
    {
        Node latitudeNode = _xPathProcessor.selectSingleNode(geocodeNode, "latitude");
        String latitude = latitudeNode.getTextContent();
        
        Node longitudeNode = _xPathProcessor.selectSingleNode(geocodeNode, "longitude");
        String longitude = longitudeNode.getTextContent();
        
        if (StringUtils.isNotEmpty(latitude) && StringUtils.isNotEmpty(longitude))
        {
            return Optional.of(new Geocode(Double.valueOf(latitude), Double.valueOf(longitude)));
        }
        else
        {
            throw new IllegalArgumentException("Invalid geocode values: latitude='" + latitude + "', longitude='" + longitude + "'.");
        }
    }
    
    private Optional<Binary> _getSingleBinaryAttributeValue(Node binaryNode)
    {
        String value = binaryNode.getTextContent();
        if (StringUtils.isNotEmpty(value))
        {
            try
            {
                Pattern pattern = Pattern.compile("filename=\"([^\"]+)\"");
                
                URL url = new URL(value);
                HttpURLConnection connection = (HttpURLConnection) url.openConnection();
                connection.setConnectTimeout(1000);
                connection.setReadTimeout(2000);
                
                String contentType = Objects.toString(connection.getContentType(), "application/unknown");
                String contentEncoding = Objects.toString(connection.getContentEncoding(), "");
                String contentDisposition = Objects.toString(connection.getHeaderField("Content-Disposition"), "");
                String filename = URIUtils.decode(FilenameUtils.getName(connection.getURL().getPath()));
                if (StringUtils.isEmpty(filename))
                {
                    Matcher matcher = pattern.matcher(contentDisposition);
                    if (matcher.matches())
                    {
                        filename = matcher.group(1);
                    }
                    else
                    {
                        filename = "unknown";
                    }
                }
                
                try (InputStream is = connection.getInputStream())
                {
                    ByteArrayOutputStream bos = new ByteArrayOutputStream();
                    IOUtils.copy(is, bos);
                    
                    Binary binary = new Binary();
                    binary.setLastModificationDate(ZonedDateTime.now());
                    binary.setInputStream(new ByteArrayInputStream(bos.toByteArray()));
                    
                    if (StringUtils.isNotEmpty(filename))
                    {
                        binary.setFilename(filename);
                    }
                    if (StringUtils.isNotEmpty(contentType))
                    {
                        binary.setMimeType(contentType);
                    }
                    if (StringUtils.isNotEmpty(contentEncoding))
                    {
                        binary.setEncoding(contentEncoding);
                    }
                    
                    return Optional.of(binary);
                }
            }
            catch (Exception e)
            {
                throw new IllegalArgumentException("Unable to fetch file from URL '" + value + "', it will be ignored.", e);
            }
        }
        else
        {
            return Optional.empty();
        }
    }
    
    private Optional<RichText> _getSingleRichTextAttributeValue(Node richTextNode) throws IOException
    {
        NodeList docbookNodes = richTextNode.getChildNodes();
        for (int i = 0; i < docbookNodes.getLength(); i++)
        {
            Node docbookNode = docbookNodes.item(i);
            if (docbookNode.getNodeType() == Node.ELEMENT_NODE && "article".equals(docbookNode.getLocalName()))
            {
                try
                {
                    String docbook = _serializeNode(docbookNode);

                    RichText richText = new RichText();
                    richText.setEncoding("UTF-8");
                    richText.setLastModificationDate(ZonedDateTime.now());
                    richText.setMimeType("text/xml");
                    richText.setInputStream(new ByteArrayInputStream(docbook.getBytes("UTF-8")));
                    return Optional.of(richText);
                }
                catch (TransformerException e)
                {
                    throw new IOException("Error serializing a docbook node.", e);
                }
            }
        }
        
        // No article found, return an empty Optional
        return Optional.empty();
    }
    
    private String _serializeNode(Node node) throws TransformerException
    {
        Transformer transformer = TransformerFactory.newInstance().newTransformer();
        
        Properties format = new Properties();
        format.put(OutputKeys.METHOD, "xml");
        format.put(OutputKeys.ENCODING, "UTF-8");
        
        transformer.setOutputProperties(format);
        
        StringWriter writer = new StringWriter();
        DOMSource domSource = new DOMSource(node);
        StreamResult result = new StreamResult(writer);
        
        transformer.transform(domSource, result);
        
        return writer.toString();
    }
    
    private <T> Optional<T> _getSingleDefaultAttributeValue(Node valueNode, ElementType<T> type)
    {
        String valueAsString = valueNode.getTextContent();
        return Optional.of(type.castValue(valueAsString));
    }
}
