/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.webdeploystarter;

import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.UserIdentity;
import org.ametys.runtime.model.ElementDefinition;
import org.ametys.runtime.model.type.ElementType;
import org.ametys.runtime.parameter.ValidationResult;
import org.ametys.runtime.parameter.Validator;
import org.ametys.runtime.plugin.PluginsManager;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;
import org.ametys.web.site.SiteConfigurationManager;

/**
 * This class read a WEB-INF/param/start-actions.xml file to act
 */
public class Init extends org.ametys.plugins.deploystarter.Init
{
    private SiteConfigurationManager _siteConfigurationManager;
    private SiteManager _siteManager;
    private ObservationManager _observationManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        if (!PluginsManager.getInstance().isSafeMode())
        {
            _siteConfigurationManager = (SiteConfigurationManager) manager.lookup(SiteConfigurationManager.ROLE);
            _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
            _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        }
    }
    
    @Override
    protected void _act(Configuration configuration) throws Exception
    {
        super._act(configuration);
        if (!PluginsManager.getInstance().isSafeMode())
        {
            _actSites(configuration.getChild("sites"));
        }
    }

    private void _actSites(Configuration sitesConfiguration) throws Exception
    {
        getLogger().debug("'start actions' acting on sites...");

        UserIdentity adminUser = new UserIdentity("admin", "admin_population");
        
        for (Configuration siteConfiguration : sitesConfiguration.getChildren("site"))
        {
            // All this code is mostly duplicated from SiteDAO :( because #configure cannot be used simply CMS-8849

            String siteName = siteConfiguration.getAttribute("name");
            Site site = _siteManager.getSite(siteName);
            
            if (site == null)
            {
                throw new ConfigurationException("The site '" + siteName + "' does not exists and thus cannot be edited", siteConfiguration);
            }
            
            // Site updating event
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_SITE, site);
            _observationManager.notify(new Event(ObservationConstants.EVENT_SITE_UPDATING, adminUser, eventParams));

            for (Configuration paramConfiguration : siteConfiguration.getChild("params").getChildren())
            {
                String paramName = paramConfiguration.getName();
                ElementDefinition siteParameter = _siteConfigurationManager.getSiteParameter(site, paramName);
                ElementType type = siteParameter.getType();
                
                String untypedValue = paramConfiguration.getValue("");
                Object typedValue = type.castValue(untypedValue);
                
                Validator validator = siteParameter.getValidator();
                if (validator != null)
                {
                    ValidationResult validationResult = validator.validate(typedValue);
                    if (validationResult.hasErrors())
                    {
                        throw new ConfigurationException("Erreur in parameter: " + validationResult.getErrors(), paramConfiguration);
                    }
                }
                if (type.getManagedClass() == String.class)
                {
                    site.setValue(paramName,  typedValue == null ? "" : (String) typedValue);
                }
                else if (type.getManagedClass() == Boolean.class)
                {
                    site.setValue(paramName,  typedValue == null ? false : (Boolean) typedValue);
                }
                else if (typedValue == null)
                {
                    site.removeValue(paramName);
                }
                else
                {
                    site.setValue(paramName, typedValue);
                }
            }
            
            getLogger().info("'start actions' updated site {}", siteName);
            
            site.getNode().getSession().save();
            
            // Reload this site's configuration.
            _siteConfigurationManager.reloadSiteConfiguration(site);
            
            _observationManager.notify(new Event(ObservationConstants.EVENT_SITE_UPDATED, adminUser, eventParams));
        }
    }
}
