/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
 import { callMethod } from '../../../helper/ServerCommHelper';
 import i18n from 'i18n';
 import AmetysFront from 'AmetysFront';
 const resultsPerPage = 15;
 export const actions = {

    /**
     * Get the rights of the current user
     * @return  {Object} Rights map of the current user
     */
     async getUserRights({ commit }) {
       AmetysFront.Event.fire('loader', true);
       await callMethod({
         role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
         methodName: 'getUserRights',
       })
         .then( data => {
         AmetysFront.Event.fire('loader', false);
           commit('SET_USER_RIGHTS', data);
         })
         .catch(() => {
           AmetysFront.Event.fire('loaderFail', {
             title: i18n.PLUGINS_WORKSPACES_FORUM_RIGHTS_ERROR_MSG,
             text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
           });
         });
     },

   /**
    * Get threads list
    * @param   {String}  q  filter by search query
    * @param   {String}  category  filter by categroy id
    * @param   {Number}  tag  filter by tag id
    * @param   {Number}  closed  filter by closeInfo field
    * @param   {Boolean}  accepted  filter by whether threads have accepted answer
    * @param   {Boolean}  voted  filter by whether threads have any votes
    * @param   {Boolean}  hasNotification  filter by whether threads have notification
    * @return  {Object} should return object which contains {data,pagination}
    */
   async getThreadIds({ commit }, { q, category, tag, closed, accepted, hasNotification }) {
     AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS });
     await callMethod({
       role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
       methodName: 'getThreadIds',
       parameters: [q || null, category || null, tag || null, closed, accepted, hasNotification],
     })
       .then( threadIds => {
         AmetysFront.Event.fire('loaderEnd');
         commit('SET_THREAD_IDS', threadIds);
       })
       .catch((error) => {
         AmetysFront.Event.fire('loaderFail', {
           title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
           text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
           details: error
         });
       });
   },

    /**
     * Get threads list
     * @param   {String}  q  filter by search query
     * @param   {String}  category  filter by categroy id
     * @param   {Number}  tag  filter by tag id
     * @param   {Number}  closed  filter by closeInfo field
     * @param   {Boolean}  accepted  filter by whether threads have accepted answer
     * @param   {Boolean}  voted  filter by whether threads have any votes
     * @return  {Object} should return object which contains {data,pagination}
     */
    async getThreadsByIds({ commit, state }, page) {
      AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS });

      var threadIds = state.threadIds.slice((page - 1) * resultsPerPage, resultsPerPage * page);
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
        methodName: 'getThreadsByIds',
        parameters: [threadIds],
      })
        .then( threads => {
          AmetysFront.Event.fire('loaderEnd');

          var data = {"threads" : threads};
          data.pagination = {
            "perPage": resultsPerPage,
            "page": page,
            "lastPage": state.threadIds.length == 0 ? 1 : Math.ceil(state.threadIds.length / resultsPerPage),
            "total": state.threadIds.length,
          };


          data.concatenate = page != 1
          commit('SET_THREADS', data);
        })
        .catch((error) => {
          AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
            details: error
          });
        });
    },

    /**
     * Get last thread of the page
     * @param   {String}  q  filter by search query
     * @param   {String}  category  filter by categroy id
     * @param   {Number}  tag  filter by tag id
     * @param   {Number}  closed  filter by closeInfo field
     * @param   {Boolean} accepted  filter by whether threads have accepted answer
     * @param   {Boolean} voted  filter by whether threads have any votes
     * @param   {Number}  page  a requested page for infinite scrolling
     * @return  {Object} should return object which contains {data,pagination}
     */
    async getLastThread({ commit, state }, page) {
      AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS });

      var threadIds = state.threadIds.slice(resultsPerPage * page - 1, resultsPerPage * page);
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
        methodName: 'getThreadsByIds',
        parameters: [threadIds],
      })
        .then( threads => {
          AmetysFront.Event.fire('loaderEnd');
          if (threads[0] != null)
          {
            var data = {"threads" : threads};
            data.pagination = {
              "perPage": resultsPerPage,
              "page": page,
              "lastPage": state.threadIds.length == 0 ? 1 : Math.ceil(state.threadIds.length / resultsPerPage),
              "total": state.threadIds.length,
            };
            data.concatenate = true;
            commit('SET_THREADS', data);
          }
        })
        .catch((error) => {
          AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
            details: error
          });
        });
    },

   /**
    * Get detailed threads to view
    * @param   {Number}  id  id of the requested thread
    * @return  {Object} should return detailed thread object
    */
    async getThread({ commit }, { id }) {
     AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREAD });
     await callMethod({
       role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
       methodName: 'getThread',
       parameters: [id],
     })
       .then( data => {
         AmetysFront.Event.fire('loaderEnd');
         if (data)
         {
           commit('SET_THREAD', data);
           commit('SET_INITIAL_THREAD_CATEGORY', data.category);
         }
       })
       .catch((error) => {
         AmetysFront.Event.fire('loaderFail', {
           title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREAD_ERROR_TITLE,
           text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
           details: error
         });
       });
    },

    /**
     * [Create the detailed thread]
     *
     * @param   {Array}  tags  tag object ,will contain {id,text,color}, new tags won't have any id
     * @param   {String}  title  title
     * @param   {Number}  status  status id
     * @param   {String}  content  content
     * @param   {Array}  files  array of new uploded files,shouldn't delete old files,for deleting old files we use other method `deleteThreadFile()`
     */

    async createThread({ commit }, params) {
      AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_CREATING_THREAD });
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
        methodName: 'addThread',
        parameters: [params.thread, params.query.q, params.query.category, params.query.tag, params.query.closed, params.newFiles, params.newFileNames, params.query.accepted, params.query.hasNotification],
      })
        .then(data => {
          commit('tags/ADD_TAGS', data.newTags, { root: true });
          commit('ADD_THREAD', data.thread);
          commit('ADD_CATEGORY_THREAD_COUNT', data.thread.category);
          AmetysFront.Event.fire('loaderEnd', { text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_CREATED });
        })
        .catch((error) => {
          AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FORUM_CREATING_THREAD_ERROR_TITLE,
            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
            details: error
          });
        });
    },

    /**
     * [Reset the thread]
     *
     * @param   {[Object]}  category  [category object]
     */

    async resetThread({ commit }, params) {
      commit('RESET_THREAD', params);
    },

    /**
     * [Update the detailed thread]
     *
     * @param   {[Int]}  id  [id of the thread ]
     * @param   {[Array]}  tags  [tag object ,will contain {id,text,color}, new tags won't have any id]
     * @param   {[String]}  title  [title]
     * @param   {[Int]}  status  [status id]
     * @param   {[Object]}  category  [category object]
     * @param   {[String]}  content  [content]
     * @param   {[Array]}  files  [array of new uploded files,shouldn't delete old files,for deleting old files we use other method `deleteThreadFile()`]

     *
     * @return  {[Object]} [should return detailed thread object like  the `getThread()` method response]
     */
    /* params is FormData */
    async updateThread({ commit }, params) {
      return new Promise(function(resolve, reject)
      {

        AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_EDITING_THREAD });
        callMethod({
          role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
          methodName: 'editThread',
          parameters: [params.thread, params.query.q, params.query.category, params.query.tag, params.query.closed, params.newFiles, params.newFileNames, params.deleteFiles, params.query.accepted, params.query.hasNotification],
        })
          .then(data => {
            AmetysFront.Event.fire('loaderEnd', { text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_EDITED });
            commit('UPDATE_THREAD', data.thread);
            commit('tags/ADD_TAGS', data.newTags, { root: true });
            commit('SET_INITIAL_THREAD_CATEGORY', data.thread.category ? data.thread.category : '');
            resolve({thread: data.thread});
          })
          .catch(() => {
            AmetysFront.Event.fire('loaderFail', {
              title: i18n.PLUGINS_WORKSPACES_FORUM_EDITING_THREAD_ERROR_TITLE,
              text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
            });
            reject();
        });
      });

    },

    /**
     * [Delete detailed thread]
     *
     * @param   {[Int]}  id  [id of the thread]
     *
     * @return  {[Object]} [should return object which has id of the successfully deleted thread]
     */

    async deleteThread({ commit }, { id }) {
      AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_DELETING_THREAD });
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
        methodName: 'deleteThread',
        parameters: [id],
      })
        .then(thread => {
          AmetysFront.Event.fire('loaderEnd', { text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_DELETED });
            if (thread.id) commit('SET_THREAD', null);
            commit('REMOVE_THREAD', thread);
        })
        .catch(() => {
          AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FORUM_DELETING_THREAD_ERROR_TITLE,
            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
          });
        });
    },

   /**
    * Get the number of threads for a given category
    * @param   {String}  id  the id of the category
    * @return  {Number} should return the number of threads
    */

    async getCategoryThreadCount({ commit }, { id }) {
      AmetysFront.Event.fire('loaderStart', { text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_CATEGORY_THREAD_COUNT });
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
        methodName: 'getCategoryThreadCount',
        parameters: [id],
      })
        .then(( count ) => {
          AmetysFront.Event.fire('loaderEnd');
          commit('SET_CATEGORY_THREAD_COUNT', { id: id, count: count });
        })
        .catch(() => {
          AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_CATEGORY_THREAD_COUNT_ERROR_TITLE,
            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
          });
        });
    },

   /**
    * Get the tags used in any thread
    * @return  {[String]} list of used tags in any thread
    */
    async loadUsedTags({ commit }) {
        try
        {
            let tags = await callMethod({
              role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                methodName: "getUsedTags",
                parameters: []
            })

            tags.sort(function(c1,c2) {
                if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) == AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return 0;
                else if (AmetysFront.Utils.deemphasize(c1.text.toLowerCase()) < AmetysFront.Utils.deemphasize(c2.text.toLowerCase())) return -1;
                else return 1;
            });

            commit('SET_USED_TAGS', tags);
        }
        catch (e)
        {
            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL,
                text : i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL_TEXT,
                details: e
            });

        }
    },

    /**
     * Clear notification
     * @param   {[String]} id  [id of the thread]
     * @param   {Boolean}  filterByNotification filter by whether threads have any notification
     */
    async clearUnopenedThreadNotification({ commit }, { id, filterByNotification }) {
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadUserPreferencesDAO',
        methodName: 'clearUnopenedThreadNotification',
        parameters: [id],
      }).then( () => {
        commit('UPDATE_THREAD_WITH_NOTIFICATION', {id: id, filterByNotification: filterByNotification});
      });
    },

    /**
     * Clear unread comments notifications
     * @param   {[String]} id  [id of the thread]
     * @param   {Boolean}  filterByNotification filter by whether threads have any notification
     * @param   {[Array]}  commentIds  [list of read comments]
     */
    async clearUnreadCommentsNotification({ commit }, { thread, filterByNotification, commentIds }) {

      let lastSeenCommentId = commentIds[commentIds.length - 1];
      await callMethod({
        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadUserPreferencesDAO',
        methodName: 'clearUnreadCommentsNotification',
        parameters: [thread.id, lastSeenCommentId],
      }).then( () => {
        commit('MAKE_COMMENTS_READ', {id: thread.id, commentIds: commentIds});
        let lastCommentId = thread.comments[thread.comments.length - 1].id;
        if (lastCommentId == lastSeenCommentId)
        {
          commit('UPDATE_THREAD_WITH_NOTIFICATION', {id: thread.id, filterByNotification: filterByNotification});
        }
      });
    },
};
