/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { mapActions, mapGetters } from 'vuex';

export default {
  data() {
    return {
      threadLoading: true,
      threadCreate: false,
      threadModal: false,
      editMode: false,
    };
  },
  computed: {
    ...mapGetters({
      thread: 'thread/THREAD',
    }),
  },
  watch: {
    async threadModal(newValue) {
      if(!newValue){
        this.threadCreate = false;
        this.editMode = false;
      }else if(this.threadCreate){
         this.editMode = true;
      }
    }
  },
  methods: {
    ...mapActions({
      clearUnopenedThreadNotification: 'thread/clearUnopenedThreadNotification',
      getThread: 'thread/getThread',
      updateThread: 'thread/updateThread',
      createThread: 'thread/createThread',
      deleteThread: 'thread/deleteThread',
      getLastThread: 'thread/getLastThread',
      resetThread: 'thread/resetThread',
      clearUnreadCommentsNotification: 'thread/clearUnreadCommentsNotification',
    }),

    /*
     * Function that is called when a thread is opened, to load all details
     * The Vue component that call this function is the ForumModule.vue
     */
    async handleThread(threadId) {
      this.threadLoading = true;
      await this.getThread({ id: threadId });
      this.threadLoading = false;
      if (this.thread.hasUnopenedThreadNotification) {
       setTimeout(() => {
        this.clearUnopenedThreadNotification({ id: this.thread.id, filterByNotification: this.getQuery().hasNotification });
        }, 2000);
      }
    },

    /*
     * Function that is called when a thread is created
     */
    async handleCreateThread(thread, newFiles, newFileNames) {
      await this.createThread({
      "thread": thread,
      "query": this.getQuery(),
      "newFiles": newFiles,
      "newFileNames": newFileNames});
      await this.getUsedTags();
    },

    /*
     * Function that is called when a thread is updated
     */
    async handleUpdateThread(thread, newFiles, newFileNames, deleteFiles) {
      let data = await this.updateThread({
      "thread": thread,
      "query": this.getQuery(),
      "newFiles": newFiles,
      "newFileNames": newFileNames,
      "deleteFiles": deleteFiles});
      await this.getUsedTags();
      return data.thread;
    },

    /*
     * Function that is called when a thread is deleted
     */
    async handleDeleteThread(params) {
      await this.deleteThread(params);
      await this.getUsedTags();
      const query = this.getQuery();
      // Only replace deleted thread by another thread if there are still threads to load
      if (this.threadsPagination && this.threadsPagination.page != this.threadsPagination.lastPage)
      {
        await this.getLastThread(query.page || 1);
      }
    },
    handleChangeEditMode(val){
      this.editMode = val;
    },

    async handleClearUnreadCommentsNotification(params) {
      let filterByNotification = this.getQuery().hasNotification
      await this.clearUnreadCommentsNotification({thread: params.thread, filterByNotification: filterByNotification, commentIds: params.commentIds});
    },

    /*
     * Function that is called when a thread is created
     */
    async handleResetThread(category) {
      await this.resetThread({"category": category});
    },
  },
};
