/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces;

import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Optional;
import java.util.Set;

import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.TransformerFactoryConfigurationError;
import javax.xml.transform.dom.DOMResult;
import javax.xml.transform.sax.SAXTransformerFactory;
import javax.xml.transform.sax.TransformerHandler;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.configuration.Configuration;
import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.configuration.DefaultConfigurationBuilder;
import org.apache.avalon.framework.configuration.DefaultConfigurationSerializer;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceNotFoundException;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.sax.SAXParser;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.ContentDAO.TagMode;
import org.ametys.cms.repository.ModifiableWorkflowAwareContent;
import org.ametys.cms.transformation.Configuration2XMLValuesTransformer;
import org.ametys.cms.workflow.AbstractContentWorkflowComponent;
import org.ametys.cms.workflow.ContentWorkflowHelper;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.ProfileAssignmentStorageExtensionPoint;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.extractor.ModelAwareValuesExtractor;
import org.ametys.plugins.repository.data.extractor.xml.ModelAwareXMLValuesExtractor;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.NameHelper;
import org.ametys.plugins.workflow.component.CheckRightsCondition;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.model.Model;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.PageDAO;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.service.Service;
import org.ametys.web.service.ServiceExtensionPoint;
import org.ametys.web.skin.Skin;
import org.ametys.web.skin.SkinTemplate;
import org.ametys.web.skin.SkinTemplateZone;
import org.ametys.web.skin.SkinsManager;

import com.opensymphony.workflow.InvalidActionException;
import com.opensymphony.workflow.WorkflowException;

/**
 * Component allowing to create and fill a page from a configuration file
 */
public class PagePopulator extends AbstractLogEnabled implements Serviceable, Component
{
    /** The avalon role */
    public static final String ROLE = PagePopulator.class.getName();
    
    /** the source resolver */
    protected SourceResolver _sourceResolver;
    /** the i18n utils component */
    protected I18nUtils _i18nUtils;
    /** the service extension point */
    protected ServiceExtensionPoint _serviceEP;
    /** the observation manager */
    protected ObservationManager _observationManager;
    /** the workflow helper */
    protected ContentWorkflowHelper _workflowHelper;
    /** the current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** the page dao */
    protected PageDAO _pageDAO;
    /** the profile assignment storage extension point */
    protected ProfileAssignmentStorageExtensionPoint _profileAssignementStorageEP;
    /** the skins manager */
    protected SkinsManager _skinsManager;
    /** the content type extension point */
    protected ContentTypeExtensionPoint _contentTypeEP;
    /** Excalibur SaxParser */
    protected SAXParser _saxParser;

    public void service(ServiceManager manager) throws ServiceException
    {
        _contentTypeEP = (ContentTypeExtensionPoint) manager.lookup(ContentTypeExtensionPoint.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _pageDAO = (PageDAO) manager.lookup(PageDAO.ROLE);
        _profileAssignementStorageEP = (ProfileAssignmentStorageExtensionPoint) manager.lookup(ProfileAssignmentStorageExtensionPoint.ROLE);
        _serviceEP = (ServiceExtensionPoint) manager.lookup(ServiceExtensionPoint.ROLE);
        _skinsManager = (SkinsManager) manager.lookup(SkinsManager.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _workflowHelper = (ContentWorkflowHelper) manager.lookup(ContentWorkflowHelper.ROLE);
        _saxParser = (SAXParser) manager.lookup(SAXParser.ROLE);
    }
    
    /**
     * Create a new page based on a configuration file.
     * 
     * @param parent the parent where the page should be inserted
     * @param path the path where the configuration can be found
     * @return the newly created page or {@code Optional#empty()} if no page was created.
     * @throws IOException if an error occurred while reading the file
     * @throws SAXException if an error occurred while parsing the configuration file
     * @throws ConfigurationException if the configuration is not valid; A required info is missing.
     */
    public Optional<ModifiablePage> initPage(ModifiableSitemapElement parent, String path) throws IOException, SAXException, ConfigurationException
    {
        Source cfgFile = null;
        try
        {
            cfgFile = _sourceResolver.resolveURI(path);
            if (!cfgFile.exists())
            {
                throw new SourceNotFoundException(cfgFile.getURI() + " does not exist");
            }
            
            try (InputStream is = cfgFile.getInputStream())
            {
                Configuration configuration = new DefaultConfigurationBuilder().build(is);
                
                return initPage(parent, configuration);
            }
        }
        catch (ConfigurationException e)
        {
            throw new ConfigurationException("There is an issue with configuration file '" + cfgFile.getURI() + "'. This prevented the initialization of the page.", e);
        }
        finally
        {
            _sourceResolver.release(cfgFile);
        }
    }

    /**
     * Create and configure a new page based on a configuration.
     * 
     * @param parent the parent where the page should be added
     * @param configuration the page configuration
     * @return the newly created page or {@code Optional#empty()} if no page was created.
     * @throws ConfigurationException if the configuration is not valid; A required info is missing.
     */
    public Optional<ModifiablePage> initPage(ModifiableSitemapElement parent, Configuration configuration) throws ConfigurationException
    {
        Optional<ModifiablePage> page = createPage(parent, configuration);
        if (page.isPresent())
        {
            ModifiablePage newPage = page.get();
            configurePage(newPage, configuration);
            
            setReaderAccess(newPage, configuration);

            newPage.saveChanges();
        }
        return page;
    }

    /**
     * Create a new page based on a configuration.
     * 
     * @param parent the sitemap element where the page should be added
     * @param configuration the page configuration
     * @return the newly created page or {@code Optional#empty()} if the page already exist.
     * @throws ConfigurationException if the configuration is not valid; A required info is missing.
     */
    protected Optional<ModifiablePage> createPage(ModifiableSitemapElement parent, Configuration configuration) throws ConfigurationException
    {
        String lang = parent.getSitemapName();
        I18nizableText i18nTitle = I18nizableText.parseI18nizableText(configuration.getChild("title"), "application");
        String title = _i18nUtils.translate(i18nTitle, lang);
        // Title should not be missing, but just in case if the i18n message or the whole catalog does not exists in the requested language
        // to prevent a non-user-friendly error and still generate the project workspace.
        title = StringUtils.defaultIfBlank(title, "Missing title");
        
        String name = configuration.getAttribute("name", NameHelper.filterName(title));
        
        if (!parent.hasChild(name))
        {
            return Optional.of(_pageDAO.createPage(parent, name, title, null));
        }
        return Optional.empty();
    }

    /**
     * Use a configuration to edit a page.
     * 
     * Via configuration, it's possible to define the page tags, template and zone items (service or content)
     * @param newPage the page that needs configuration
     * @param configuration the configuration describing the expected page
     * @throws ConfigurationException if the configuration is not valid
     */
    protected void configurePage(ModifiablePage newPage, Configuration configuration) throws ConfigurationException
    {
        Configuration tagsCfg = configuration.getChild("tags", true);
        List<String> tags = Arrays.stream(tagsCfg.getChildren("tag"))
            .map(cfg -> cfg.getValue(StringUtils.EMPTY))
            .filter(StringUtils::isNotEmpty)
            .toList();
        if (!tags.isEmpty())
        {
            _pageDAO.tag(newPage, tags, TagMode.INSERT);
        }
        
        String templateName = configuration.getAttribute("template", null);
        if (templateName != null)
        {
            Skin skin = _skinsManager.getSkin(newPage.getSite().getSkinId());
            if (skin == null)
            {
                // This should never be the case but just to be sure, terminate the creation.
                getLogger().warn("The site is configured with an unexisting skin. Impossible to configure the page.");
                return;
            }
            SkinTemplate template = skin.getTemplate(templateName);
            if (template == null)
            {
                throw new ConfigurationException("Trying to configure page with an unexisting template named '" + templateName + "' for skin '" + skin.getId() + "'.");
            }
            newPage.setType(PageType.CONTAINER);
            newPage.setTemplate(templateName);
            
            Configuration templateParams = configuration.getChild("parameters", false);
            if (templateParams != null)
            {
                DataContext context = DataContext.newInstance();
                context.withLocale(Locale.forLanguageTag(newPage.getSitemapName()));
                
                try
                {
                    Element paramsElement = _interpretConfiguration(templateParams, context);
                    ModifiableModelAwareDataHolder templateParametersHolder = newPage.getTemplateParametersHolder();
                    ModelAwareValuesExtractor extractor = new ModelAwareXMLValuesExtractor(paramsElement, templateParametersHolder.getModel());
                    templateParametersHolder.synchronizeValues(extractor.extractValues());

                    Map<String, Object> eventParams = new HashMap<>();
                    eventParams.put(ObservationConstants.ARGS_SITEMAP_ELEMENT, newPage);
                    _observationManager.notify(new Event(ObservationConstants.EVENT_VIEW_PARAMETERS_MODIFIED, _currentUserProvider.getUser(), eventParams));
                }
                catch (Exception e)
                {
                    getLogger().warn("Failed to set template parameters for page '" + newPage.getName() + "'.", e);
                }
            }
            
            Map<String, SkinTemplateZone> templateZones = template.getZones();
            for (Configuration zoneCfg : configuration.getChildren("zone"))
            {
                SkinTemplateZone templateZone = templateZones.get(zoneCfg.getAttribute("id"));
                if (templateZone == null)
                {
                    throw new ConfigurationException("Trying to configure unexisting page zone '" + zoneCfg.getAttribute("id") + "' for template '" + templateName + "' in skin '" + skin.getId() + "'.");
                }
                createAndConfigureZone(newPage, zoneCfg);
            }
            
            // Notify change after the set template (this also seems to covers all the indexation and live sync from new zone item…)
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(ObservationConstants.ARGS_PAGE, newPage);
            eventParams.put(ObservationConstants.ARGS_PAGE_ID, newPage.getId());
            _observationManager.notify(new Event(ObservationConstants.EVENT_PAGE_CHANGED, _currentUserProvider.getUser(), eventParams));
        }
    }
    
    /**
     * Create and configure a zone based on configuration
     * @param newPage the new page where the zone should be added
     * @param zoneCfg the configuration to use
     * @throws ConfigurationException if the configuration is invalid
     */
    protected void createAndConfigureZone(ModifiablePage newPage, Configuration zoneCfg) throws ConfigurationException
    {
        String zoneId = zoneCfg.getAttribute("id");
        ModifiableZone zone = newPage.createZone(zoneId);
        for (Configuration itemCfg : zoneCfg.getChildren())
        {
            if (StringUtils.equals(itemCfg.getName(), "service"))
            {
                createAndConfigureServiceItem(zone, itemCfg);
            }
            else if (StringUtils.equals(itemCfg.getName(), "content"))
            {
                createAndConfigureContentItem(zone, itemCfg);
            }
        }
    }
    
    /**
     * Create and configure a zone item based on a service configuration
     * @param zone the zone where the zone item should be added
     * @param serviceCfg the configuration to use
     * @throws ConfigurationException if the configuration is invalid
     */
    protected void createAndConfigureServiceItem(ModifiableZone zone, Configuration serviceCfg) throws ConfigurationException
    {
        String serviceId = serviceCfg.getAttribute("id");
        Service service = _serviceEP.getExtension(serviceId);
        if (service != null)
        {
            ModifiableZoneItem item = zone.addZoneItem();
            item.setType(ZoneType.SERVICE);
            item.setServiceId(serviceId);
            
            DataContext dataContext = DataContext.newInstance();
            dataContext.withLocale(Locale.forLanguageTag(zone.getSitemapElement().getSitemapName()));
            
            try
            {
                // Configuration may requires interpretation before extraction of the value
                // for exemple to translate i18n keys
                Element xmlValues = _interpretConfiguration(serviceCfg, dataContext);
                
                // provide the result to XML values extractor
                ModelAwareValuesExtractor extractor = new ModelAwareXMLValuesExtractor(xmlValues, service);
                item.getServiceParameters().synchronizeValues(extractor.extractValues());
            }
            catch (Exception e)
            {
                throw new ConfigurationException("Failed to extract the value from configuration for item with service id '" + serviceId + "'.", e);
            }
        }
        else
        {
            throw new ConfigurationException("Trying to create unexisting service '" + serviceId + "' for page '" + zone.getSitemapElement().getName() + "'.");
        }
    }
    
    /**
     * Create a new content based on configuration and add it to a new zone item
     * @param zone the zone where the zone item should be added
     * @param contentCfg the configuration to use
     * @throws ConfigurationException if the configuration is invalid
     */
    protected void createAndConfigureContentItem(ModifiableZone zone, Configuration contentCfg) throws ConfigurationException
    {
        Map<String, Object> params = new HashMap<>();
        params.put(org.ametys.web.workflow.CreateContentFunction.SITE_KEY, zone.getSitemapElement().getSiteName());
        ModifiableWorkflowAwareContent content;
        
        try
        {
            Configuration cTypesCfg = contentCfg.getChild("contentTypes");
            Configuration[] cTypeCfgs = cTypesCfg.getChildren("contentType");
            
            String[] cTypeIds = new String[cTypeCfgs.length];
            Model[] cTypes = new Model[cTypeCfgs.length];
            
            // initialize default workflow name
            String workflowName = "content";
            
            int i = 0;
            for (Configuration cfg : cTypeCfgs)
            {
                String cTypeId = cfg.getAttribute("id");
                ContentType contentType = _contentTypeEP.getExtension(cTypeId);
                if (contentType == null)
                {
                    throw new ConfigurationException("Could not create new content for page '" + zone.getSitemapElement().getName() + "'. The configuration file references an unexisting content type '" + cTypeId + "'.");
                }
                cTypes[i] = contentType;
                cTypeIds[i++] = cTypeId;
                
                // try to find a default workflow name based on content type
                Optional<String> defaultWorkflow = contentType.getDefaultWorkflowName();
                if (defaultWorkflow.isPresent())
                {
                    workflowName = defaultWorkflow.get();
                }
            }
            
            // use required workflow name or computed workflow name based on content type
            Configuration workflow = contentCfg.getChild("workflow");
            workflowName = workflow.getAttribute("name", "content");
            
            DataContext dataContext = DataContext.newInstance();
            dataContext.withLocale(Locale.forLanguageTag(zone.getSitemapElement().getSitemapName()));
            
            // Configuration may requires interpretation before extraction of the value
            // for example to translate i18n keys
            Element xmlValues = _interpretConfiguration(contentCfg, dataContext);
            
            // provide the result to XML values extractor
            ModelAwareValuesExtractor extractor = new ModelAwareXMLValuesExtractor(xmlValues, Arrays.asList(cTypes));
            Map<String, Object> contentValues = extractor.extractValues();
            
            String title = (String) contentValues.get("title");
            if (title == null)
            {
                throw new ConfigurationException("Failed to retrieve a translation for the provided configuration.", contentCfg.getChild("title"));
            }
            String name = NameHelper.filterName(contentCfg.getAttribute("name", title));
            int createAction = workflow.getAttributeAsInteger("init-action-id", 1);
            content = (ModifiableWorkflowAwareContent) _workflowHelper.createContent(workflowName, createAction, name, title, cTypeIds, null, zone.getSitemapElement().getSitemapName(), params).get(AbstractContentWorkflowComponent.CONTENT_KEY);
            
            content.synchronizeValues(contentValues);
            
            Configuration tagsCfg = contentCfg.getChild("tags");
            for (Configuration tag : tagsCfg.getChildren("tag"))
            {
                content.tag(tag.getValue());
            }
            
            content.saveChanges();
            
            int validateAction = workflow.getAttributeAsInteger("validate-action-id", -1);
            if (validateAction > 0)
            {
                try
                {
                    // Current user most probably don't have any right on the context so we bypass
                    // the check right
                    Map<String, Object> inputs = new HashMap<>();
                    inputs.put(CheckRightsCondition.FORCE, true);
                    _workflowHelper.doAction(content, validateAction, inputs);
                }
                catch (WorkflowException | InvalidActionException e)
                {
                    getLogger().warn("Failed to validate new content '" + content.getId() + "'.");
                }
            }
            ModifiableZoneItem item = zone.addZoneItem();
            item.setType(ZoneType.CONTENT);
            item.setContent(content);
        }
        catch (AmetysRepositoryException | WorkflowException e)
        {
            getLogger().warn("Could not create new content for page '" + zone.getSitemapElement().getName() + "'.");
        }
        catch (Exception e)
        {
            getLogger().warn("Failed to extract content value for page '" + zone.getSitemapElement().getName() + "'.", e);
        }
    }

    private Element _interpretConfiguration(Configuration contentCfg, DataContext dataContext)
            throws SAXException, ConfigurationException
    {
        DOMResult domResult = new DOMResult();
        
        try
        {
            TransformerHandler th = ((SAXTransformerFactory) TransformerFactory.newInstance()).newTransformerHandler();
            th.setResult(domResult);
            
            Configuration2XMLValuesTransformer handler = new Configuration2XMLValuesTransformer(th, dataContext, _i18nUtils);
            new DefaultConfigurationSerializer().serialize(handler, contentCfg);
            Element values = ((Document) domResult.getNode()).getDocumentElement();
            return values;
        }
        catch (TransformerConfigurationException | TransformerFactoryConfigurationError e)
        {
            throw new IllegalStateException("Failed to retrive transformer handler. Impossible to interpret the configuration", e);
        }
    }
    
    /**
     * Set page reader access based on configuration
     * @param newPage the newly created page
     * @param configuration the page configuration
     * @throws ConfigurationException if an unrecognized group is present in configuration
     */
    protected void setReaderAccess(ModifiablePage newPage, Configuration configuration) throws ConfigurationException
    {
        Configuration accessCfg = configuration.getChild("reader-access", true);
        for (Configuration cfg : accessCfg.getChildren())
        {
            String name = cfg.getName();
            switch (name)
            {
                case "anonymous":
                    _setAnonymousPermission(newPage, cfg);
                    break;
                case "any-connected":
                    _setAnyConnectedPermission(newPage, cfg);
                    break;
                default :
                    throw new ConfigurationException("Unknown identity found in configuration. Could not define reader permission", cfg);
            }
        }
    }
    
    private void _setAnyConnectedPermission(Page page, Configuration cfg)
    {
        boolean deny = cfg.getAttributeAsBoolean("deny", false);
        if (deny)
        {
            _profileAssignementStorageEP.denyProfileToAnyConnectedUser(RightManager.READER_PROFILE_ID, page);
            _notifyACLChange(page, Set.of(RightManager.READER_PROFILE_ID));
        }
        else
        {
            _profileAssignementStorageEP.allowProfileToAnyConnectedUser(RightManager.READER_PROFILE_ID, page);
            _notifyACLChange(page, Set.of(RightManager.READER_PROFILE_ID));
        }
    }

    private void _setAnonymousPermission(Page page, Configuration cfg)
    {
        boolean deny = cfg.getAttributeAsBoolean("deny", false);
        if (deny)
        {
            _profileAssignementStorageEP.denyProfileToAnonymous(RightManager.READER_PROFILE_ID, page);
            _notifyACLChange(page, Set.of(RightManager.READER_PROFILE_ID));
        }
        else
        {
            _profileAssignementStorageEP.allowProfileToAnonymous(RightManager.READER_PROFILE_ID, page);
            _notifyACLChange(page, Set.of(RightManager.READER_PROFILE_ID));
        }
    }
    
    /**
     * Utility method to notify a change of ACL on a context
     * @param context the impacted context
     * @param profilesId the assigned or removed profiles
     */
    protected void _notifyACLChange(Object context, Set<String> profilesId)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, context);
        eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, profilesId);
        
        _observationManager.notify(new Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
}
