/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.documents;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.w3c.dom.Element;
import org.xml.sax.ContentHandler;
import org.xml.sax.SAXException;

import org.ametys.core.model.type.AbstractElementType;
import org.ametys.plugins.explorer.resources.jcr.JCRResource;
import org.ametys.plugins.workspaces.activities.documents.AbstractResourceReferenceElementType.ResourceReference;
import org.ametys.runtime.model.exception.BadItemTypeException;
import org.ametys.runtime.model.type.DataContext;

/**
 * Abstract class for elements of type resourceReference
 */
public abstract class AbstractResourceReferenceElementType extends AbstractElementType<ResourceReference>
{
    public boolean isSimple()
    {
        return false;
    }
    
    @Override
    public String toString(ResourceReference value)
    {
        if (value != null)
        {
            return _jsonUtils.convertObjectToJson(_singleValueToJSON(value, DataContext.newInstance()));
        }
        else
        {
            return null;
        }
    }
    
    @Override
    protected ResourceReference convertValue(Object value) throws BadItemTypeException
    {
        if (value instanceof String json)
        {
            return _json2ResourceReference(_jsonUtils.convertJsonToMap(json));
        }
        return super.convertValue(value);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected ResourceReference _singleValueFromJSON(Object json, DataContext context) throws BadItemTypeException
    {
        if (json instanceof Map jsonMap)
        {
            return _json2ResourceReference(jsonMap);
        }
        else
        {
            throw new BadItemTypeException("Unable to convert the given json value '" + json + "' into a " + getManagedClass().getName());
        }
    }
    
    private ResourceReference _json2ResourceReference(Map<String, Object> jsonMap)
    {
        String id = (String) jsonMap.get(ResourceReference.ID);
        String name = (String) jsonMap.get(ResourceReference.NAME);
        String oldName = (String) jsonMap.get(ResourceReference.OLD_NAME);
        String mimeType = (String) jsonMap.get(ResourceReference.MIME_TYPE);
        String version = (String) jsonMap.get(ResourceReference.VERSION);
        return new ResourceReference(id, name, oldName, mimeType, version);
    }
    
    @Override
    protected Object _singleTypedValueToJSON(ResourceReference value, DataContext context)
    {
        Map<String, Object> json = new HashMap<>();
        json.put(ResourceReference.ID, value.id());
        json.put(ResourceReference.NAME, value.name());
        String oldName = value.oldName();
        if (oldName != null)
        {
            json.put(ResourceReference.OLD_NAME, oldName);
        }
        json.put(ResourceReference.MIME_TYPE, value.mimeType());
        String baseVersionName = value.baseVersionName();
        if (baseVersionName != null)
        {
            json.put(ResourceReference.VERSION, baseVersionName);
        }
        return json;
    }
    
    @Override
    protected boolean _useJSONForEdition()
    {
        return true;
    }
    
    @Override
    protected ResourceReference _singleValueFromXML(Element element, Optional<Object> additionalData)
    {
        String id = element.getAttribute(ResourceReference.ID);
        String name = element.getAttribute(ResourceReference.NAME);
        String oldName = null;
        if (element.hasAttribute(ResourceReference.OLD_NAME))
        {
            oldName = element.getAttribute(ResourceReference.OLD_NAME);
        }
        String mimeType = element.getAttribute(ResourceReference.MIME_TYPE);
        String version = null;
        if (element.hasAttribute(ResourceReference.VERSION))
        {
            version = element.getAttribute(ResourceReference.VERSION);
        }
        return new ResourceReference(id, name, oldName, mimeType, version);
    }
    
    @Override
    protected void _singleTypedNotEnumeratedValueToSAX(ContentHandler contentHandler, String tagName, ResourceReference value, DataContext context, AttributesImpl attributes) throws SAXException
    {
        AttributesImpl localAttributes = new AttributesImpl(attributes);
        localAttributes.addCDATAAttribute(ResourceReference.ID, value.id());
        localAttributes.addCDATAAttribute(ResourceReference.NAME, value.name());
        String oldName = value.oldName();
        if (oldName != null)
        {
            localAttributes.addCDATAAttribute(ResourceReference.OLD_NAME, oldName);
        }
        localAttributes.addCDATAAttribute(ResourceReference.MIME_TYPE, value.mimeType());
        String baseVersionName = value.baseVersionName();
        if (baseVersionName != null)
        {
            localAttributes.addCDATAAttribute(ResourceReference.VERSION, baseVersionName);
        }
        XMLUtils.createElement(contentHandler, tagName, localAttributes, value.name());
    }
    
    /**
     * Represent a reference to a {@link JCRResource}
     * 
     * @param id the resource id
     * @param name the resource name
     * @param oldName the previous resource name if the resource was renamed, null otherwise.
     * @param mimeType the resource mimeType
     * @param baseVersionName the resource version, can be null.
     */
    public record ResourceReference(String id, String name, String oldName, String mimeType, String baseVersionName)
    {
        /** the resource id */
        public static final String ID = "id";
        /** the resource name */
        public static final String NAME = "name";
        /** the resource old name */
        public static final String OLD_NAME = "oldName";
        /** the resource mimeType */
        public static final String MIME_TYPE = "mimeType";
        /** the resource version */
        public static final String VERSION = "version";
    }
}
