/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.UUID;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.IllegalClassException;
import org.apache.commons.lang3.StringUtils;
import org.apache.jackrabbit.util.Text;

import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ModifiableExplorerNode;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectIterator;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.calendars.Calendar.CalendarVisibility;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventJSONHelper;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendar;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarFactory;
import org.ametys.plugins.workspaces.project.ProjectConstants;

/**
 * Calendar DAO
 */
public class CalendarDAO extends AbstractCalendarDAO
{
    /** Avalon Role */
    public static final String ROLE = CalendarDAO.class.getName();

    /** The tasks list JSON helper */
    protected CalendarEventJSONHelper _calendarEventJSONHelper;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) manager.lookup(CalendarEventJSONHelper.ROLE);
    }
        
    /**
     * Get calendar info
     * @param calendar The calendar
     * @param recursive True to get data for sub calendars
     * @param includeEvents True to also include child events
     * @param useICSFormat true to use ICS Format for dates
     * @return the calendar data in a map
     */
    public Map<String, Object> getCalendarData(Calendar calendar, boolean recursive, boolean includeEvents, boolean useICSFormat)
    {
        Map<String, Object> result = new HashMap<>();
        
        result.put("id", calendar.getId());
        result.put("title", Text.unescapeIllegalJcrChars(calendar.getName()));
        result.put("description", calendar.getDescription());
        result.put("templateDesc", calendar.getTemplateDescription());
        result.put("color", calendar.getColor());
        result.put("visibility", calendar.getVisibility().name().toLowerCase());
        result.put("public", calendar.getVisibility() == CalendarVisibility.PUBLIC);
        result.put("workflowName", calendar.getWorkflowName());
        
        if (recursive)
        {
            List<Map<String, Object>> calendarList = new LinkedList<>();
            result.put("calendars", calendarList);
            AmetysObjectIterable<AmetysObject> children = calendar.getChildren();
            for (AmetysObject child : children)
            {
                if (child instanceof Calendar)
                {
                    calendarList.add(getCalendarData((Calendar) child, recursive, includeEvents, useICSFormat));
                }
            }
        }
        
        if (includeEvents)
        {
            List<Map<String, Object>> eventList = new LinkedList<>();
            result.put("events", eventList);
            
            AmetysObjectIterable<AmetysObject> children = calendar.getChildren();
            for (AmetysObject child : children)
            {
                if (child instanceof CalendarEvent)
                {
                    eventList.add(_calendarEventJSONHelper.eventAsJson((CalendarEvent) child, false, useICSFormat));
                }
            }
        }

        result.put("rights", _extractCalendarRightData(calendar));
        result.put("token", getCalendarIcsToken(calendar, true));
        
        
        return result;
    }
    
    /**
     * Get calendar info
     * @param calendar The calendar
     * @return the calendar data in a map
     */
    public Map<String, Object> getCalendarProperties(Calendar calendar)
    {
        return getCalendarData(calendar, false, false, false);
    }
    /**
     * Add a calendar
     * @param inputName The desired name for the calendar
     * @param color The calendar color
     * @param isPublic true if the calendar is public
     * @return The result map with id, parentId and name keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    @Callable
    public Map<String, Object> addCalendar(String inputName, String color, boolean isPublic) throws IllegalAccessException
    {
        String rootId = _getCalendarRoot().getId();
        return addCalendar(rootId, inputName, StringUtils.EMPTY, StringUtils.EMPTY, color, isPublic ? CalendarVisibility.PUBLIC.name() : CalendarVisibility.PRIVATE.name(), "calendar-default", false);
    }
    
    /**
     * Add a calendar
     * @param id The identifier of the parent in which the calendar will be added
     * @param inputName The desired name for the calendar
     * @param description The calendar description
     * @param templateDesc The calendar template description
     * @param color The calendar color
     * @param visibility The calendar visibility
     * @param workflowName The calendar workflow name
     * @param renameIfExists True to rename if existing
     * @return The result map with id, parentId and name keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    public Map<String, Object> addCalendar(String id, String inputName, String description, String templateDesc, String color, String visibility, String workflowName, Boolean renameIfExists) throws IllegalAccessException
    {
        return addCalendar(_resolver.resolveById(id), inputName, description, templateDesc, color, visibility, workflowName, renameIfExists, true, true);
    }
        
    /**
     * Add a calendar
     * @param parent The parent in which the calendar will be added
     * @param inputName The desired name for the calendar
     * @param description The calendar description
     * @param templateDesc The calendar template description
     * @param color The calendar color
     * @param visibility The calendar visibility
     * @param workflowName The calendar workflow name
     * @param renameIfExists True to rename if existing
     * @param checkRights true to check if the current user have enough rights to create the calendar
     * @param notify True to notify the calendar creation
     * @return The result map with id, parentId and name keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    public Map<String, Object> addCalendar(ModifiableTraversableAmetysObject parent, String inputName, String description, String templateDesc, String color, String visibility, String workflowName, Boolean renameIfExists, Boolean checkRights, boolean notify) throws IllegalAccessException
    {
        String originalName = Text.escapeIllegalJcrChars(inputName);
        
        // Check user right
        if (checkRights)
        {
            _explorerResourcesDAO.checkUserRight(parent, RIGHTS_CALENDAR_ADD);
        }
        
        if (BooleanUtils.isNotTrue(renameIfExists) && parent.hasChild(originalName))
        {
            getLogger().warn("Cannot create the calendar with name '" + originalName + "', an object with same name already exists.");
            return Map.of("message", "already-exist");
        }
        
        if (!_explorerResourcesDAO.checkLock(parent))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify the object '" + parent.getName() + "' but it is locked by another user");
            return Map.of("message", "locked");
        }
        
        int index = 2;
        String name = originalName;
        while (parent.hasChild(name))
        {
            name = originalName + " (" + index + ")";
            index++;
        }
        
        JCRCalendar calendar = parent.createChild(name, JCRCalendarFactory.CALENDAR_NODETYPE);
        calendar.setWorkflowName(workflowName);
        calendar.setDescription(description);
        calendar.setTemplateDescription(templateDesc);
        calendar.setColor(color);
        calendar.setVisibility(StringUtils.isNotEmpty(visibility) ? CalendarVisibility.valueOf(visibility.toUpperCase()) : CalendarVisibility.PRIVATE);
        parent.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, calendar.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, calendar.getPath());
        
        if (notify)
        {
            _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_CREATED, _currentUserProvider.getUser(), eventParams));
        }
        
        return getCalendarProperties(calendar);
    }

    /**
     * Edit a calendar
     * @param id The identifier of the calendar
     * @param inputName The new name
     * @param templateDesc The new calendar template description
     * @param color The calendar color
     * @param isPublic true if the calendar is public
     * @return The result map with id and name keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    @Callable
    public Map<String, Object> editCalendar(String id, String inputName, String templateDesc, String color, boolean isPublic) throws IllegalAccessException
    {
        CalendarVisibility visibility = isPublic ? CalendarVisibility.PUBLIC : CalendarVisibility.PRIVATE;
        
        assert id != null;
        String rename = Text.escapeIllegalJcrChars(inputName);
        
        AmetysObject object = _resolver.resolveById(id);
        if (!(object instanceof ModifiableCalendar))
        {
            throw new IllegalClassException(ModifiableCalendar.class, object.getClass());
        }
        
        ModifiableCalendar calendar = (ModifiableCalendar) object;
        
        // Check user right
        _explorerResourcesDAO.checkUserRight(object, RIGHTS_CALENDAR_EDIT);
        
        String name = calendar.getName();
        ModifiableTraversableAmetysObject parent = calendar.getParent();
        
        if (!StringUtils.equals(rename, name) && parent.hasChild(rename))
        {
            getLogger().warn("Cannot edit the calendar with the new name '" + inputName + "', an object with same name already exists.");
            return Map.of("message", "already-exist");
        }
        
        if (!_explorerResourcesDAO.checkLock(calendar))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to modify calendar '" + object.getName() + "' but it is locked by another user");
            return Map.of("message", "locked");
        }
        
        if (!StringUtils.equals(name, rename))
        {
            int index = 2;
            name = Text.escapeIllegalJcrChars(rename);
            while (parent.hasChild(name))
            {
                name = rename + " (" + index + ")";
                index++;
            }
            calendar.rename(name);
        }
        
        calendar.setTemplateDescription(templateDesc);
        calendar.setColor(color);
        calendar.setVisibility(visibility);
        
        parent.saveChanges();
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, calendar.getId());
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, calendar.getPath());
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_UPDATED, _currentUserProvider.getUser(), eventParams));

        return getCalendarProperties(calendar);
    }
    
    /**
     * Delete a calendar
     * @param id The id of the calendar
     * @return The result map with id, parent id and message keys
     * @throws IllegalAccessException If the user has no sufficient rights
     */
    @Callable
    public Map<String, Object> deleteCalendar(String id) throws IllegalAccessException
    {
        Map<String, Object> result = new HashMap<>();

        assert id != null;
        
        AmetysObject object = _resolver.resolveById(id);
        if (!(object instanceof ModifiableCalendar))
        {
            throw new IllegalClassException(ModifiableCalendar.class, object.getClass());
        }
        
        ModifiableCalendar calendar = (ModifiableCalendar) object;
        
        // Check user right
        _explorerResourcesDAO.checkUserRight(object, RIGHTS_CALENDAR_DELETE);
        
        if (!_explorerResourcesDAO.checkLock(calendar))
        {
            getLogger().warn("User '" + _currentUserProvider.getUser() + "' try to delete calendar'" + object.getName() + "' but it is locked by another user");
            result.put("message", "locked");
            return result;
        }
        
        ModifiableExplorerNode parent = calendar.getParent();
        String parentId = parent.getId();
        String name = calendar.getName();
        String path = calendar.getPath();
        
        calendar.remove();
        parent.saveChanges();
     
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, id);
        eventParams.put(ObservationConstants.ARGS_PARENT_ID, parentId);
        eventParams.put(ObservationConstants.ARGS_NAME, name);
        eventParams.put(ObservationConstants.ARGS_PATH, path);
        
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.calendars.ObservationConstants.EVENT_CALENDAR_DELETED, _currentUserProvider.getUser(), eventParams));
        
        result.put("id", id);
        result.put("parentId", parentId);
        
        return result;
    }
        
    /**
     * Get or create the calendar ICS token
     * @param calendar The calendar
     * @param createIfNotExisting Create the token if none exists for the given calendar
     * @return The token
     */
    public String getCalendarIcsToken(Calendar calendar, boolean createIfNotExisting)
    {
        String token = calendar.getIcsUrlToken();
        
        if (createIfNotExisting && token == null && calendar instanceof ModifiableCalendar)
        {
            token = UUID.randomUUID().toString();
            ((ModifiableCalendar) calendar).setIcsUrlToken(token);
            ((ModifiableCalendar) calendar).saveChanges();
        }
        
        return token;
    }
    
    /**
     * Retrieve the calendar for the matching ICS token
     * @param token The ICS token
     * @return The calendar, or null if not found
     */
    public Calendar getCalendarFromIcsToken(String token)
    {
        if (StringUtils.isEmpty(token))
        {
            return null;
        }
        
        Expression expr = new StringExpression(JCRCalendar.CALENDAR_ICS_TOKEN, Operator.EQ, token);
        String calendarsQuery = QueryHelper.getXPathQuery(null, JCRCalendarFactory.CALENDAR_NODETYPE, expr);
        AmetysObjectIterable<Calendar> calendars = _resolver.query(calendarsQuery);
        AmetysObjectIterator<Calendar> calendarsIterator = calendars.iterator();
        
        if (calendarsIterator.getSize() > 0)
        {
            return calendarsIterator.next();
        }

        return null;
    }
    
    /**
     * Internal method to extract the data concerning the right of the current user for a calendar
     * @param calendar The calendar
     * @return The map of right data. Keys are the rights id, and values indicates whether the current user has the right or not.
     */
    protected  Map<String, Object> _extractCalendarRightData(Calendar calendar)
    {
        Map<String, Object> rightsData = new HashMap<>();
        UserIdentity user = _currentUserProvider.getUser();
        
        // Add
        rightsData.put("add-event", _rightManager.hasRight(user, RIGHTS_EVENT_ADD, calendar) == RightResult.RIGHT_ALLOW);
        
        // edit - delete
        rightsData.put("edit", _rightManager.hasRight(user, RIGHTS_CALENDAR_EDIT, calendar) == RightResult.RIGHT_ALLOW);
        rightsData.put("delete", _rightManager.hasRight(user, RIGHTS_CALENDAR_DELETE, calendar) == RightResult.RIGHT_ALLOW);
        
        return rightsData;
    }
    
    /**
     * Get the data of every available calendar for the current project
     * @return the list of calendars
     */
    @Callable
    public List<Map<String, Object>> getCalendars()
    {
        List<Map<String, Object>> calendarsData = new ArrayList<>();
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        AmetysObjectIterable<Calendar> calendars = calendarModule.getCalendars(_getProject());
        if (calendars != null)
        {
            for (Calendar calendar : calendars)
            {
                if (calendarModule.canView(calendar))
                {
                    calendarsData.add(this.getCalendarProperties(calendar));
                }
            }
        }
        
        return calendarsData;
    }

    /**
     * Get the colors of calendars
     * @return colors
     */
    @Callable
    public Map<String, CalendarColorsComponent.CalendarColor> getColors()
    {
        return _calendarColors.getColors();
    }
    
    /**
     * Get user rights from project name
     * @return the user rights
     */
    @Callable
    public Map<String, Object> getUserRights()
    {
        Map<String, Object> results = new HashMap<>();
        ModifiableTraversableAmetysObject calendarRoot = _getCalendarRoot();
        
        UserIdentity user = _currentUserProvider.getUser();
        results.put("canCreateCalendar", _rightManager.hasRight(user, RIGHTS_CALENDAR_ADD, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canEditCalendar", _rightManager.hasRight(user, RIGHTS_CALENDAR_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveCalendar", _rightManager.hasRight(user, RIGHTS_CALENDAR_DELETE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canCreateEvent", _rightManager.hasRight(user, RIGHTS_EVENT_ADD, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canEditEvent", _rightManager.hasRight(user, RIGHTS_EVENT_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveAnyEvent", _rightManager.hasRight(user, RIGHTS_EVENT_DELETE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canRemoveSelfEvent", _rightManager.hasRight(user, RIGHTS_EVENT_DELETE_OWN, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canCreateTags", _rightManager.hasRight(user, ProjectConstants.RIGHT_PROJECT_ADD_TAG, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canHandleResource", _rightManager.hasRight(user, RIGHTS_HANDLE_RESOURCE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("canBookResource", _rightManager.hasRight(user, RIGHTS_BOOK_RESOURCE, calendarRoot) == RightResult.RIGHT_ALLOW);
        results.put("sharePrivateCalendar", _rightManager.hasRight(user, RIGHTS_EVENT_EDIT, calendarRoot) == RightResult.RIGHT_ALLOW);
        
        return results;
    }

    /**
     * Get the calendar root
     * @return the calendar root
     */
    protected ModifiableTraversableAmetysObject _getCalendarRoot()
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        return calendarModule.getCalendarsRoot(_getProject(), true);
    }
    
    /**
     * Get the data of calendar used to store resources
     * @return the calendar used to store resources
     */
    @Callable
    public Map<String, Object> getResourceCalendar()
    {
        CalendarWorkspaceModule calendarModule = (CalendarWorkspaceModule) _workspaceModuleEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        
        Calendar calendar = calendarModule.getResourceCalendar(_getProject());
        return this.getCalendarProperties(calendar);
    }
}
