/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.actions;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.cocoon.ActionResultGenerator;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.calendars.CalendarDAO;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventJSONHelper;
import org.ametys.plugins.workspaces.calendars.helper.RecurrentEventHelper;

/**
 * Action providing an event of a project.
 */
public class GetEvent extends ServiceableAction
{
    /** The object resolver */
    protected AmetysObjectResolver _resolver;

    /** Calendar manager for workspaces */
    protected CalendarDAO _calendarDAO;
    
    /** Calendar event JSON helper */
    protected CalendarEventJSONHelper _calendarEventJSONHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _calendarDAO = (CalendarDAO) smanager.lookup(CalendarDAO.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _calendarEventJSONHelper = (CalendarEventJSONHelper) smanager.lookup(CalendarEventJSONHelper.ROLE);
    }

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Map<String, Object> result = new HashMap<>();
        request.setAttribute(ActionResultGenerator.MAP_REQUEST_ATTR, result);
        
        String eventId = request.getParameter("eventId");
        if (eventId == null)
        {
            return EMPTY_MAP;
        }
        
        CalendarEvent event = _resolver.resolveById(eventId);
        if (event != null)
        {
            //get the working days
            List<Integer> workingDayStrings = RecurrentEventHelper.getWorkingDays();
            List<String> icsWorkingDays = new ArrayList<>();
            for (Integer day : workingDayStrings)
            {
                switch (day)
                {
                    case java.util.Calendar.MONDAY:
                        icsWorkingDays.add("MO");
                        break;
                    case java.util.Calendar.TUESDAY:
                        icsWorkingDays.add("TU");
                        break;
                    case java.util.Calendar.WEDNESDAY:
                        icsWorkingDays.add("WE");
                        break;
                    case java.util.Calendar.THURSDAY:
                        icsWorkingDays.add("TH");
                        break;
                    case java.util.Calendar.FRIDAY:
                        icsWorkingDays.add("FR");
                        break;
                    case java.util.Calendar.SATURDAY:
                        icsWorkingDays.add("SA");
                        break;
                    case java.util.Calendar.SUNDAY:
                        icsWorkingDays.add("SU");
                        break;
                    default:
                        break;
                }
            }
            
            result.put("events", _calendarEventJSONHelper.eventAsJson(event, false, true));
            result.put("workingDays", icsWorkingDays);
        }

        return EMPTY_MAP;
    }
}
