/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.workflow;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang.IllegalClassException;

import org.ametys.core.observation.Event;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workspaces.calendars.Calendar;
import org.ametys.plugins.workspaces.calendars.ModifiableCalendar;
import org.ametys.plugins.workspaces.calendars.ObservationConstants;
import org.ametys.plugins.workspaces.calendars.events.CalendarEvent;
import org.ametys.plugins.workspaces.calendars.events.ModifiableCalendarEvent;
import org.ametys.plugins.workspaces.workflow.AbstractNodeWorkflowComponent;
import org.ametys.runtime.i18n.I18nizableText;

import com.opensymphony.module.propertyset.PropertySet;
import com.opensymphony.workflow.Workflow;
import com.opensymphony.workflow.WorkflowException;
import com.opensymphony.workflow.spi.WorkflowEntry;

/**
 * Action for editing a calendar event
 */
public class EditEventFunction extends AddEventFunction
{
    /** Workflow provider */
    protected WorkflowProvider _workflowProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _workflowProvider = (WorkflowProvider) smanager.lookup(WorkflowProvider.ROLE);
    }
    
    @Override
    public void execute(Map transientVars, Map args, PropertySet ps) throws WorkflowException
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> jsParameters = (Map<String, Object>) transientVars.get("parameters");
        if (jsParameters == null)
        {
            throw new WorkflowException("no parameters defined");
        }
       
        String eventId = (String) jsParameters.get("id");
        CalendarEvent event = _resolver.resolveById(eventId);
        
        String parentId = (String) jsParameters.get("parentId");
        Calendar calendar = _resolver.resolveById(parentId);
        
        if (!(calendar instanceof ModifiableCalendar))
        {
            throw new IllegalClassException(ModifiableCalendar.class, calendar.getClass());
        }
        
        if (!(event instanceof ModifiableCalendarEvent))
        {
            throw new IllegalClassException(ModifiableCalendarEvent.class, event.getClass());
        }
        
        ModifiableCalendarEvent mEvent = (ModifiableCalendarEvent) event;
        ModifiableCalendar mCalendar = (ModifiableCalendar) calendar;
        
        String choice = (String) jsParameters.get("choice");
        
        @SuppressWarnings("unchecked")
        Map<String, Object> result = (Map<String, Object>) transientVars.get("result");
        
        if ("unit".equals(choice))
        {
            // Create a new event from the edited occurrence
            _createEventFromOccurrence(mEvent, mCalendar, transientVars, jsParameters);
            
            // Exclude this occurrence from the initial event
            String occurrenceDateAsString = (String) jsParameters.get("occurrenceDate");
            ZonedDateTime occurrenceDate = DateUtils.parseZonedDateTime(occurrenceDateAsString);
            _excludeOccurrence(mEvent, occurrenceDate);
            
            mEvent.saveChanges();
            
            _notifyListeners(mEvent);
        }
        else
        {
            // FIXME EXPLORER-441 Remove "renameIfExists" on calendar events workflow
            // boolean renameIfExists = (Boolean) jsParameters.get("renameIfExists");
            assert parentId != null;
            
            AmetysObject object = _resolver.resolveById(parentId);
            if (!(object instanceof ModifiableCalendar))
            {
                throw new IllegalClassException(ModifiableCalendar.class, object.getClass());
            }
            
            long workflowId = ((WorkflowEntry) transientVars.get("entry")).getId();
            event.setWorkflowId(workflowId);
            
            // Compute the event's dates from the dates of the modified occurrence
            ZonedDateTime[] dates = _computeEventDates(event, jsParameters);
            
            _setEventData(mEvent, transientVars, jsParameters);
            
            mEvent.setStartDate(dates[0]);
            mEvent.setEndDate(dates[1]);
            
            mEvent.saveChanges();
            
            result.put("id", event.getId());
            result.put("parentId", parentId);
            result.put("event", _calendarEventJSONHelper.eventAsJsonWithOccurrences(event, false, event.getStartDate(), event.getEndDate()));
            
            // Notify listeners
            _notifyListeners(mEvent);
        }
    }
    
    /**
     * Create a new event from the occurrence of a event 
     * @param initialEvent The initial event
     * @param parentCalendar The parent calendar
     * @param transientVars The transient variable
     * @param jsParameters The JS parameters
     */
    protected void _createEventFromOccurrence (CalendarEvent initialEvent, Calendar parentCalendar, Map transientVars, Map<String, Object> jsParameters)
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> result = (Map<String, Object>) transientVars.get("result");
        
        // Computed workflow parameters to create a new event
        if (!jsParameters.containsKey("title"))
        {
            jsParameters.put("title", initialEvent.getTitle());
            jsParameters.put("desc", initialEvent.getDescription());
            jsParameters.put("fullDay", initialEvent.getFullDay());
        }
        jsParameters.put("recurrenceType", EventRecurrenceTypeEnum.NEVER.toString());
        jsParameters.put("untilDate", null);
        
        Map<String, Object> inputs = new HashMap<>();
        inputs.put("parameters", jsParameters);
        inputs.put(AbstractNodeWorkflowComponent.EXPLORERNODE_KEY, parentCalendar);
        inputs.put("result", result);
        
        String workflowName = parentCalendar.getWorkflowName();
        try
        {
            // Create a new event from the edited occurrence
            Workflow workflow = _workflowProvider.getAmetysObjectWorkflow();
            workflow.initialize(workflowName, 1, inputs);
        }
        catch (WorkflowException e)
        {
            throw new AmetysRepositoryException("Unable to create a new event with workflow '" + workflowName + "' and action 1'", e);
        }
    }
    
    /**
     * Exclude a occurrence of a event
     * @param event The event
     * @param occurrenceDate The date to exclude
     */
    protected void _excludeOccurrence (ModifiableCalendarEvent event, ZonedDateTime occurrenceDate)
    {
        List<ZonedDateTime> excludedOccurrences = new ArrayList<>(event.getExcludedOccurences());
        
        excludedOccurrences.add(occurrenceDate.truncatedTo(ChronoUnit.DAYS));
        
        // Exclude the occurrence
        event.setExcludedOccurrences(excludedOccurrences);
    }
    
    /**
     * Compute the start date and end date of the event from the edited occurrence
     * @param event The event
     * @param jsParameters The JS parameters
     * @return an array with computed start and end dates
     */
    protected ZonedDateTime[] _computeEventDates (CalendarEvent event, Map<String, Object> jsParameters)
    {
        // Date of the edited occurrence before edition
        String occurrenceDateAsString = (String) jsParameters.get("occurrenceDate");
        ZonedDateTime occurrenceDate = DateUtils.parseZonedDateTime(occurrenceDateAsString);
        
        // Dates of the event before edition
        long oldDiff = ChronoUnit.SECONDS.between(event.getStartDate(), event.getEndDate());
        
        ZonedDateTime occurrenceEndDate = occurrenceDate.plusSeconds(oldDiff);
        
        String startDateAsString = (String) jsParameters.get("startDate");
        ZonedDateTime startDate = DateUtils.parseZonedDateTime(startDateAsString);
        String endDateAsString = (String) jsParameters.get("endDate");
        ZonedDateTime endDate = DateUtils.parseZonedDateTime(endDateAsString);
        
        long startDiff = ChronoUnit.SECONDS.between(occurrenceDate, startDate);
        long endDiff = ChronoUnit.SECONDS.between(occurrenceEndDate, endDate);
        
        // Compute the new start and end date from the edited occurrence
        return new ZonedDateTime[] {event.getStartDate().plusSeconds(startDiff), event.getEndDate().plusSeconds(endDiff)};
    }
    
    
    /**
     * Notify listeners that the event has been updated
     * @param event The updated event
     */
    @Override
    protected void _notifyListeners (CalendarEvent event)
    {
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CALENDAR, event.getParent());
        eventParams.put(ObservationConstants.ARGS_CALENDAR_EVENT, event);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, event.getId());
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_PARENT_ID, event.getParent().getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    @Override
    public I18nizableText getLabel()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_EDIT_EVENT_FUNCTION_LABEL");
    }
}
