/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.categories;

import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

/**
 * Helper for project's categories
 *
 */
public class CategoryHelper implements Serviceable, Component
{
    /** The avalon role */
    public static final String ROLE = CategoryHelper.class.getName();
    
    private CategoryProviderExtensionPoint _categoryProviderEP;

    private CategoryJCRProvider _jcrCategoryProvider;

    private CategoryColorsComponent _categoryColorsComponent;
    
    public void service(ServiceManager smanager) throws ServiceException
    {
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _jcrCategoryProvider = (CategoryJCRProvider) _categoryProviderEP.getExtension(CategoryJCRProvider.class.getName());
        _categoryColorsComponent = (CategoryColorsComponent) smanager.lookup(CategoryColorsComponent.ROLE);
    }
    
    /**
     * Get the root categories
     * @return the root categories
     */
    public Collection<Category> getRootCategories()
    {
        return _jcrCategoryProvider.getTags(Collections.EMPTY_MAP).values();
    }
    
    /**
     * Get the leaf categories
     * @return the leaf categories
     */
    public Collection<Category> getLeafCategories()
    {
        Collection<Category> rootCategories = getRootCategories();
        
        List<Category> leafCategories = new ArrayList<>();
        for (Category category : rootCategories)
        {
            leafCategories.addAll(getLeafCategories(category));
        }
        
        return leafCategories;
    }
    
    /**
     * Get the leaf categories of a given category
     * @param category the category
     * @return the leaf categories or the category it self if it has no child
     */
    public Collection<Category> getLeafCategories(Category category)
    {
        Map<String, Category> childTags = category.getTags();
        if (childTags.isEmpty())
        {
            return Collections.singletonList(category);
        }
        else
        {
            List<Category> leafCategories = new ArrayList<>();
            for (Category childCategory : childTags.values())
            {
                leafCategories.addAll(getLeafCategories(childCategory));
            }
            return leafCategories;
        }
    }
    
    /**
     * Get the (flattened) child categories recursively
     * @param category the category
     * @return the child categories
     */
    public Collection<Category> getChildCategories(Category category)
    {
        List<Category> childCategories = new ArrayList<>();
        
        for (Category childCategory : category.getTags().values())
        {
            childCategories.add(childCategory);
            childCategories.addAll(getChildCategories(childCategory));
        }
        
        return childCategories;
    }
    
    /**
     * Get the colors of a category.
     * @param category the category. Can be null to get default color.
     * @return the colors associated to this category or the default color if there is no category or if the category has no color
     */
    public Map<String, String> getCategoryColor(Category category)
    {
        Category categoryForColor = category;
        while (categoryForColor != null && categoryForColor.getColor(false) == null)
        {
            // if category has no color, try to get color from its closest parent category
            categoryForColor = categoryForColor.getParent();
        }
        
        String colorIndex = StringUtils.defaultIfBlank(categoryForColor != null ? categoryForColor.getColor(false) : "", _categoryColorsComponent.getDefaultKey());
        Map<String, String> color =  _categoryColorsComponent.getColors().get(colorIndex);
        if (color == null)
        {
            color = _categoryColorsComponent.getColors().get(_categoryColorsComponent.getDefaultKey());
        }
        return color;
    }
}
