/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.cmis;

import java.io.IOException;
import java.io.InputStream;
import java.math.BigInteger;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.chemistry.opencmis.commons.data.Acl;
import org.apache.chemistry.opencmis.commons.data.ContentStream;
import org.apache.chemistry.opencmis.commons.data.ExtensionsData;
import org.apache.chemistry.opencmis.commons.data.FailedToDeleteData;
import org.apache.chemistry.opencmis.commons.data.ObjectData;
import org.apache.chemistry.opencmis.commons.data.ObjectInFolderList;
import org.apache.chemistry.opencmis.commons.data.ObjectParentData;
import org.apache.chemistry.opencmis.commons.data.Properties;
import org.apache.chemistry.opencmis.commons.data.RepositoryInfo;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinition;
import org.apache.chemistry.opencmis.commons.definitions.TypeDefinitionList;
import org.apache.chemistry.opencmis.commons.enums.IncludeRelationships;
import org.apache.chemistry.opencmis.commons.enums.UnfileObject;
import org.apache.chemistry.opencmis.commons.enums.VersioningState;
import org.apache.chemistry.opencmis.commons.exceptions.CmisInvalidArgumentException;
import org.apache.chemistry.opencmis.commons.exceptions.CmisPermissionDeniedException;
import org.apache.chemistry.opencmis.commons.impl.server.AbstractCmisService;
import org.apache.chemistry.opencmis.commons.server.CallContext;
import org.apache.chemistry.opencmis.commons.spi.Holder;
import org.apache.commons.io.IOUtils;
import org.apache.excalibur.source.Source;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * AbstractCmisService implementation
 */
public class CmisServiceImpl extends AbstractCmisService
{
    private CmisServiceFactory _factory;
    private CallContext _context;
    private CmisRepository _repository;
    private boolean _isAuthenticated;
    /**
     * CmisServiceImpl implementation
     * @param context the call context
     * @param factory the factory used to build this
     */
    public CmisServiceImpl(CallContext context, CmisServiceFactory factory)
    {
        this._factory = factory;
        this._context = context;
        this._isAuthenticated = false;
        this._repository = new CmisRepository();
    }
    
    private Project authenticateAndGetProject(String encodedProjectId)
    {
        String decodedProjectId = URIUtils.decode(URIUtils.decode(encodedProjectId));
        Project project = this.getProject(decodedProjectId, _factory);
        UserIdentity user = this.authenticate(getCallContext(), project.getName(), _factory);
        Set<String> grantedSites = _factory.getSiteManager().getGrantedSites(user);
        
        if (user != null && grantedSites.contains(project.getName()) && isDocumentModuleActivated(project))
        {
            return project;
        }
        
        throw new CmisPermissionDeniedException("Authentication process failure");
    }
    
    @Override
    public List<RepositoryInfo> getRepositoryInfos(ExtensionsData extension)
    {
        List<RepositoryInfo> result = new ArrayList<>();

        AmetysObjectIterable<Project> projects = _factory.getProjectManager().getProjects();
        
        this.authenticate(getCallContext(), null, _factory);
        UserIdentity user = _factory.getCurrentUserProvider().getUser();
        
        Set<String> grantedSites = _factory.getSiteManager().getGrantedSites(user);
        for (Project project : projects)
        {
            if (user != null && grantedSites.contains(project.getName()) && isDocumentModuleActivated(project))
            {
                result.add(_repository.getRepositoryInfo(getCallContext(), project, _factory));
            }
        }
        return result;
    }

    @Override
    public TypeDefinitionList getTypeChildren(String repositoryId, String typeId, Boolean includePropertyDefinitions, BigInteger maxItems, BigInteger skipCount,
            ExtensionsData extension)
    {
        authenticateAndGetProject(repositoryId);
        return _repository.getTypeChildren(getCallContext(), typeId, includePropertyDefinitions, maxItems, skipCount, _factory);
    }

    @Override
    public TypeDefinition getTypeDefinition(String repositoryId, String typeId, ExtensionsData extension)
    {
        authenticateAndGetProject(repositoryId);
        return _repository.getTypeDefinition(getCallContext(), typeId, _factory);
    }

    @Override
    public ObjectInFolderList getChildren(String repositoryId, String folderId, String filter, String orderBy, Boolean includeAllowableActions,
            IncludeRelationships includeRelationships, String renditionFilter, Boolean includePathSegment, BigInteger maxItems, BigInteger skipCount, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedFolderId = URIUtils.decode(URIUtils.decode(folderId));
        return _repository.getChildren(getCallContext(), decodedFolderId, project, renditionFilter, includeAllowableActions, includePathSegment, maxItems, skipCount, this, _factory);
    }

    @Override
    public List<ObjectParentData> getObjectParents(String repositoryId, String objectId, String filter, Boolean includeAllowableActions, IncludeRelationships includeRelationships,
            String renditionFilter, Boolean includeRelativePathSegment, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedObjectId = URIUtils.decode(URIUtils.decode(objectId));
        return _repository.getObjectParents(getCallContext(), decodedObjectId, project, renditionFilter, includeAllowableActions, includeRelativePathSegment, this, _factory);
    }

    @Override
    public ObjectData getObject(String repositoryId, String objectId, String filter, Boolean includeAllowableActions, IncludeRelationships includeRelationships,
            String renditionFilter, Boolean includePolicyIds, Boolean includeAcl, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedObjectId = URIUtils.decode(URIUtils.decode(objectId));
        return _repository.getObject(getCallContext(), project, decodedObjectId, null, renditionFilter, includeAllowableActions, includeAcl, this, _factory);
    }
    
    @Override
    public ContentStream getContentStream(String repositoryId, String objectId, String streamId, BigInteger offset,
            BigInteger length, ExtensionsData extension)
    {
        authenticateAndGetProject(repositoryId);
        String decodedObjectId = URIUtils.decode(URIUtils.decode(objectId));
        return _repository.getContentStream(getCallContext(), decodedObjectId, offset, length, _factory);
    }
    
    @Override
    public ObjectData getObjectByPath(String repositoryId, String path, String filter, Boolean includeAllowableActions,
            IncludeRelationships includeRelationships, String renditionFilter, Boolean includePolicyIds,
            Boolean includeAcl, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        return _repository.getObjectByPath(getCallContext(), project, path, renditionFilter, includeAllowableActions, includeAcl, this, _factory);
    }
    
    @Override
    public String createFolder(String repositoryId, Properties properties, String folderId, List<String> policies,
            Acl addAces, Acl removeAces, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedFolderId = URIUtils.decode(URIUtils.decode(folderId));
        return _repository.createFolder(getCallContext(), properties, project, decodedFolderId, _factory);
    }
    
    @Override
    public String createDocument(String repositoryId, Properties properties, String folderId,
            ContentStream contentStream, VersioningState versioningState, List<String> policies, Acl addAces,
            Acl removeAces, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedFolderId = URIUtils.decode(URIUtils.decode(folderId));
        return _repository.createDocument(getCallContext(), properties, project, decodedFolderId, contentStream, versioningState, _factory);
    }

    @Override
    public void deleteObjectOrCancelCheckOut(String repositoryId, String objectId, Boolean allVersions,
            ExtensionsData extension)
    {
        authenticateAndGetProject(repositoryId);
        String decodedObjectId = URIUtils.decode(URIUtils.decode(objectId));
        _repository.deleteObject(getCallContext(), decodedObjectId, _factory);
    }
    
    @Override
    public FailedToDeleteData deleteTree(String repositoryId, String folderId, Boolean allVersions,
            UnfileObject unfileObjects, Boolean continueOnFailure, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedFolderId = URIUtils.decode(URIUtils.decode(folderId));
        return _repository.deleteTree(getCallContext(), project, decodedFolderId, _factory);
    }
    
    @Override
    public ObjectData getFolderParent(String repositoryId, String folderId, String filter, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        String decodedFolderId = URIUtils.decode(URIUtils.decode(folderId));
        return _repository.getFolderParent(getCallContext(), project, decodedFolderId, filter, this, _factory);
    }
    
    @Override
    public void updateProperties(String repositoryId, Holder<String> objectId, Holder<String> changeToken,
            Properties properties, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        _repository.updateProperties(getCallContext(), project, objectId, properties, this, _factory);
    }
    
    @Override
    public void setContentStream(String repositoryId, Holder<String> objectId, Boolean overwriteFlag,
            Holder<String> changeToken, ContentStream contentStream, ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        _repository.changeContentStream(getCallContext(), project, objectId, overwriteFlag, contentStream, false, _factory);
    }
    
    @Override
    public void moveObject(String repositoryId, Holder<String> objectId, String targetFolderId, String sourceFolderId,
            ExtensionsData extension)
    {
        Project project = authenticateAndGetProject(repositoryId);
        _repository.moveObject(getCallContext(), project, objectId, targetFolderId, this, _factory);
    }
    
    private UserIdentity authenticate(CallContext context, String projectName, CmisServiceFactory factory)
    {
        if (this._isAuthenticated)
        {
            return _factory.getCurrentUserProvider().getUser();
        }
        
        if (context == null)
        {
            throw new CmisPermissionDeniedException("No user context!");
        }
        
        String userName = context.getUsername();
        String password = context.getPassword();
        
        Source source = null;
        try
        {
            if (projectName == null)
            {
                source = factory.getSourceResolver().resolveURI("cocoon:/_authenticate?token=" + URIUtils.encodeParameter(password));
            }
            else
            {
                HttpServletRequest request = (HttpServletRequest) context.get(CallContext.HTTP_SERVLET_REQUEST);
                request.setAttribute("projectName", projectName);
                source = factory.getSourceResolver().resolveURI("cocoon:/_authenticate/" + projectName + "?token=" + URIUtils.encodeParameter(password));
            }
            try (InputStream is = source.getInputStream())
            {
                IOUtils.toString(is, StandardCharsets.UTF_8);
            }
        }
        catch (IOException e)
        {
            throw new CmisPermissionDeniedException("Authentication process failure", e);
        }
        finally 
        {
            factory.getSourceResolver().release(source);
        }
        
        UserIdentity user = _factory.getCurrentUserProvider().getUser();
        if (user == null || !user.getLogin().equals(userName))
        {
            throw new CmisPermissionDeniedException("Authentication process failure");
        }
        this._isAuthenticated = true;
        return user;
    }
    
    private Project getProject(String projectId, CmisServiceFactory factory)
    {
        AmetysObject repositoryAmetysObject = factory.getResolver().resolveById(projectId);
        if (repositoryAmetysObject instanceof Project)
        {
            return (Project) repositoryAmetysObject;
        }
        else
        {
            throw new CmisInvalidArgumentException("Project is not valid.");
        }
    }
    
    /**
     * Sets the call context.
     * 
     * This method should only be called by the service factory.
     * @param context context
     */
    public void setCallContext(CallContext context)
    {
        this._context = context;
    }

    /**
     * Gets the call context.
     * @return CallContext
     */
    public CallContext getCallContext()
    {
        return _context;
    }
    
    private boolean isDocumentModuleActivated(Project project)
    {
        return _factory.getProjectManager().isModuleActivated(project, _factory.getDocumentModule().getId());
    }

}
