/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.minisite;

import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.SitemapElement;

/**
 * Abstract observer for events which have to be convert to minisite page event
 *
 */
public abstract class AbstractConvertMinisiteEventObserver extends AbstractLogEnabled implements Observer, Serviceable
{
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The observation manager */
    protected ObservationManager _observationManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _observationManager = (ObservationManager) serviceManager.lookup(ObservationManager.ROLE);
    }
    
    public int getPriority(Event event)
    {
        return MAX_PRIORITY;
    }
    
    /**
     * Determines if the page is part of minisite project module (root page excluded)
     * @param project the project. Can not be null
     * @param page the page
     * @return true if the page is a page of minisite
     */
    protected boolean isMinisitePage(Project project, SitemapElement page)
    {
        Set<Page> modulePages = _projectManager.getModulePages(project, MiniSiteWorkspaceModule.MINISITE_MODULE_ID);
        if (!modulePages.isEmpty())
        {
            Page minisiteRootPage = modulePages.iterator().next();
            return page.getPathInSitemap().startsWith(minisiteRootPage.getPathInSitemap() + "/");
        }
        
        return false;
    }
}
