/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.stream.Collectors;

import org.apache.commons.lang.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Observer to send mail notifications on workspace member removal
 */
public class RemoveMemberMailManagersNotifierObserver extends AbstractMemberMailNotifierObserver
{
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_MEMBER_DELETED);
    }

    @Override
    protected String getMailBodyURI(Event event, Project project)
    {
        return "cocoon://_plugins/workspaces/notification-mail-member-removed-manager";
    }
    
    @Override
    protected List<String> getUserToNotify(Event event, Project project)
    {
        // Recipients are project managers
        return Arrays.stream(project.getManagers())
                .map(_userManager::getUser)
                .filter(Objects::nonNull)
                .map(User::getEmail)
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toList());
    }
    
    @Override
    protected I18nizableText getI18nSubject(Event event, Project project)
    {
        return new I18nizableText("plugin." + _pluginName, "PROJECT_MAIL_MANAGER_NOTIFICATION_SUBJECT_MEMBER_DELETED", List.of(project.getTitle()));
    }
}
