/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.Map;

import org.apache.avalon.framework.activity.Initializable;
import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.user.CurrentUserProvider;

/**
 * Notification helper for the Webdav actions
 */
public class ResourceNotifierHelper implements Serviceable, Component, Initializable
{
    /** Avalon role. */
    public static final String ROLE = ResourceNotifierHelper.class.getName();
    
    /** Time of expiration of a log (5 minutes) */
    private static final long __HISTORY_EXPIRATION_TIME = 5 * 60 * 1000; 
    /** Time duration between 2 cleansing of logs queue (10 seconds) */
    private static final long __TIME_DURATION_BEFORE_NEXT_CLEAN = 10 * 1000; 
    
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    private LinkedList<Map<String, Object>> _sentHistory;
    
    private long _lastCleanTime;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }

    public void initialize() throws Exception
    {
        _sentHistory = new LinkedList<>();
    }
    
    /**
     * Notify the observation manager when a resource is updated
     * @param resourceId The resource updated
     * @return True if the resource was recently modified
     */
    public boolean resourceRecentlyModified(String resourceId)
    {
        if (resourceId == null)
        {
            return false;
        }
        
        long now = System.currentTimeMillis();
        synchronized (_sentHistory)
        {
            _cleanupHistory();
        
            for (Map<String, Object> map : _sentHistory)
            {
                if (map.get("resourceId").equals(resourceId))
                {
                    return true;
                }
            }
            
            Map<String, Object> history = new HashMap<>();
            history.put("time", now);
            history.put("resourceId", resourceId);
            _sentHistory.addLast(history);
        }
        
        return false;
    }

    private void _cleanupHistory()
    {
        if (_lastCleanTime != 0 && _lastCleanTime + __TIME_DURATION_BEFORE_NEXT_CLEAN > System.currentTimeMillis())
        {
            // The time between 2 cleansing has not yet expired, do nothing
            return;
        }

        long from = System.currentTimeMillis() - __HISTORY_EXPIRATION_TIME;
        // Remove history before 'from'
        Map<String, Object> oldHistory = _sentHistory.peek();
        while (oldHistory != null && (long) oldHistory.get("time") < from)
        {
            _sentHistory.remove();
            oldHistory = _sentHistory.peek();
        }
        
        _lastCleanTime = System.currentTimeMillis();
    }


}
