/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.notification.preferences;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesExtensionPoint;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for notifications preferences
 */
public class NotificationPreferencesHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** the id of the user prefs for notification preferences */
    public static final String USERPREFS_ID = "workspaces.notifications";
    
    /** The avalon role */
    public static final String ROLE = NotificationPreferencesHelper.class.getName();
    
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The user preferences manager */
    protected UserPreferencesManager _userPrefManager;
    /** The user preferences extension point */
    protected UserPreferencesExtensionPoint _userPrefEP;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** To get the current user */
    protected CurrentUserProvider _currentUserProvider;
    
    /**
     * The frequency of notifications
     *
     */
    public static enum Frequency 
    {
        /** Each frequency */
        EACH,
        /** Daily frequency */
        DAILY,
        /** Weekly frequency */
        WEEKLY,
        /** Monthly frequency */
        MONTHLY
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _userPrefManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE + ".FO");
        _userPrefEP = (UserPreferencesExtensionPoint) manager.lookup(UserPreferencesExtensionPoint.ROLE + ".FO");
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
    }
    
    /**
     * Determines if a user asked to be notify for a given project and frequency
     * @param user the user
     * @param projectName the project name
     * @param frequency the required frequency (each, daily, weekly or monthly)
     * @return true if the user need to be notified
     */
    public boolean askedToBeNotified(User user, String projectName, Frequency frequency)
    {
        return askedToBeNotified(user.getIdentity(), projectName, frequency);
    }
    
    /**
     * Determines if a user asked to be notify for a given project and frequency
     * @param userIdentity the user identity
     * @param projectName the project name
     * @param frequency the required frequency (each, daily, weekly or monthly). Cannot be null.
     * @return true if the user need to be notified
     */
    public boolean askedToBeNotified(UserIdentity userIdentity, String projectName, Frequency frequency)
    {
        try
        {
            Map<String, Object> notificationPreferences = getNotificationPreferences(userIdentity);
            
            if ((Boolean) notificationPreferences.get("disable"))
            {
                return false;
            }
            
            @SuppressWarnings("unchecked")
            Map<String, Object> projectsPreferences = (Map<String, Object>) notificationPreferences.get("projects");
            if (!projectsPreferences.isEmpty() && projectsPreferences.get(projectName) != null)
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> projectPreference = (Map<String, Object>) projectsPreferences.get(projectName);
                return !(Boolean) projectPreference.get("disable") && StringUtils.equalsIgnoreCase(frequency.name(), (String) (projectPreference.get("frequency")));
            }
            else
            {
                return StringUtils.equalsIgnoreCase(frequency.name(), (String) notificationPreferences.get("frequency"));
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Failed to retrieve the user preferences for user {}", UserIdentity.userIdentityToString(userIdentity), e);
            return false;
        }
    }
    
    /**
     * Get the collection of paused project for a given user
     * @param user The user to consider
     * @return The set of projects names that are paused or null for a general pause
     */
    @SuppressWarnings("unchecked")
    public Set<String> getPausedProjects(UserIdentity user)
    {
        try
        {
            Map<String, Object> notificationPreferences = getNotificationPreferences(user);
            
            if (Boolean.TRUE.equals(notificationPreferences.get("disable")))
            {
                return null;
            }
            
            Set<String> paused = new HashSet<>();
            
            Map<String, Object> projectsPreferences = (Map<String, Object>) notificationPreferences.get("projects");
            for (Entry<String, Object> projectPreference : projectsPreferences.entrySet())
            {
                if (Boolean.TRUE.equals(((Map<String, Object>) projectPreference.getValue()).get("disable")))
                {
                    paused.add(projectPreference.getKey());
                }
            }
            
            return paused;
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Failed to retrieve the user preferences for user {}", UserIdentity.userIdentityToString(user), e);
            return Set.of();
        }
    }
    
    /**
     * Change the pause status for the notifications of one project
     * @param projectName The project name to change
     * @param pause The pause status
     * @return true is the change was done successfully
     */
    @SuppressWarnings("unchecked")
    @Callable
    public boolean setPauseProject(String projectName, boolean pause)
    {
        UserIdentity user = _currentUserProvider.getUser();
        try
        {
            Map<String, Object> notificationPreferences = getNotificationPreferences(user);
            
            Map<String, Object> projectsPreferences = (Map<String, Object>) notificationPreferences.computeIfAbsent("projects", s -> new HashMap());
            Map<String, Object> projectPreference = (Map<String, Object>) projectsPreferences.computeIfAbsent(projectName, s -> new HashMap());
            
            if (!pause && notificationPreferences.get("frequency").equals(projectPreference.get("frequency")))
            {
                projectsPreferences.remove(projectName); 
            }
            else
            {
                projectPreference.put("disable", pause);
                projectPreference.computeIfAbsent("frequency", s -> notificationPreferences.get("frequency"));
            }

            String preferencesAsString = _jsonUtils.convertObjectToJson(notificationPreferences);
                
            Map<String, String> values = _userPrefManager.getUnTypedUserPrefs(user, _getStorageContext(), Map.of());
            
            values.put(USERPREFS_ID, preferencesAsString);
            
            _userPrefManager.setUserPreferences(user, _getStorageContext(), Map.of(), values);
            
            return true;
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("Could not change favorite status of user " + UserIdentity.userIdentityToString(user) + " on project " + projectName);
            return false; 
        }
    }
    
    /**
     * Determines if a user has at least one notification preference with the given frequency
     * @param user the user
     * @param frequency the frequency
     * @return true if the user has at least one notification preference with the given frequency 
     */
    public boolean hasFrequencyInPreferences(User user, Frequency frequency)
    {
        // TODO We should use a cache here to avoid computing the userPref multiple time for the same user
        try
        {
            String notificationPreferencesJSON = getNotificationPreferencesAsString(user);
            return notificationPreferencesJSON.contains(frequency.name().toLowerCase());
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Failed to retrieve the user preferences for user {}", user.getFullName(), e);
        }
        return false;
    }
    
    private boolean _isDisabled(Map<String, Map<String, Object>> projectPrefs, String projectName)
    {
        return projectPrefs.containsKey(projectName) && (Boolean) projectPrefs.get(projectName).get("disable");
    }
    
    private boolean _matchFrequency(Map<String, Map<String, Object>> projectPrefs, String projectName, Frequency frequency)
    {
        return projectPrefs.containsKey(projectName) && StringUtils.equalsIgnoreCase(frequency.name(), (String) projectPrefs.get(projectName).get("frequency"));
    }
    
    /**
     * Get the user's project set to receive notification for given frequency
     * @param user the user
     * @param frequency the matching frequency
     * @return The names of user's project's with same frequency
     */
    public Set<String> getUserProjectsWithFrequency(User user, Frequency frequency)
    {
        try
        {
            Map<String, Object> notificationPreferences = getNotificationPreferences(user);
            if ((Boolean) notificationPreferences.get("disable"))
            {
                // Notifications are disabled for user
                return Collections.EMPTY_SET;
            }
            
            String globalFrequency = (String) notificationPreferences.get("frequency");
            @SuppressWarnings("unchecked")
            Map<String, Map<String, Object>> customProjectsPrefs = (Map<String, Map<String, Object>>) notificationPreferences.get("projects");
            
            if (globalFrequency.equals(frequency.name().toLowerCase()))
            {
                // Get all user projects and filter custom projects
                Set<Project> userProjects = _projectManager.getUserProjects(user.getIdentity()).keySet();
                
                return userProjects.stream()
                    .map(Project::getName)
                    .filter(p -> !_isDisabled(customProjectsPrefs, p))
                    .filter(p -> !customProjectsPrefs.containsKey(p) || _matchFrequency(customProjectsPrefs, p, frequency))
                    .collect(Collectors.toSet());
            }
            else
            {
                // Get custom projects with same frequency
                return customProjectsPrefs.entrySet()
                    .stream()
                    .filter(e -> !(Boolean) e.getValue().get("disable"))
                    .filter(e -> frequency.name().toLowerCase().equals(e.getValue().get("frequency")))
                    .map(e -> e.getKey())
                    .collect(Collectors.toSet());
                            
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Failed to retrieve the user preferences for user {}", user.getFullName(), e);
        }
        
        return Collections.EMPTY_SET;
    }
    
    /**
     * Get the user's notification preferences
     * @param user the user 
     * @return the user's notification preferences
     * @throws UserPreferencesException if failed to retrieve user's preferences
     */
    public Map<String, Object> getNotificationPreferences(User user) throws UserPreferencesException
    {
        return getNotificationPreferences(user.getIdentity());
    }
    
    /**
     * Get the user's notification preferences
     * @param userIdentity the user identity
     * @return the user's notification preferences
     * @throws UserPreferencesException if failed to retrieve user's preference
     */
    public Map<String, Object> getNotificationPreferences(UserIdentity userIdentity) throws UserPreferencesException
    {
        String prefAsString = getNotificationPreferencesAsString(userIdentity);
        return _jsonUtils.convertJsonToMap(prefAsString);
    }
    
    /**
     * Get the user's notification preferences as json String
     * @param user the user
     * @return the user's notification preferences as json String
     * @throws UserPreferencesException if failed to retrieve user's preference
     */
    public String getNotificationPreferencesAsString(User user) throws UserPreferencesException
    {
        return getNotificationPreferencesAsString(user.getIdentity());
    }
    
    
    /**
     * Get the user's notification preferences as json String
     * @param userIdentity the user identity
     * @return the user's notification preferences as json String
     * @throws UserPreferencesException if failed to retrieve user's preference
     */
    public String getNotificationPreferencesAsString(UserIdentity userIdentity) throws UserPreferencesException
    {
        String notificationPreferencesJSON = _userPrefManager.getUserPreferenceAsString(userIdentity, _getStorageContext(), Map.of(), USERPREFS_ID);
        if (notificationPreferencesJSON == null)
        {
            // get default value
            notificationPreferencesJSON = (String) _userPrefEP.getUserPreference(Map.of(), USERPREFS_ID).getDefaultValue();
        }
        
        return notificationPreferencesJSON;
    }

    /**
     * Delete the user preferences for notification on the given project
     * @param userIdentity the user identity
     * @param projectName the name of the project to delete
     */
    public void deleteProjectNotificationPreferences(UserIdentity userIdentity, String projectName)
    {
        try
        {
            Map<String, String> values = _userPrefManager.getUnTypedUserPrefs(userIdentity, _getStorageContext(), Map.of());
            
            String notificationPreferencesJSON = values.get(USERPREFS_ID);
            Map<String, Object> notificationPreferences = _jsonUtils.convertJsonToMap(notificationPreferencesJSON);
            if (notificationPreferences.containsKey("projects"))
            {
                @SuppressWarnings("unchecked")
                Map<String, Object> projectsPreferences = (Map<String, Object>) notificationPreferences.get("projects");
                if (projectsPreferences.containsKey(projectName))
                {
                    projectsPreferences.remove(projectName);
                    notificationPreferencesJSON = _jsonUtils.convertObjectToJson(notificationPreferences);
                    
                    values.put(USERPREFS_ID, notificationPreferencesJSON);
                    
                    _userPrefManager.setUserPreferences(userIdentity, _getStorageContext(), Map.of(), values);
                }
            }
        }
        catch (UserPreferencesException e)
        {
            getLogger().warn("Failed to delete the notification user preference for project {} of user {}", projectName, UserIdentity.userIdentityToString(userIdentity), e);
        }
    }

    private String _getStorageContext()
    {
        return "/sites/" + _projectManager.getCatalogSiteName();
    }
    

}
