/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;
import org.apache.http.annotation.Obsolete;

import org.ametys.core.right.Profile;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.right.RightProfilesDAO;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.about.AboutWorkspaceModule;
import org.ametys.plugins.workspaces.alert.AlertWorkspaceModule;
import org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.members.MembersWorkspaceModule;
import org.ametys.plugins.workspaces.minisite.MiniSiteWorkspaceModule;
import org.ametys.plugins.workspaces.news.NewsWorkspaceModule;
import org.ametys.plugins.workspaces.project.ProjectConstants;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.wall.WallContentModule;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper related to rights management for projects.
 */
public class ProjectRightHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** Avalon Role */
    public static final String ROLE = ProjectRightHelper.class.getName();
    
    @Obsolete // For v1 project only
    private static final String __PROJECT_RIGHT_PROFILE = "PROJECT";
    
    /** Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    
    /** Project manager */
    protected ProjectManager _projectManager;
    
    /** Right manager */
    protected RightManager _rightManager;
    
    /** Right profiles manager */
    protected RightProfilesDAO _rightProfilesDao;
    
    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** Workspace Module ExtensionPoint */
    protected WorkspaceModuleExtensionPoint _workspaceModuleEP;
    
    /** Association ContentTypeId, Module */
    protected Map<String, WorkspaceModule> _contentTypesToModule;

    /** Module managers EP */
    protected WorkspaceModuleExtensionPoint _moduleManagerEP;

    private Set<String> _profileIds;

    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _rightProfilesDao = (RightProfilesDAO) manager.lookup(RightProfilesDAO.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _contentTypesToModule = Map.of(
                WorkspacesConstants.WALL_CONTENT_CONTENT_TYPE_ID, _workspaceModuleEP.getModule(WallContentModule.WALLCONTENT_MODULE_ID),
                WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID, _workspaceModuleEP.getModule(NewsWorkspaceModule.NEWS_MODULE_ID),
                WorkspacesConstants.PROJECT_ALERT_CONTENT_TYPE_ID, _workspaceModuleEP.getModule(AlertWorkspaceModule.ALERT_MODULE_ID),
                WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE, _workspaceModuleEP.getModule(MiniSiteWorkspaceModule.MINISITE_MODULE_ID),
                WorkspacesConstants.ABOUT_CONTENT_TYPE, _workspaceModuleEP.getModule(AboutWorkspaceModule.ABOUT_MODULE_ID)
        );
    }
    
    /**
     * The association of all project content types and associated modules
     * @return The association
     */
    public Map<String, WorkspaceModule> getProjectContentTypesAndModules()
    {
        return _contentTypesToModule;
    }
    
    /**
     * Retrieves all project profiles ids given the "profile list" configuration parameter
     * Profile order is guaranteed to be the same as in the configuration parameter.
     * @return the projects
     */
    public synchronized Set<String> getProfilesIds()
    {
        if (_profileIds == null)
        {
            String rawProjectProfileIds = StringUtils.defaultString(Config.getInstance().getValue("workspaces.profile.list"));
            _profileIds = Arrays.stream(StringUtils.split(rawProjectProfileIds, ',')).collect(Collectors.toSet());
        }
        return _profileIds;
    }
    
    /**
     * Retrieves all project profile given the "profile list" configuration parameter
     * Profile order is guaranteed to be the same as in the configuration parameter.
     * @return the projects
     */
    public Set<Profile> getProfiles()
    {
        // getProfiles(null) to get only shared profile
        Map<String, Profile> profileMap = _rightProfilesDao.getProfiles(null).stream().collect(Collectors.toMap(Profile::getId, item -> item));
        
        // Collect project profiles (unexisting entries are filtered out).
        return getProfilesIds().stream()
            .map(id -> 
            {
                Profile p = profileMap.get(id);
                
                // log null entries
                if (p == null)
                {
                    getLogger().warn("Could not find profile with id '{}'.", id);
                }
                
                return p;
            })
            .filter(Objects::nonNull)
            .collect(Collectors.toSet());
    }
    
    /**
     * Get the list of profiles and the list of modules available for rights affectation in the project.
     * @param projectName The project to check if the modules are activated. Can be null to ignore
     * @return the project rights data
     */
    @Callable
    public Map<String, Object> getProjectRightsData(String projectName)
    {
        // profiles
        List<Object> profiles = getProfiles()
                .stream()
                .map(this::_getProfileRightData)
                .collect(Collectors.toList());

        Project project = projectName != null ? _projectManager.getProject(projectName) : null;

        // modules
        Stream<Map<String, Object>> stream = _moduleManagerEP.getExtensionsIds().stream().map(moduleId -> _moduleManagerEP.getExtension(moduleId)).map(module -> _getModuleRightData(project, module));
        List<Object> modules = stream.filter(Objects::nonNull).collect(Collectors.toList());
        
        Map<String, Object> result = new HashMap<>();
        result.put("profiles", profiles);
        result.put("modules", modules);
        
        return result;
    }
    
    private Map<String, Object> _getProfileRightData(Profile profile)
    {
        Map<String, Object> data = new HashMap<>();
        data.put("id", profile.getId());
        data.put("label", profile.getLabel());
        return data;
    }
    
    private Map<String, Object> _getModuleRightData(Project project, WorkspaceModule module)
    {
        if (project != null && !_projectManager.isModuleActivated(project, module.getId()))
        {
            return null;
        }
        
        Map<String, Object> data = new HashMap<>();
        data.put("id", module.getId());
        data.put("label", module.getModuleTitle());
        return data;
    }    
    
    /**
     * Determines if the current user can view the members of a project
     * @param project the project
     * @return true if user can view members
     */
    public boolean canViewMembers(Project project)
    {
        return _rightManager.currentUserHasReadAccess(project);
    }
    
    /**
     * Determines if the current user has right to add member on project
     * @param project the project
     * @return true if user can add member
     */
    public boolean canAddMember(Project project)
    {
        MembersWorkspaceModule module = _moduleManagerEP.getModule(MembersWorkspaceModule.MEMBERS_MODULE_ID);
        if (module != null && _projectManager.isModuleActivated(project, module.getId()))
        {
            AmetysObject moduleRoot = module.getModuleRoot(project, false);
            return moduleRoot != null && _rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_ADD_MEMBER, moduleRoot) == RightResult.RIGHT_ALLOW;
        }
        
        return false;
    }
    
    /**
     * Determines if the current user has right to edit member on project
     * @param project the project
     * @return true if user can edit member
     */
    public boolean canEditMember(Project project)
    {
        return canAddMember(project);
    }
    
    /**
     * Determines if the current user has right to add member on project
     * @param project the project
     * @return true if user can remove member
     */
    public boolean canRemoveMember(Project project)
    {
        return _hasRightOnMembers(project, ProjectConstants.RIGHT_PROJECT_REMOVE_MEMBER);
    }
    
    private boolean _hasRightOnMembers(Project project, String rightId)
    {
        MembersWorkspaceModule module = _moduleManagerEP.getModule(MembersWorkspaceModule.MEMBERS_MODULE_ID);
        if (module != null && _projectManager.isModuleActivated(project, module.getId()))
        {
            AmetysObject moduleRoot = module.getModuleRoot(project, false);
            return moduleRoot != null && _rightManager.currentUserHasRight(rightId, moduleRoot) == RightResult.RIGHT_ALLOW;
        }
        
        return false;
    }
    
    /**
     * Determines if the current user has right to add tags on project
     * @param project the project
     * @return true if user can add tags
     */
    public boolean canAddTag(Project project)
    {
        return _hasRightOnTagsOrPlaces(project, ProjectConstants.RIGHT_PROJECT_ADD_TAG);
    }
    
    /**
     * Determines if the current user has right to remove tags on project
     * @param project the project
     * @return true if user can remove tags
     */
    public boolean canRemoveTag(Project project)
    {
        return _hasRightOnTagsOrPlaces(project, ProjectConstants.RIGHT_PROJECT_DELETE_TAG);
    }
    
    private boolean _hasRightOnTagsOrPlaces(Project project, String rightId)
    {
        WorkspaceModule module = _moduleManagerEP.getModule(CalendarWorkspaceModule.CALENDAR_MODULE_ID);
        if (module != null && _projectManager.isModuleActivated(project, module.getId()))
        {
            AmetysObject moduleRoot = module.getModuleRoot(project, false);
            if (moduleRoot != null && _rightManager.currentUserHasRight(rightId, moduleRoot) == RightResult.RIGHT_ALLOW)
            {
                return true;
            }
        }
        
        module = _moduleManagerEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
        if (module != null && _projectManager.isModuleActivated(project, module.getId()))
        {
            AmetysObject moduleRoot = module.getModuleRoot(project, false);
            if (moduleRoot != null && _rightManager.currentUserHasRight(rightId, moduleRoot) == RightResult.RIGHT_ALLOW)
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Test if the current user has the right on the project
     * @param rightId The right id
     * @param project The project
     * @return true if has right
     */
    public boolean hasRight(String rightId, Project project)
    {
        return _rightManager.hasRight(_currentUserProvider.getUser(), rightId, project) == RightResult.RIGHT_ALLOW;
    }
    
    /**
     * Test if the current user has a read access on the project
     * @param project The project
     * @return true if has read access
     */
    public boolean hasReadAccess(Project project)
    {
        return _rightManager.hasReadAccess(_currentUserProvider.getUser(), project);
    }
    
    /**
     * Test if the current user has the right on an explorer node
     * @param rightId The right id
     * @param explorerNode The explorer node
     * @return true if has right
     */
    public boolean hasRight(String rightId, ExplorerNode explorerNode)
    {
        return _rightManager.hasRight(_currentUserProvider.getUser(), rightId, explorerNode) == RightResult.RIGHT_ALLOW;
    }
}
