/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.accesscontroller;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ContentQueryHelper;
import org.ametys.cms.repository.ContentTypeExpression;
import org.ametys.cms.rights.ContentAccessController;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.RightsException;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.ChainedAmetysObjectIterable;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.categories.CategoryCMSTag;
import org.ametys.plugins.workspaces.keywords.KeywordCMSTag;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectManager.UnknownCatalogSiteException;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.PageQueryHelper;
import org.ametys.web.repository.page.Zone;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.rights.PageAccessController;

/**
 * Give the read right on a catalog news if the user is a member of a project that the thematic(s) or keywords match the news' tag
 * or if it the creator of the news
 */
public class CatalogNewsAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    private ProjectManager _projectManager;
    private ContentTypesHelper _cTypeHelper;
    private ProjectMemberManager _projectMemberManager;
    private AmetysObjectResolver _resolver;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _cTypeHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
    }
    
    public boolean isSupported(Object object)
    {
        try
        {
            return object instanceof WebContent content
                    && _cTypeHelper.isInstanceOf(content, WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID)
                    && StringUtils.equals(content.getSiteName(), _projectManager.getCatalogSiteName())
                    || object instanceof Page && _isCatalogNewsPage((Page) object);
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
            return false;
        }
    }
    
    private boolean _isCatalogNewsPage(Page page)
    {
        try
        {
            if (page.getSiteName().equals(_projectManager.getCatalogSiteName()) && PageType.CONTAINER == page.getType())
            {
                if (page.hasZone("default"))
                {
                    Zone defaultZone = page.getZone("default");
                    ZoneItem cZoneItem = defaultZone.getZoneItems().stream()
                            .filter(z -> z.getType() == ZoneType.CONTENT)
                            .filter(z -> _cTypeHelper.isInstanceOf(z.getContent(), WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID))
                            .findFirst()
                            .orElse(null);
                    
                    return cZoneItem != null;
                }
            }
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
        }
        
        return false;
    }
    
    private Content _getNewsContent(Object object)
    {
        if (object instanceof Content)
        {
            return (Content) object;
        }
        else
        {
            Page page = (Page) object;
            
            Optional< ? extends ZoneItem> zoneItem = page.getZone("default").getZoneItems().stream()
                    .filter(z -> z.getType() == ZoneType.CONTENT)
                    .filter(z -> _cTypeHelper.isInstanceOf(z.getContent(), WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID))
                    .findFirst();
            
            return (Content) zoneItem.map(zi -> zi.getContent()).orElse(null);
            
        }
    }
    
    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        Content content = _getNewsContent(object);
        
        if (content != null && content.getCreator().equals(user))
        {
            return AccessResult.USER_ALLOWED;
        }
        
        Set<String> tags = content != null  ? content.getTags() : Collections.EMPTY_SET;
        
        Set<String> categories = tags.stream()
            .filter(tag -> tag.startsWith(CategoryCMSTag.TAG_PREFIX))
            .map(tag -> tag.substring(CategoryCMSTag.TAG_PREFIX.length()))
            .collect(Collectors.toSet());
        
        Set<String> keywords = tags.stream()
                .filter(tag -> tag.startsWith(KeywordCMSTag.TAG_PREFIX))
                .map(tag -> tag.substring(KeywordCMSTag.TAG_PREFIX.length()))
                .collect(Collectors.toSet());
        
        if (!categories.isEmpty() || !keywords.isEmpty())
        {
            Map<Project, MemberType> userProjects = _projectManager.getUserProjects(user, categories, keywords);
            if (!userProjects.isEmpty())
            {
                return userProjects.values().contains(MemberType.USER) ? AccessResult.USER_ALLOWED : AccessResult.GROUP_ALLOWED;
            }
        }
        
        return AccessResult.UNKNOWN;
    }
    
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        Map<UserIdentity, AccessResult> accessResults = new HashMap<>();
        
        Content content = _getNewsContent(object);
        
        if (content != null)
        {
            accessResults.put(content.getCreator(), AccessResult.USER_ALLOWED);
        }
        
        Set<String> tags = content != null  ? content.getTags() : Collections.EMPTY_SET;
        
        Set<String> categories = tags.stream()
                .filter(tag -> tag.startsWith(CategoryCMSTag.TAG_PREFIX))
                .map(tag -> tag.substring(CategoryCMSTag.TAG_PREFIX.length()))
                .collect(Collectors.toSet());
            
        Set<String> keywords = tags.stream()
                    .filter(tag -> tag.startsWith(KeywordCMSTag.TAG_PREFIX))
                    .map(tag -> tag.substring(KeywordCMSTag.TAG_PREFIX.length()))
                    .collect(Collectors.toSet());
        
        if (!categories.isEmpty() || !keywords.isEmpty())
        {
            for (Project project : _projectManager.getProjects(categories, keywords, true))
            {
                Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, false);
                for (ProjectMember member : projectMembers)
                {
                    if (MemberType.USER == member.getType())
                    {
                        accessResults.put(member.getUser().getIdentity(), AccessResult.USER_ALLOWED);
                    }
                }
            }
        }
        
        return accessResults;
    }
    
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        Map<GroupIdentity, AccessResult> accessResults = new HashMap<>();
        
        Content content = _getNewsContent(object);
        Set<String> tags = content != null  ? content.getTags() : Collections.EMPTY_SET;
        
        Set<String> categories = tags.stream()
                .filter(tag -> tag.startsWith(CategoryCMSTag.TAG_PREFIX))
                .map(tag -> tag.substring(CategoryCMSTag.TAG_PREFIX.length()))
                .collect(Collectors.toSet());
            
        Set<String> keywords = tags.stream()
                    .filter(tag -> tag.startsWith(KeywordCMSTag.TAG_PREFIX))
                    .map(tag -> tag.substring(KeywordCMSTag.TAG_PREFIX.length()))
                    .collect(Collectors.toSet());
        
        if (!categories.isEmpty() || !keywords.isEmpty())
        {
            for (Project project : _projectManager.getProjects(categories, keywords, true))
            {
                Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, false);
                for (ProjectMember member : projectMembers)
                {
                    if (MemberType.GROUP == member.getType())
                    {
                        accessResults.put(member.getGroup().getIdentity(), AccessResult.GROUP_ALLOWED);
                    }
                }
            }
        }
        
        return accessResults;
    }

    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return Map.of();
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.UNKNOWN;
    }

    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    @Override
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object object)
    {
        Content content = _getNewsContent(object);
        
        if (content != null)
        {
            if (content.getCreator().equals(user))
            {
                return new AccessExplanation(
                        getId(),
                        AccessResult.USER_ALLOWED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOG_NEWS_ACCESS_CONTROLLER_CREATOR_EXPLANATION", Map.of("title", new I18nizableText(content.getTitle())))
                    );
            }
        
            Set<String> tags = content.getTags();
            
            Set<String> categories = tags.stream()
                .filter(tag -> tag.startsWith(CategoryCMSTag.TAG_PREFIX))
                .map(tag -> tag.substring(CategoryCMSTag.TAG_PREFIX.length()))
                .collect(Collectors.toSet());
            
            Set<String> keywords = tags.stream()
                    .filter(tag -> tag.startsWith(KeywordCMSTag.TAG_PREFIX))
                    .map(tag -> tag.substring(KeywordCMSTag.TAG_PREFIX.length()))
                    .collect(Collectors.toSet());
            
            if (!categories.isEmpty() || !keywords.isEmpty())
            {
                Map<Project, MemberType> userProjects = _projectManager.getUserProjects(user, categories, keywords);
                if (!userProjects.isEmpty())
                {
                    AccessResult permission = userProjects.values().contains(MemberType.USER) ? AccessResult.USER_ALLOWED : AccessResult.GROUP_ALLOWED;
                    List<String> projectTitle = new ArrayList<>();
                    for (Project project : userProjects.keySet())
                    {
                        projectTitle.add(project.getTitle());
                    }
                    
                    String key = "PLUGINS_WORKSPACES_CATALOG_NEWS_ACCESS_CONTROLLER_PROJECT_MEMBER_EXPLANATION";
                    if (projectTitle.size() > 1)
                    {
                        key += "_MULTIPLE";
                    }
                    
                    return new AccessExplanation(
                            getId(),
                            permission,
                            new I18nizableText("plugin.workspaces", key, Map.of("title", new I18nizableText(content.getTitle()), "projects", AccessExplanation.elementsToI18nizableText(projectTitle)))
                        );
                }
            }
        }
        
        return getStandardAccessExplanation(AccessResult.UNKNOWN, object);
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult permission, Object object)
    {
        if (AccessResult.UNKNOWN.equals(permission))
        {
            String title;
            if (object instanceof Page page)
            {
                title = page.getTitle();
            }
            else if (object instanceof Content content)
            {
                title = content.getTitle();
            }
            else
            {
                throw new RightsException("Unsupported object " + object.toString());
            }
            
            return new AccessExplanation(
                    getId(),
                    permission,
                    new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_CATALOG_NEWS_ACCESS_CONTROLLER_UNKNOWN_EXPLANATION", Map.of("title", new I18nizableText(title)))
                );
        }
        else
        {
            return super.getStandardAccessExplanation(permission, object);
        }
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        if (object instanceof Content)
        {
            return ContentAccessController.CONTENT_CONTEXT_CATEGORY;
        }
        else if (object instanceof Page)
        {
            return PageAccessController.PAGE_CONTEXT_CATEGORY;
        }
        throw new RightsException("Unsuppported context: " + object.toString());
    }

    @Override
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Content news)
        {
            return new I18nizableText(news.getTitle());
        }
        else if (object instanceof Page page)
        {
            return new I18nizableText(PageAccessController.getPageObjectLabel(page));
        }
        throw new RightsException("Unsuppported context: " + object.toString());
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        try
        {
            if (StringUtils.equals(siteName, _projectManager.getCatalogSiteName()))
            {
                Expression cTypeExpr = new ContentTypeExpression(Operator.EQ, WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID);
                
                Expression siteExpression = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, siteName);
                String contentQuery = ContentQueryHelper.getContentXPathQuery(new AndExpression(siteExpression, cTypeExpr));
                
                String pageQuery = PageQueryHelper.getPageXPathQuery(siteName, null, null, null, null);
                
                return new ChainedAmetysObjectIterable<>(List.of(_resolver.query(contentQuery), _resolver.query(pageQuery)));
            }
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
        }
        return List.of();
    }
}
