/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.accesscontroller;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.codec.binary.StringUtils;

import org.ametys.core.right.RightsException;
import org.ametys.plugins.explorer.ExplorerNode;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.rights.ResourceAccessController;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * AccessController handling access to module resources
 */
public class ModuleResourceAccessController extends ResourceAccessController implements Contextualizable
{
    private ProjectManager _projectManager;
    private SiteManager _siteManager;
    private Context _context;
    private WorkspaceModuleExtensionPoint _moduleEP;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _moduleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public boolean isSupported(Object object)
    {
        String path = null;
        if (object instanceof ExplorerNode node)
        {
            path = node.getPath();
        }
        else if (object instanceof Resource resource)
        {
            path = resource.getPath();
        }
            
        if (path != null)
        {
            return path.startsWith("/ametys:plugins/workspaces/projects/")
                    && path.contains("/ametys-internal:resources/");
        }
        else
        {
            return false;
        }
    }
    
    @Override
    protected boolean _isSupportedStoredContext(Object storedObject)
    {
        if (isSupported(storedObject))
        {
            // filter to retrieve only resource under the current project
            String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
            Site site = _siteManager.getSite(siteName);
            
            if (site != null)
            {
                List<Project> projects = _projectManager.getProjectsForSite(site);
                if (!projects.isEmpty())
                {
                    String path = ((AmetysObject) storedObject).getPath();
                    // object path contains current project
                    return path.contains("/ametys:plugins/workspaces/projects/" + projects.get(0).getName());
                }
            }
        }
        return false;
    }
    
    @Override
    protected Set< ? extends Object> _convertWorkspaceToRootRightContexts(Set<Object> workspacesContexts)
    {
        // Do not grant access to workspace
        return null;
    }
    
    @Override
    protected I18nizableText _getExplanationI18nText(PermissionDetails details)
    {
        // Workspace assignment are only positive and for user or group so we create custom explanation only for those
        return new I18nizableText("plugin.workspaces", _getAccessExplanationI18nKey("PLUGINS_WORKSPACES_MODULE_RESOURCE_ACCESS_CONTROLLER_", details), _getExplanationI18nParams(details));
    }
    
    @Override
    protected I18nizableText getObjectLabelForExplanation(Object object) throws RightsException
    {
        Map<String, I18nizableTextParameter> params = Map.of("path", _getObjectLabelWithPath(object));
        if (object instanceof Resource)
        {
            return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_MODULE_RESOURCE_ACCESS_CONTROLLER_RESOURCE_CONTEXT_EXPLANATION_LABEL", params);
        }
        else if (object instanceof ExplorerNode node)
        {
            AmetysObject parent = node.getParent();
            if (StringUtils.equals(parent.getName(), "ametys-internal:resources"))
            {
                return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_MODULE_RESOURCE_ACCESS_CONTROLLER_MODULE_CONTEXT_EXPLANATION_LABEL", params);
            }
            else
            {
                return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_MODULE_RESOURCE_ACCESS_CONTROLLER_FOLDER_CONTEXT_EXPLANATION_LABEL", params);
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }

    @Override
    public I18nizableText getObjectLabel(Object object) throws RightsException
    {
        // Prefix the path with "Module "
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_MODULE_RESOURCE_ACCESS_CONTROLLER_MODULE_CONTEXT_LABEL",
            Map.of("path", _getObjectLabelWithPath(object)));
    }
    
    /**
     * Generate a label like ModuleName > FolderName > resourceName
     */
    private I18nizableText _getObjectLabelWithPath(Object object)
    {
        if (object instanceof Resource resource)
        {
            return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_PATH",
                    Map.of("parent", getObjectLabel(resource.getParent()),
                           "child", new I18nizableText(resource.getName()))
                    );
        }
        else if (object instanceof ExplorerNode node)
        {
            AmetysObject parent = node.getParent();
            if (StringUtils.equals(parent.getName(), "ametys-internal:resources"))
            {
                
                return _moduleEP.getModuleByName(node.getName()).getModuleTitle();
            }
            else
            {
                return new I18nizableText("plugin.explorer", "PLUGINS_EXPLORER_RESOURCE_ACCESS_CONTROLLER_PATH",
                        Map.of("parent", getObjectLabel(node.getParent()),
                               "child", new I18nizableText(node.getName()))
                        );
            }
        }
        throw new RightsException("Unsupported object " + object.toString());
    }
    
    @Override
    public I18nizableText getObjectCategory(Object object)
    {
        return ProjectAccessController.WORKSPACE_CONTEXT_CATEGORY;
    }

}
