package org.ametys.plugins.workspaces.project.rights.accesscontroller;
/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.commons.codec.binary.StringUtils;

import org.ametys.core.group.GroupIdentity;
import org.ametys.core.right.AccessController;
import org.ametys.core.right.AccessExplanation;
import org.ametys.core.right.AllowedUsers;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightsException;
import org.ametys.core.right.RightsExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.impl.right.AbstractRightBasedAccessController;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectManager.UnknownCatalogSiteException;
import org.ametys.plugins.workspaces.report.ReportHelper;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.SiteAwareAmetysObject;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.rights.PageAccessController;

/**
 * This {@link AccessController} handles read access to the report page.
 */
public class ReportsPageAccessController extends AbstractRightBasedAccessController implements Serviceable
{
    /** The reports helper */
    protected ReportHelper _reportHelper;
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The right extension point */
    protected RightsExtensionPoint _rightEP;
    private RightManager _rightManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _reportHelper = (ReportHelper) manager.lookup(ReportHelper.ROLE);
        _resolver = (AmetysObjectResolver) manager .lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _rightEP = (RightsExtensionPoint) manager.lookup(RightsExtensionPoint.ROLE);
    }
    
    public boolean isSupported(Object object)
    {
        return object instanceof Page && isReportPage((Page) object);
    }

    public AccessResult getReadAccessPermission(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        if (_reportHelper.hasSuperRight(user) || _projectManager.isManager(user))
        {
            return AccessResult.USER_ALLOWED;
        }
        return AccessResult.USER_DENIED;
    }
    
    @Override
    public AccessExplanation explainReadAccessPermission(UserIdentity user, Set<GroupIdentity> groups, Object object)
    {
        AccessResult accessResult = getReadAccessPermission(user, groups, object);
        // handle the user allowed here to be able to provide more information than
        // in the getStandardExplanation method
        if (accessResult == AccessResult.USER_ALLOWED)
        {
            if (_reportHelper.hasSuperRight(user))
            {
                if (_projectManager.isManager(user))
                {
                    return new AccessExplanation(
                            getId(),
                            AccessResult.USER_ALLOWED,
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_HAS_RIGHT_AND_MANAGER_EXPLANATION",
                                    Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                            );
                }
                else
                {
                    return new AccessExplanation(
                            getId(),
                            AccessResult.USER_ALLOWED,
                            new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_HAS_RIGHT_EXPLANATION",
                                    Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                            );
                }
            }
            else if (_projectManager.isManager(user))
            {
                return new AccessExplanation(
                        getId(),
                        AccessResult.USER_ALLOWED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_MANAGER_EXPLANATION",
                                Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                        );
            }
        }
        return getStandardAccessExplanation(accessResult, object);
    }
    
    public AccessResult getReadAccessPermissionForAnonymous(Object object)
    {
        return AccessResult.ANONYMOUS_DENIED;
    }
    
    public AccessResult getReadAccessPermissionForAnyConnectedUser(Object object)
    {
        return AccessResult.ANY_CONNECTED_DENIED;
    }
    
    public Map<UserIdentity, AccessResult> getReadAccessPermissionByUser(Object object)
    {
        Map<UserIdentity, AccessResult> results = new HashMap<>();
        
        AllowedUsers allowedUsers = _rightManager.getAllowedUsers(ReportHelper.REPORT_ALL_RIGHT_ID, "/cms");
        
        for (UserIdentity user : allowedUsers.getAllowedUsers())
        {
            results.put(user, AccessResult.USER_ALLOWED);
        }
        
        Set<UserIdentity> managers = _projectManager.getManagers();
        for (UserIdentity manager : managers)
        {
            results.put(manager, AccessResult.USER_ALLOWED);
        }
        
        return results;
    }
    
    public Map<GroupIdentity, AccessResult> getReadAccessPermissionByGroup(Object object)
    {
        Map<GroupIdentity, AccessResult> results = new HashMap<>();
        
        AllowedUsers allowedUsers = _rightManager.getAllowedUsers(ReportHelper.REPORT_ALL_RIGHT_ID, "/cms");
        
        for (GroupIdentity group : allowedUsers.getAllowedGroups())
        {
            results.put(group, AccessResult.GROUP_ALLOWED);
        }
        
        return results;
    }
    
    public AccessResult getPermission(UserIdentity user, Set<GroupIdentity> userGroups, String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public Map<String, AccessResult> getPermissionByRight(UserIdentity user, Set<GroupIdentity> userGroups, Object object)
    {
        return Map.of();
    }

    public AccessResult getPermissionForAnonymous(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    public AccessResult getPermissionForAnyConnectedUser(String rightId, Object object)
    {
        return AccessResult.UNKNOWN;
    }
    
    @Override
    public AccessExplanation getStandardAccessExplanation(AccessResult accessResult, Object object)
    {
        switch (accessResult)
        {
            case USER_DENIED:
                return new AccessExplanation(
                        getId(),
                        AccessResult.USER_DENIED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_DENIED_EXPLANATION",
                                Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                        );
            case ANONYMOUS_DENIED:
                return new AccessExplanation(
                        getId(),
                        AccessResult.ANONYMOUS_DENIED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_DENIED_EXPLANATION",
                                Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                        );
            case ANY_CONNECTED_DENIED:
                return new AccessExplanation(
                        getId(),
                        AccessResult.ANY_CONNECTED_DENIED,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_DENIED_EXPLANATION",
                                Map.of("right", _rightEP.getExtension(ReportHelper.REPORT_ALL_RIGHT_ID).getLabel(), "page", getObjectLabel(object)))
                        );
            case UNKNOWN:
                return new AccessExplanation(
                        getId(),
                        AccessResult.UNKNOWN,
                        new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_REPORT_PAGE_ACCESS_CONTROLLER_UNKNOWN_EXPLANATION",
                                Map.of("page", getObjectLabel(object)))
                        );
            default:
                return super.getStandardAccessExplanation(accessResult, object);
        }
    }
    
    public Map<UserIdentity, AccessResult> getPermissionByUser(String rightId, Object object)
    {
        return Map.of();
    }

    public Map<GroupIdentity, AccessResult> getPermissionByGroup(String rightId, Object object)
    {
        return Map.of();
    }

    public boolean hasUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups, String rightId)
    {
        return false;
    }

    public boolean hasUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts, UserIdentity user, Set<GroupIdentity> userGroups)
    {
        return false;
    }

    public boolean hasAnonymousAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnonymousAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyPermissionOnWorkspace(Set<Object> workspacesContexts, String rightId)
    {
        return false;
    }

    public boolean hasAnyConnectedUserAnyReadAccessPermissionOnWorkspace(Set<Object> workspacesContexts)
    {
        return false;
    }
    
    /**
     * Determines if the page is a report page
     * @param page the page
     * @return true if the page is a report page
     */
    protected boolean isReportPage(Page page)
    {
        try
        {
            if (page.getSiteName().equals(_projectManager.getCatalogSiteName()) && PageType.CONTAINER == page.getType() && page.hasZone("default"))
            {
                try (AmetysObjectIterable< ? extends ZoneItem> zoneItems = page.getZone("default").getZoneItems())
                {
                    ZoneItem cZoneItem = zoneItems.stream()
                            .filter(z -> z.getType() == ZoneType.SERVICE)
                            .filter(z -> z.getServiceId().equals(ReportHelper.REPORT_SERVICE_ID))
                            .findFirst()
                            .orElse(null);
                    
                    return cZoneItem != null;
                }
            }
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
        }
        
        return false;
    }
    
    public I18nizableText getObjectLabel(Object object)
    {
        if (object instanceof Page page)
        {
            return new I18nizableText(PageAccessController.getPageObjectLabel(page));
        }
        throw new RightsException("Unsupported context: " + object.toString());
    }

    public I18nizableText getObjectCategory(Object object)
    {
        return PageAccessController.PAGE_CONTEXT_CATEGORY;
    }

    @Override
    protected Iterable< ? extends Object> getHandledObjects(UserIdentity identity, Set<GroupIdentity> groups)
    {
        String siteName = WebHelper.getSiteName(ContextHelper.getRequest(_context));
        
        try
        {
            if (StringUtils.equals(siteName, _projectManager.getCatalogSiteName()))
            {
                Expression siteExpr = new StringExpression(SiteAwareAmetysObject.METADATA_SITE, Operator.EQ, siteName);
                Expression reportService = () -> "ametys-internal:zones/default/ametys-internal:zoneItems/ametys:zoneItem/@ametys-internal:service = '" + ReportHelper.REPORT_SERVICE_ID + "'";
                
                String query = QueryHelper.getXPathQuery(null, "ametys:page", new AndExpression(siteExpr, reportService));
                
                return _resolver.query(query);
            }
        }
        catch (UnknownCatalogSiteException e)
        {
            // Ignore the no catalog sitename exception
        }
        
        return List.of();
    }
}
