/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.time.ZonedDateTime;
import java.util.Collection;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SearchResults;
import org.ametys.cms.search.Sort.Order;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProviderExtensionPoint;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

/**
 * Abstract generator for search module of type content
 *
 */
public abstract class AbstractContentSolrSearchModuleGenerator extends AbstractSolrSearchModuleGenerator
{
    /** The content searcher factory */
    protected ContentSearcherFactory _contentSearcherFactory;
    /** The content types helper */
    protected ContentTypesHelper _cTypesHelper;
    /** The tag provider */
    protected TagProviderExtensionPoint _tagProviderEP;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _contentSearcherFactory = (ContentSearcherFactory) manager.lookup(ContentSearcherFactory.ROLE);
        _cTypesHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
        _tagProviderEP = (TagProviderExtensionPoint) smanager.lookup(TagProviderExtensionPoint.ROLE);
    }
    
    @Override
    protected void saxHit(AmetysObject object, String lang) throws Exception
    {
        if (object instanceof Content)
        {
            Content content = (Content) object;
            View view = _cTypesHelper.getView("search", content.getTypes(), content.getMixinTypes());
            if (view == null)
            {
                // fallback to main view
                view = _cTypesHelper.getView("main", content.getTypes(), content.getMixinTypes());
            }
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", content.getId());
            attrs.addCDATAAttribute("title", content.getTitle());
            
            attrs.addCDATAAttribute("createdAt", DateUtils.zonedDateTimeToString(content.getCreationDate()));
            attrs.addCDATAAttribute("creator", UserIdentity.userIdentityToString(content.getCreator()));
            attrs.addCDATAAttribute("lastModifiedAt", DateUtils.zonedDateTimeToString(content.getLastModified()));
            ZonedDateTime lastValidatedAt = content.getLastValidationDate();
            if (lastValidatedAt != null)
            {
                attrs.addCDATAAttribute("lastValidatedAt", DateUtils.zonedDateTimeToString(lastValidatedAt));
            }
            
            attrs.addCDATAAttribute("lastContributor", UserIdentity.userIdentityToString(content.getLastContributor()));
            
            XMLUtils.startElement(contentHandler, "hit", attrs);
            
            XMLUtils.startElement(contentHandler, "attributes");
            content.dataToSAX(contentHandler, view, DataContext.newInstance().withLocale(new Locale(lang)).withEmptyValues(false));
            XMLUtils.endElement(contentHandler, "attributes");
            
            saxAdditionalInformation(content);
            
            XMLUtils.endElement(contentHandler, "hit");
        }
        else
        {
            getLogger().error("Object '" + object.getId() + "' is not a Content.");
        }
    }
    
    /**
     * SAX additional information on content
     * @param content the content
     * @throws SAXException if an error occurs while saxing
     */
    protected void saxAdditionalInformation(Content content) throws SAXException
    {
        saxPage(content);
        saxTags(content);
    }
    
    /**
     * SAX page content
     * @param content the content
     * @throws SAXException  if an error occured while saxing
     */
    protected void saxPage(Content content) throws SAXException
    {
        if (content instanceof WebContent)
        {
            Collection<Page> referencingPages = ((WebContent) content).getReferencingPages();
            if (!referencingPages.isEmpty())
            {
                XMLUtils.createElement(contentHandler, "page", referencingPages.iterator().next().getId());
            }
        }
    }
    
    /**
     * SAX content's tags
     * @param content the content
     * @throws SAXException  if an error occured while saxing
     */
    protected void saxTags(Content content) throws SAXException
    {
        Set<String> tags = content.getTags();
        if (!tags.isEmpty())
        {
            XMLUtils.startElement(contentHandler, "tags");
            
            for (String tagName : tags)
            {
                Map<String, Object> contextParameters = new HashMap<>();
                if (content instanceof WebContent)
                {
                    contextParameters.put("siteName", ((WebContent) content).getSiteName());
                }
                Tag tag = _tagProviderEP.getTag(tagName, contextParameters);

                if (tag != null)
                {
                    AttributesImpl attrs = new AttributesImpl();
                    if (tag.getParentName() != null)
                    {
                        attrs.addCDATAAttribute("parent", tag.getParentName());
                    }
                    
                    XMLUtils.startElement(contentHandler, tagName, attrs);
                    tag.getTitle().toSAX(contentHandler);
                    XMLUtils.endElement(contentHandler, tagName);
                }
            }
            
            XMLUtils.endElement(contentHandler, "tags");
        }
    }
    
    @Override
    protected SearchResults<Content> getSearchResults(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception
    {
        SimpleContentSearcher searcher = getSearcher();
        
        Query query = getQuery(siteName, lang, textfield, request);
        
        return searcher
                .withLimits(offset, limit)
                .addSort(getSortFieldName(), getSortOrder())
                .searchWithFacets(query);
    }
    
    /**
     * Get the searcher
     * @return the searcher
     */
    protected abstract SimpleContentSearcher getSearcher();

    /**
     * The sort field name
     * @return the sort field name
     */
    protected abstract String getSortFieldName();
    
    /**
     * The sort order
     * @return the sort order
     */
    protected abstract Order getSortOrder();

    /**
     * Get the search query
     * @param siteName the current site name
     * @param lang the current language
     * @param textfield the search input
     * @param request the request
     * @return the search query
     */
    protected abstract Query getQuery(String siteName, String lang, String textfield, Request request);
}
