/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.query;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;

import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;

/**
 * Represents a {@link Query} testing the keywords property of a project resource.
 */
public class KeywordQuery implements Query
{
    private Operator _operator;
    private List<String> _keywords;
    
    /**
     * Build a KeywordQuery to test if the keywords property is equal to one of the given keywords
     * @param keywords the keywords.
     */
    public KeywordQuery(String... keywords)
    {
        this(Operator.EQ, keywords);
    }
    
    /**
     * Build a KeywordQuery to test if the keywords property is equal to one of the given keywords
     * @param keywords the keywords.
     */
    public KeywordQuery(Collection<String> keywords)
    {
        this(Operator.EQ, keywords);
    }
    
    /**
     * Build a KeywordQuery to test if the keywords property is equal or different to one of the given keywords
     * @param operator the operator (equal ot not-equal)
     * @param keywords the keywords.
     */
    public KeywordQuery(Operator operator, String... keywords)
    {
        this(operator, Arrays.asList(keywords));
    }
    
    /**
     * Build a KeywordQuery.
     * @param operator the operator.
     * @param keywords the keywords.
     */
    public KeywordQuery(Operator operator, Collection<String> keywords)
    {
        if (Operator.EQ != operator && Operator.NE != operator)
        {
            throw new IllegalArgumentException("Test operator '" + operator + "' is unknown for test's expression.");
        }
        
        _operator = operator;
        _keywords = new ArrayList<>(keywords);
    }
    
    @Override
    public String build() throws QuerySyntaxException
    {
        int count = _keywords.size();
        
        StringBuilder sb = new StringBuilder();
        
        if (_operator == Operator.NE)
        {
            sb.append('-');
        }
        if (count > 1)
        {
            sb.append('(');
        }
        
        boolean first = true;
        for (String keyword : _keywords)
        {
            if (!first)
            {
                sb.append(" OR ");
            }
            sb.append(SolrWorkspacesConstants.KEYWORDS).append(':').append(keyword);
            first = false;
        }
        
        if (count > 1)
        {
            sb.append(')');
        }
        
        return sb.toString();
    }

}
