/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tags;

import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.commons.lang.StringUtils;

import org.ametys.cms.tag.Tag;
import org.ametys.cms.tag.TagProvider;
import org.ametys.cms.tag.jcr.AbstractJCRTagsDAO;
import org.ametys.cms.tag.jcr.JCRTag;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.workspaces.project.ProjectConstants;

/**
 * Component for operations on JCR tags
 */
public class ProjectJCRTagsDAO extends AbstractJCRTagsDAO
{
    /** The Avalon role */
    public static final String ROLE = ProjectJCRTagsDAO.class.getName();
    
    /** The tag provider extension point */
    protected ProjectTagProviderExtensionPoint _tagProviderExtPt;
    
    /** The rights manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _tagProviderExtPt = (ProjectTagProviderExtensionPoint) serviceManager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }
    
    @Override
    public ModifiableTraversableAmetysObject _getTagRootObject (String tagProviderId, Map<String, Object> contextualParameters) throws RepositoryException
    {
        ProjectJCRTagProvider provider = (ProjectJCRTagProvider) _tagProviderExtPt.getExtension(tagProviderId);
        return provider.getRootNode(contextualParameters);
    }
    
    @Override
    protected void _checkUserRight() throws IllegalStateException
    {
        if (_rightManager.currentUserHasRight(ProjectConstants.RIGHT_PROJECT_HANDLE_TAGS, "/admin") != RightResult.RIGHT_ALLOW)
        {
            UserIdentity user = _currentUserProvider.getUser();
            throw new IllegalStateException("User '" + user + "' tried to access a privileged feature without convenient right. Should have right '" + ProjectConstants.RIGHT_PROJECT_HANDLE_TAGS + "' on context '/admin'");
        }
    }

    @Override
    protected Tag _getTagFromName(String name, Map<String, Object> contextualParameters)
    {
        return _tagProviderExtPt.getTag(name, contextualParameters);
    }
    
    @Override
    protected JCRTag _createJCRTag(String parentId, String name, String title, String description, Map<String, Object> otherParameters, Map<String, Object> contextualParameters) throws RepositoryException
    {
        ModifiableTraversableAmetysObject parent = null;
        if (StringUtils.isEmpty(parentId))
        {
            parent = _getTagRootObject(ProjectJCRTagProvider.class.getName(), contextualParameters);
        }
        else
        {
            parent = _resolver.resolveById(parentId);
        }
        
        ProjectTagJCR jcrTag = parent.createChild(name, ProjectTagFactory.TAG_NODETYPE);
        jcrTag.setTitle(title);
        jcrTag.setDescription(description);
        
        parent.saveChanges();
        
        return jcrTag;
    }
    
    @Override
    protected JCRTag _updateJCRTag(String tagId, String title, String description, Map<String, Object> otherParameters) throws UnknownAmetysObjectException
    {
        ProjectTagJCR jcrTag = _resolver.resolveById(tagId);
        jcrTag.setTitle(title);
        jcrTag.setDescription(description);
        
        jcrTag.saveChanges();
        
        return jcrTag;
    }

    @Override
    protected Set<TagProvider< ? extends Tag>> _getTagProviders()
    {
        Set<TagProvider<? extends Tag>> providers = new HashSet<>();
        
        Set<String> ids = _tagProviderExtPt.getExtensionsIds();
        for (String id : ids)
        {
            TagProvider<? extends Tag> provider = _tagProviderExtPt.getExtension(id);
            providers.add(provider);
        }
        
        return providers;
    }
    
    @Override
    @Callable(rights = ProjectConstants.RIGHT_PROJECT_HANDLE_TAGS, context = "/admin")
    public Map<String, Object> getTagRootNode(String tagProviderId, Map<String, Object> contextualParameters) throws ProcessingException
    {
        return super.getTagRootNode(tagProviderId, contextualParameters);
    }
    
    @Override
    @Callable(rights = ProjectConstants.RIGHT_PROJECT_HANDLE_TAGS, context = "/admin")
    public Map<String, Object> getTag(String tagId)
    {
        return super.getTag(tagId);
    }
}
