/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks;

import java.util.HashMap;
import java.util.Map;

import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.AbstractWorkspaceDAO;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * DAO for interacting with tasks of a project
 */
public abstract class AbstractWorkspaceTaskDAO extends AbstractWorkspaceDAO
{
    /** Rights to handle task */
    public static final String RIGHTS_HANDLE_TASK = "Plugin_Workspace_Handle_Task";
    
    /** Rights to delete task */
    public static final String RIGHTS_DELETE_TASK = "Plugin_Workspace_Delete_Task";
    
    /** Rights to comment task */
    public static final String RIGHTS_COMMENT_TASK = "Plugin_Workspace_Comment_Task";
    
    /** Rights to handle task list */
    public static final String RIGHTS_HANDLE_TASK_LIST = "Plugin_Workspace_Handle_Tasks_List";
    
    /** Rights to delete task  list */
    public static final String RIGHTS_DELETE_TASK_LIST = "Plugin_Workspace_Delete_Tasks_List";
    
    /** Rights to delete task comment */
    public static final String RIGHTS_DELETE_COMMENT_TASK = "Plugin_Workspace_Delete_Comment_Task";
    
    /**
     * Get the module root of task
     * @param projectName the project name
     * @return the module root
     */
    protected ModifiableResourceCollection _getModuleRoot(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        
        TasksWorkspaceModule taskModule = _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        return taskModule.getModuleRoot(project, false);
    }
    
    /**
     * Get the tasks lists root
     * @param projectName the project name
     * @return the tasks lists root
     */
    protected ModifiableTraversableAmetysObject _getTasksListsRoot(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        
        TasksWorkspaceModule taskModule = _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        return taskModule.getTasksListsRoot(project, true);
    }
    
    /**
     * Get the tasks root
     * @param projectName the project name
     * @return the tasks root
     */
    protected ModifiableTraversableAmetysObject _getTasksRoot(String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        
        TasksWorkspaceModule taskModule = _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        return taskModule.getTasksRoot(project, true);
    }
    
    /**
     * Get user rights from project name
     * @param projectName the project name
     * @return the user rights
     */
    @Callable
    public Map<String, Object> getUserRights(String projectName)
    {
        Map<String, Object> results = new HashMap<>();
        
        ModifiableTraversableAmetysObject tasksRoot = _getTasksRoot(projectName);
        ModifiableTraversableAmetysObject tasksListsRoot = _getTasksListsRoot(projectName);
        
        UserIdentity user = _currentUserProvider.getUser();
        results.put("canWriteTaskList", _rightManager.hasRight(user, RIGHTS_HANDLE_TASK_LIST, tasksListsRoot) == RightResult.RIGHT_ALLOW);
        results.put("canDeleteTaskList", _rightManager.hasRight(user, RIGHTS_DELETE_TASK_LIST, tasksListsRoot) == RightResult.RIGHT_ALLOW);
        results.put("canWriteTask", _rightManager.hasRight(user, RIGHTS_HANDLE_TASK, tasksRoot) == RightResult.RIGHT_ALLOW);
        results.put("canDeleteTask", _rightManager.hasRight(user, RIGHTS_DELETE_TASK, tasksRoot) == RightResult.RIGHT_ALLOW);
        results.put("canCommentTask", _rightManager.hasRight(user, RIGHTS_COMMENT_TASK, tasksRoot) == RightResult.RIGHT_ALLOW);
        results.put("canDeleteCommentTask", _rightManager.hasRight(user, RIGHTS_DELETE_COMMENT_TASK, tasksRoot) == RightResult.RIGHT_ALLOW);
        
        return results;
    }

}
