/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.servlet.multipart.Part;
import org.apache.commons.lang.IllegalClassException;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.observation.Event;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.html.HTMLTransformer;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.plugins.workspaces.tasks.Task.CheckItem;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTask;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTaskFactory;
import org.ametys.plugins.workspaces.tasks.json.TaskJSONHelper;

/**
 * DAO for interacting with tasks of a project
 */
public class WorkspaceTaskDAO extends AbstractWorkspaceTaskDAO
{
    /** The Avalon role */
    public static final String ROLE = WorkspaceTaskDAO.class.getName();
    
    /** The HTML transformer */
    protected HTMLTransformer _htmlTransformer;
    
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    
    /** The tag provider extension point */
    protected ProjectTagProviderExtensionPoint _tagProviderExtPt;
        
    /** The task JSON helper */
    protected TaskJSONHelper _taskJSONHelper;
    
    /** The task list DAO */
    protected WorkspaceTasksListDAO _workspaceTasksListDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _htmlTransformer = (HTMLTransformer) manager.lookup(HTMLTransformer.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _tagProviderExtPt = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _taskJSONHelper = (TaskJSONHelper) manager.lookup(TaskJSONHelper.ROLE);
        _workspaceTasksListDAO = (WorkspaceTasksListDAO) manager.lookup(WorkspaceTasksListDAO.ROLE);
    }

    /**
     * Get the tasks from project
     * @return the list of tasks
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public List<Map<String, Object>> getTasks() throws IllegalAccessException
    {
        String projectName = _getProjectName();
        
        ModifiableResourceCollection moduleRoot = _getModuleRoot(projectName);
        if (!_rightManager.currentUserHasReadAccess(moduleRoot))
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to get tasks without reader right");
        }
        
        List<Map<String, Object>> tasksInfo = new ArrayList<>();
        Project project = _projectManager.getProject(projectName);
        for (Task task : getProjectTasks(project))
        {
            tasksInfo.add(_taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName()));
        }
        
        return tasksInfo;
    }
    
    /**
     * Add a new task to the tasks list
     * @param tasksListId the tasks list id
     * @param parameters The task parameters
     * @param newFiles the files to add
     * @param newFileNames the file names to add 
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> addTask(String tasksListId, Map<String, Object> parameters, List<Part> newFiles, List<String> newFileNames) throws IllegalAccessException
    {
        String projectName = _getProjectName();
        
        if (StringUtils.isBlank(tasksListId))
        {
            throw new IllegalArgumentException("Tasks list id is mandatory to create a new task");
        }
        
        ModifiableTraversableAmetysObject tasksRoot = _getTasksRoot(projectName);
        
        // Check user right
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_HANDLE_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to add task without convenient right [" + RIGHTS_HANDLE_TASK + "]");
        }

        int index = 1;
        String name = "task-1";
        while (tasksRoot.hasChild(name))
        {
            index++;
            name = "task-" + index;
        }
        
        JCRTask task = (JCRTask) tasksRoot.createChild(name, JCRTaskFactory.TASK_NODETYPE);
        task.setTasksListId(tasksListId);
        task.setPosition(Long.valueOf(_workspaceTasksListDAO.getChildTask(tasksListId).size()));
        
        ZonedDateTime now = ZonedDateTime.now();
        task.setCreationDate(now);
        task.setLastModified(now);
        task.setAuthor(_currentUserProvider.getUser());

        Map<String, Object> attributesResults = _setTaskAttributes(task, parameters, newFiles, newFileNames, new ArrayList<>());
        
        tasksRoot.saveChanges();
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_TASK, task);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, task.getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_CREATED, _currentUserProvider.getUser(), eventParams));

        Map<String, Object> results = new HashMap<>();
        results.put("task", _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName()));
        results.putAll(attributesResults);
        return results;
    }
    
    /**
     * Edit a task
     * @param taskId The id of the task to edit
     * @param parameters The JS parameters
     * @param newFiles the new file to add
     * @param newFileNames the file names to add
     * @param deleteFiles the file to delete
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> editTask(String taskId, Map<String, Object> parameters, List<Part> newFiles, List<String> newFileNames, List<String> deleteFiles) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof JCRTask))
        {
            throw new IllegalClassException(JCRTask.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_HANDLE_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to edit task without convenient right [" + RIGHTS_HANDLE_TASK + "]");
        }
        
        JCRTask task = (JCRTask) object;
        Map<String, Object> attributesResults = _setTaskAttributes(task, parameters, newFiles, newFileNames, deleteFiles);
        
        ZonedDateTime now = ZonedDateTime.now();
        task.setLastModified(now);
            
        task.saveChanges();
            
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_TASK, task);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, taskId);
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_UPDATED, _currentUserProvider.getUser(), eventParams));
        
        // Closed status has changed
        if (attributesResults.containsKey("isClosed"))
        {
            _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_CLOSED_STATUS_CHANGED, _currentUserProvider.getUser(), eventParams));
        }

        // Assigments have changed
        if (attributesResults.containsKey("changedAssignments"))
        {
            _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_ASSIGNED, _currentUserProvider.getUser(), eventParams));
        }
        
         
        Map<String, Object> results = new HashMap<>();
        results.put("task", _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName()));
        results.putAll(attributesResults);
        return results;
    }
    
    /**
     * Move task to new position
     * @param tasksListId the tasks list id
     * @param taskId the task id to move
     * @param newPosition the new position
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> moveTask(String tasksListId, String taskId, long newPosition) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_HANDLE_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to move task without convenient right [" + RIGHTS_HANDLE_TASK + "]");
        }
        
        Task task = (Task) object;
        if (tasksListId != task.getTaskListId())
        {
            List<Task> childTasks = _workspaceTasksListDAO.getChildTask(task.getTaskListId());
            long position = 0;
            for (Task childTask : childTasks)
            {
                if (!childTask.getId().equals(taskId))
                {
                    childTask.setPosition(position);
                    position++;
                }
            }
        }
        
        task.setTasksListId(tasksListId);
        List<Task> childTasks = _workspaceTasksListDAO.getChildTask(tasksListId);
        int size = childTasks.size();
        if (newPosition > size)
        {
            throw new IllegalArgumentException("New position (" + newPosition + ") can't be greater than tasks child size (" + size + ")");
        }
        
        long position = 0;
        task.setPosition(newPosition);
        for (Task childTask : childTasks)
        {
            if (position == newPosition)
            {
                position++;
            }
            
            if (childTask.getId().equals(taskId))
            {
                childTask.setPosition(newPosition);
            }
            else
            {
                childTask.setPosition(position);
                position++;
            }
        }
        
        tasksRoot.saveChanges();
        
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Remove a task
     * @param taskId the task id to remove
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> deleteTask(String taskId) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_DELETE_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to delete task without convenient right [" + RIGHTS_DELETE_TASK + "]");
        }
        
        Map<String, Object> results = new HashMap<>();
        Task jcrTask = (Task) object;
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_TASK, jcrTask);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, taskId);
        _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_DELETING, _currentUserProvider.getUser(), eventParams));
        
        String tasksListId = jcrTask.getTaskListId();
        jcrTask.remove();
        
        // Reorder tasks position
        long position = 0;
        for (Task childTask : _workspaceTasksListDAO.getChildTask(tasksListId))
        {
            childTask.setPosition(position);
            position++;
        }
        
        tasksRoot.saveChanges();

        eventParams = new HashMap<>();
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, taskId);
        _observationManager.notify(new Event(ObservationConstants.EVENT_TASK_DELETED, _currentUserProvider.getUser(), eventParams));

        return results;
    }
    
    /**
     * Comment a task
     * @param taskId the task id
     * @param commentText the comment text
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> commentTask(String taskId, String commentText) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COMMENT_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to comment task without convenient right [" + RIGHTS_COMMENT_TASK + "]");
        }
        
        Task task = (Task) object;
        
        createComment(task, commentText, tasksRoot);
        
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Edit a task comment
     * @param taskId the task id
     * @param commentId the comment Id
     * @param commentText the comment text
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> editCommentTask(String taskId, String commentId, String commentText) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COMMENT_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to edit task without convenient right [" + RIGHTS_COMMENT_TASK + "]");
        }
        
        Task task = (Task) object;

        editComment(task, commentId, commentText, tasksRoot);
        
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Answer to a task's comment
     * @param taskId the task id
     * @param commentId the comment id
     * @param commentText the comment text
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> answerCommentTask(String taskId, String commentId, String commentText) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COMMENT_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to comment task without convenient right [" + RIGHTS_COMMENT_TASK + "]");
        }
        
        Task task = (Task) object;

        answerComment(task, commentId, commentText, tasksRoot);
        
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Delete a task's comment
     * @param taskId the task id
     * @param commentId the comment id
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> deleteCommentTask(String taskId, String commentId) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = _userManager.getUser(userIdentity);
        
        Task task = (Task) object;
        Comment comment = task.getComment(commentId);
        String authorEmail = comment.getAuthorEmail();
        if (!authorEmail.equals(user.getEmail()))
        {
            if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COMMENT_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
            {
                throw new IllegalAccessException("User '" + userIdentity + "' tried to delete an other user's comment task");
            }
        }
        
        deleteComment(task, commentId, tasksRoot);
              
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Like or unlike a task's comment
     * @param taskId the task id
     * @param commentId the comment id
     * @param liked true if the comment is liked, otherwise the comment is unliked
     * @return The task data
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    @Callable
    public Map<String, Object> likeOrUnlikeCommentTask(String taskId, String commentId, Boolean liked) throws IllegalAccessException
    {
        AmetysObject object = _resolver.resolveById(taskId);
        if (!(object instanceof Task))
        {
            throw new IllegalClassException(Task.class, object.getClass());
        }
        
        // Check user right
        ModifiableTraversableAmetysObject tasksRoot = object.getParent();
        if (_rightManager.hasRight(_currentUserProvider.getUser(), RIGHTS_COMMENT_TASK, tasksRoot) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to react a comment task without convenient right [" + RIGHTS_COMMENT_TASK + "]");
        }
        
        Task task = (Task) object;
        
        likeOrUnlikeComment(task, commentId, liked, tasksRoot);
        
        return _taskJSONHelper.taskAsJSON(task, _getSitemapLanguage(), _getSiteName());
    }
    
    /**
     * Set task's attributes
     * @param task The task to edit
     * @param parameters The JS parameters
     * @param newFiles the new file to add to the task
     * @param newFileNames the new file names to add to the task
     * @param deleteFiles the file to remove from the task
     * @return the map of results
     */
    protected Map<String, Object> _setTaskAttributes(JCRTask task, Map<String, Object> parameters, List<Part> newFiles, List<String> newFileNames, List<String> deleteFiles)
    {
        Map<String, Object> results = new HashMap<>();
        
        String label = (String) parameters.get(JCRTask.ATTRIBUTE_LABEL);
        task.setLabel(label);

        String description = (String) parameters.get(JCRTask.ATTRIBUTE_DESCRIPTION);
        task.setDescription(description);
        
        _setTaskDates(task, parameters);
        _setTaskCloseInfo(task, parameters, results);
        _setAttachments(task, newFiles, newFileNames, deleteFiles);
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> assignmentIds = (List<Map<String, Object>>) parameters.getOrDefault(JCRTask.ATTRIBUTE_ASSIGNMENTS, new ArrayList<>());
        List<UserIdentity> users = assignmentIds.stream()
            .map(m -> (String) m.get("id"))
            .map(UserIdentity::stringToUserIdentity)
            .collect(Collectors.toList());
        
        if (!task.getAssignments().equals(users))
        {
            task.setAssignments(users);
            results.put("changedAssignments", true);
        }
        
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> checkListItems = (List<Map<String, Object>>) parameters.getOrDefault(JCRTask.ATTRIBUTE_CHECKLIST, new ArrayList<>());
        List<CheckItem> checkItems = checkListItems.stream()
            .map(e -> new CheckItem((String) e.get(JCRTask.ATTRIBUTE_CHECKLIST_LABEL), (boolean) e.get(JCRTask.ATTRIBUTE_CHECKLIST_ISCHECKED)))
            .collect(Collectors.toList());
        task.setCheckListItem(checkItems);

        @SuppressWarnings("unchecked")
        List<Object> tags = (List<Object>) parameters.getOrDefault(JCRTask.ATTRIBUTE_TAGS, new ArrayList<>()); 

        List<Map<String, Object>> createdTagsJson = _handleTags(task, tags);
        
        results.put("newTags", createdTagsJson);
        return results;
    }
    
    private void _setTaskDates(JCRTask task, Map<String, Object> parameters)
    {
        String startDateAsStr = (String) parameters.get("startDate");
        LocalDate startDate = Optional.ofNullable(startDateAsStr)
                .map(date -> LocalDate.parse(date, DateTimeFormatter.ISO_LOCAL_DATE))
                .orElse(null);
        task.setStartDate(startDate);
        
        String dueDateAsStr = (String) parameters.get(JCRTask.ATTRIBUTE_DUEDATE);
        LocalDate dueDate = Optional.ofNullable(dueDateAsStr)
                .map(date -> LocalDate.parse(date, DateTimeFormatter.ISO_LOCAL_DATE))
                .orElse(null);
        task.setDueDate(dueDate);
    }
    
    private void _setTaskCloseInfo(JCRTask task, Map<String, Object> parameters, Map<String, Object> results)
    {
        @SuppressWarnings("unchecked")
        Map<String, Object> closeInfo = (Map<String, Object>) parameters.get("closeInfo");
        if (closeInfo != null && !task.isClosed())
        {
            task.close(true);
            task.setCloseAuthor(_currentUserProvider.getUser());
            task.setCloseDate(LocalDate.now());

            results.put("isClosed", true);
        }
        else if (closeInfo == null && task.isClosed())
        {
            task.close(false);
            task.setCloseAuthor(null);
            task.setCloseDate(null);

            results.put("isClosed", false);
        }
    }
    
    
    /**
     * Get all tasks from given projets
     * @param project the project
     * @return All tasks as JSON
     */
    public List<Task> getProjectTasks(Project project)
    {
        TasksWorkspaceModule taskModule = _workspaceModuleEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
        ModifiableTraversableAmetysObject tasksRoot = taskModule.getTasksRoot(project, true);
        return tasksRoot.getChildren()
            .stream()
            .filter(Task.class::isInstance)
            .map(Task.class::cast)
            .collect(Collectors.toList());
    }
    
    /**
     * Get the total number of tasks of the project
     * @param project The project
     * @return The number of tasks, or null if the module is not activated
     */
    public Long getTasksCount(Project project)
    {
        return Long.valueOf(getProjectTasks(project).size());
    }
    
    /**
     * Get project members 
     * @return the project members
     * @throws IllegalAccessException if an error occurred
     * @throws AmetysRepositoryException if an error occurred
     */
    @Callable
    public Map<String, Object> getProjectMembers() throws IllegalAccessException, AmetysRepositoryException 
    {
        String projectName = _getProjectName();
        String lang = _getSitemapLanguage();
        
        return _projectMemberManager.getProjectMembers(projectName, lang, true);
    }
}
