/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks.json;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.Task.CheckItem;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTask;
import org.ametys.plugins.workspaces.util.WorkspaceObjectJSONHelper;

/**
 * Helper to convert task to JSON
 */
public class TaskJSONHelper extends WorkspaceObjectJSONHelper implements Component
{
    /** The Avalon role */
    public static final String ROLE = TaskJSONHelper.class.getName();
    
    /**
     * Task as JSON
     * @param task the task
     * @param lang the language
     * @param siteName the site name
     * @return the task as JSON
     */
    public Map<String, Object> taskAsJSON(Task task, String lang, String siteName)
    {
        Map<String, Object> json = new HashMap<>();
        
        json.put(JCRTask.ATTRIBUTE_TASK_ID, task.getId());
        json.put(JCRTask.ATTRIBUTE_LABEL, task.getLabel());
        json.put(JCRTask.ATTRIBUTE_DESCRIPTION, task.getDescription());
        json.put(JCRTask.ATTRIBUTE_TASKSLISTID, task.getTaskListId());
        json.put("position", task.getPosition());
        json.put(JCRTask.ATTRIBUTE_CREATIONDATE, task.getCreationDate());
        json.put("closeInfo", _closeInfoAsJson(task));
        
        json.put(JCRTask.ATTRIBUTE_ASSIGNMENTS, _assignmentsToJson(task));
        
        json.put(JCRTask.ATTRIBUTE_TAGS, _getTags(task, siteName));
        
        List<Map<String, Object>> attachments = task.getAttachments()
            .stream()
            .map(this::_binaryToJson)
            .collect(Collectors.toList());
        json.put(JCRTask.ATTRIBUTE_ATTACHMENTS, attachments);
        
        json.put(JCRTask.ATTRIBUTE_CHECKLIST, _checkListToJson(task));

        json.put("comments", _commentsToJson(task.getComments(true, true), lang, siteName));
        json.put(JCRTask.ATTRIBUTE_DUEDATE, task.getDueDate());
        json.put("datePassed", task.isClosed());
        
        return json;
    }
    
    private Map<String, Object> _closeInfoAsJson(Task task)
    {
        UserIdentity closeAuthor = task.getCloseAuthor();
        
        return task.isClosed() 
                ? Map.of(JCRTask.ATTRIBUTE_CLOSEAUTHOR, _userHelper.user2json(closeAuthor), JCRTask.ATTRIBUTE_CLOSEDATE, task.getCloseDate()) 
                : null;
    }

    private List<Map<String, Object>> _assignmentsToJson(Task task)
    {
        return task.getAssignments()
            .stream()
            .map(u -> _userHelper.user2json(u))
            .collect(Collectors.toList());
    }
    
    private List<Map<String, Object>> _checkListToJson(Task task)
    {
        List<Map<String, Object>> json = new ArrayList<>();
        for (CheckItem checkItem : task.getCheckList())
        {
            json.add(Map.of(JCRTask.ATTRIBUTE_CHECKLIST_LABEL, checkItem.getLabel(), JCRTask.ATTRIBUTE_CHECKLIST_ISCHECKED, checkItem.isChecked()));
        }
        return json;
    }
    
}
