/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
try { logger; }
catch (e) { logger = Ametys.getLogger("org.ametys.core.migration.MigrationExtensionPoint.workspaces"); }

let projectManager = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.project.ProjectManager.ROLE);
let workspaceModuleExtensionPoint = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint.ROLE);

const conversion = {
  'WORKSPACES_MODULE_ALERT': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.alert.AlertWorkspaceModule'),
  'WORKSPACES_MODULE_CALENDAR': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.calendars.CalendarWorkspaceModule'),
  'WORKSPACES_MODULE_DOCUMENT': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule'),
  'WORKSPACES_MODULE_WIKI': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.editionfo.EditionFOWorkspaceModule'),
  'WORKSPACES_MODULE_MEMBERS': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.members.MembersWorkspaceModule'),
  'WORKSPACES_MODULE_NEWS': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.news.NewsWorkspaceModule'),
  'WORKSPACES_MODULE_TASK': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule'),
  'WORKSPACES_MODULE_WALLCONTENT': workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.wall.WallContentModule')
};

var countProjects = 0;
var countProjectsCategories = 0;
var countProjectOrder = 0;
var countPages = 0;

projectManager.getProjects().forEach(function (project) 
{
  logger.debug("Migration of " + project.getName());
  
  migrateProjectPageTag(project, true);
  migrateProjectACL(project, true);

  try
  {
    var liveProject = Repository.resolver.resolveById(project.getId(), Repository.liveSession);
  
    migrateProjectPageTag(liveProject, false);
    migrateProjectACL(liveProject, false);
  }
  catch (e) { }
  
  migrateProjectPageOrder(project);
  migrateProjectCategories(project);    
});

logger.info("ACL on " + countProjects + " project(s) removed");
logger.info(countProjectsCategories + " project(s) were categorized (since it is now mandatory)");
logger.info(countProjectOrder + " project had pages that were reordered to match new modules order");
logger.info(countPages + " module page(s) modified");

function migrateProjectPageOrder(project)
{
  let site = project.getSite()
    
  for (let moduleIndex in project.getModules())
    {
      let moduleId = project.getModules()[moduleIndex];
      let module = workspaceModuleExtensionPoint.getModule(moduleId);
      let pages = projectManager.getModulePages(project, module).iterator();
  
      while(pages.hasNext())
      {
        let page = pages.next();
          
        let modules = workspaceModuleExtensionPoint.getModules().iterator();
        while (modules.hasNext())
        {
          let otherModule = modules.next();
          if (otherModule.compareTo(module) > 0)
          {
            let otherModulePages = projectManager.getModulePages(project, otherModule);
            if (!otherModulePages.isEmpty())
            {
              let otherModulePage = otherModulePages.iterator().next();
              page.orderBefore(otherModulePage);
  
              try 
              {
                  let page2 = Repository.resolver.resolveById(page.getId(), Repository.liveSession);
                  let otherModulePage2 = Repository.resolver.resolveById(otherModulePage.getId(), Repository.liveSession);
                  page2.orderBefore(otherModulePage2);
              }
              catch (e) { continue; }
  
              break;
            }
          }
        }
      }
    }
      
    site.saveChanges();
    Repository.resolver.resolveById(site.getId(), Repository.liveSession).saveChanges();
}

function migrateProjectPageTag(project, defaultWorkspace)
{
  let it = project.getSites().iterator();
  while (it.hasNext())
  {
    let site = it.next();
    site.getSitemaps().forEach(function (sitemap) 
    {
      logger.debug("Migration of " + site.getName() + " /" + sitemap.getName() + " on workspace " + (defaultWorkspace ? "default" : "live"));
      for (var tag in conversion)
      {
        let xpath = org.ametys.web.repository.page.PageQueryHelper.getPageXPathQuery(site.getName(), sitemap.getName(), null, new org.ametys.web.tags.TagExpression(org.ametys.plugins.repository.query.expression.Expression.Operator.EQ, tag), null);
        Repository.query(xpath, true, defaultWorkspace ? Repository.session : Repository.liveSession).forEach(page => migratePage(page, project, tag, defaultWorkspace));
      }
    });
  }
}  

function migrateProjectACL(project, defaultWorkspace)
{
  let projectNode = project.getNode();
  if (projectNode.hasNode("ametys-internal:acl"))
  {
    if (defaultWorkspace)
    {
      countProjects++;
    }
    logger.debug("Removing ACL on workspace " + (defaultWorkspace ? "default" : "live"));
    projectNode.getNode("ametys-internal:acl").remove();      
  }
  
  // remove reader in all modules
  for (let moduleIndex in project.getModules())
  {
    let moduleId = project.getModules()[moduleIndex];
    let module = workspaceModuleExtensionPoint.getExtension(moduleId);
    let moduleRoot = module.getModuleRoot(project, false);
    if (moduleRoot)
    {
      logger.debug("Removing ACL on module " + moduleId + " on workspace " + (defaultWorkspace ? "default" : "live"));
      removeReader(moduleRoot);
    }
  }
  
  if (project.needsSave())
  {
    project.saveChanges();
  }
}

function migrateProjectCategories(project)
{
  if (project.getCategories().isEmpty())
  {
    _createDefaultCategoryIfNeeded();
    logger.debug("Project affected to the default category")
    project.addCategory("DEFAULT_MIGRATED");
    project.saveChanges();
    countProjectsCategories++;
  }
}

var checked = false;
function _createDefaultCategoryIfNeeded()
{
  if (!checked)
  {
    let categoryProviderExtensionPoint = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint.ROLE);
    if (categoryProviderExtensionPoint.getTag("DEFAULT_MIGRATED", java.util.Map.of()) == null)
    {
      logger.debug("Creating a default category for migrated projects");
      let categoryJCRDAO = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.categories.CategoryJCRDAO.ROLE);
      categoryJCRDAO.addTag(null, "DEFAULT_MIGRATED", "Default migration category", "This category was created during the migration of workspaces since categories are now mandatory", java.util.Map.of(), java.util.Map.of());
    }
      
    checked = true;
  }
}
  
function migratePage(page, project, tag, defaultWorkspace)
{
  let moduleRoot = conversion[tag].getModuleRoot(project, false);

  logger.debug("Removing tag " + tag + " from page " + page.getPathInSitemap() + " (" + page.getId() + ") on workspace " + (defaultWorkspace ? "default" : "live"));
  page.untag(tag);

  if (moduleRoot)
  {
    logger.debug("Setting module metadata on page " + page.getPathInSitemap() + " (" + page.getId() + ") on workspace " + (defaultWorkspace ? "default" : "live"));
    projectManager.tagProjectPage(page, moduleRoot);
  }
  
  removeReader(page);
  
  if (page.needsSave)
  {
    page.saveChanges();
    if (defaultWorkspace)
    {
      countPages++;
    }
  }
}

function removeReader(ametysObject)
{
  let existingProfiles = getExistingProfilesForUsers(ametysObject);
  while (existingProfiles.hasNext())
  {
    let existingProfile = existingProfiles.next();
    if (existingProfile.getValue().contains("READER"))
    {
      logger.info("Removing READER profile for " + existingProfile.getKey())
      ametysObject.removeAllowedUsers(java.util.Set.of(existingProfile.getKey()), "READER");
    }
  }
  existingProfiles = getExistingProfilesForGroups(ametysObject);
  while (existingProfiles.hasNext())
  {
    let existingProfile = existingProfiles.next();
    if (existingProfile.getValue().contains("READER"))
    {
      logger.info("Removing READER profile for " + existingProfile.getKey())
      ametysObject.removeAllowedGroups(java.util.Set.of(existingProfile.getKey()), "READER");
    }
  }
}

function getExistingProfilesForUsers(ametysObject) 
{
    // Compatible for 4.3
    if (ametysObject.getAllowedProfilesForUsers)
    {
       return ametysObject.getAllowedProfilesForUsers().entrySet().iterator();
    }
    else // Compatible for 4.4
    {
        let map = new java.util.HashMap();
        let profiles = ametysObject.getProfilesForUsers(null);
        for (let key of profiles.keySet())
        {
            map.put(key, profiles.get(key).get(org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.ALLOWED));
        }
       
        return map.entrySet().iterator();
    }
}

function getExistingProfilesForGroups(ametysObject) 
{
    // Compatible for 4.3
    if (ametysObject.getAllowedProfilesForGroups)
    {
        return ametysObject.getAllowedProfilesForGroups().entrySet().iterator();
    }
    else // Compatible for 4.4
    {
        let map = new java.util.HashMap();
        let profiles = ametysObject.getProfilesForGroups(null);
        for (let key of profiles.keySet())
        {
             map.put(key, profiles.get(key).get(org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.ALLOWED));
        }
       
        return map.entrySet().iterator();
    }
}
