/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

// Move previous event node to activity node

Repository.registerNodeType("event");
Repository.registerNodeType("eventElement");

logger.info("Start migration of events nodes for default session");
migrateEventsNodes(Repository.session);

logger.info("Start migration of events nodes for live session");
migrateEventsNodes(Repository.liveSession);

logger.info("Migration ef events nodes finished successfully.");

function migrateEventsNodes(s)
{
    let count = 0;
    const xpath = "//element(ametys-internal:events, nt:unstructured)";
    const eventsNodes = Repository.query(xpath, false, s);
            
    while (eventsNodes.hasNext())
    {
        migrateEventsNode(eventsNodes.next());
        count++;
    }
    logger.info(`${count} event${count > 1 ? "s" :""} holder have been migrated`);
    
    s.save();
    
}

function migrateEventsNode(eventsNode)
{
    let eventHolder = eventsNode.getParent();
    let errorCount = 0;
    logger.info(`migrating events in ${eventsNode.getPath()}`);
    let activitiesCollection = getOrCreateActivitiesCollection(eventHolder);
    
    let countEvents = 0;
    let firstLevelIterator = eventsNode.getNodes();
    while (firstLevelIterator.hasNext())
    {
        let secondLevelIterator = firstLevelIterator.next().getNodes();
        while (secondLevelIterator.hasNext())
        {
            let eventNodeIterator = secondLevelIterator.next().getNodes();
            while (eventNodeIterator.hasNext())
            {
                let eventNode = eventNodeIterator.next();
                // before doing anything, we check that the event node is not corrupted
                countEvents++;
                if (isCorruptedEvent(eventNode))
                {
                    logger.debug(`event node '${eventNode.getPath()}' (${eventNode.getIdentifier()}) is corrupted and won't be migrated.`);
                    errorCount++;
                }
                else
                {
                    let activityName = "ametys-activity" + eventNode.getName().substring(12);
                    let activity;
                    if (activitiesCollection.hasChild(activityName))
                    {
                        activity = activitiesCollection.getChild(activityName);
                    }
                    else
                    {
                        activity = activitiesCollection.createChild(activityName, 'ametys:activity');
                    }
                    migrateEventToActivity(eventNode, activity);
                }
            }
        }
    }
    if (errorCount > 0)
    {
        logger.warn(`${errorCount} error${errorCount > 1 ? "s" :""} occurred while migrating ${countEvents} event${errorCount > 1 ? "s" :""} to activity`);
    }
    else
    {
        logger.info(`${countEvents} event${errorCount > 1 ? "s" :""} migrated to activity`);
    }
    eventsNode.remove();
    logger.info("event holder deleted.");
}

function getOrCreateActivitiesCollection(eventHolder)
{
    let activitiesNode;
    if (eventHolder.hasNode('ametys-internal:activities'))
    {
        logger.info(`found an existing activities node for ${eventHolder.getName()}`);
        activitiesNode = eventHolder.getNode('ametys-internal:activities');
        // we need to check that the activitiesNode is of the right type
        if (!activitiesNode.isNodeType('ametys:collection'))
        {
            throw new org.ametys.core.migration.MigrationException(`Event holder node '${eventHolder.getName()}' already has an child named 'ametys-internal:activities' but the type is not 'ametys:collection'.`);
        }
    }
    else
    {
        logger.info(`creating new activities node for ${eventHolder.getName()}`);
        activitiesNode = eventHolder.addNode('ametys-internal:activities');
    }
    return Repository.resolver.resolve(activitiesNode, false);
}

function migrateEventToActivity(eventNode, activity)
{
    // migrate properties
    let propertyIterator = eventNode.getProperties();
    while (propertyIterator.hasNext())
    {
        let property = propertyIterator.next();
        let propertyName = property.getName();
        if (propertyName.substring(0, 7) == "ametys:")
        {
            if (property.isMultiple())
            {
                activity.getNode().setProperty(propertyName, property.getValues());
            }
            else
            {
                activity.getNode().setProperty(propertyName, property.getValue());
            }
        }
    }

    // migrate subnodes (ametys:author and potential file)
    let nodeIterator = eventNode.getNodes();
    let resourceReferences = [];
    while (nodeIterator.hasNext())
    {
        let subNode = nodeIterator.next();
        let subNodeName = subNode.getName()
        if (subNodeName === "ametys:author")
        {
            let login = subNode.getProperty("ametys:login").getString();
            let population = subNode.getProperty("ametys:population").getString();
            let userIdentity = new org.ametys.core.user.UserIdentity(login, population);
            activity.setValue("author", userIdentity);
        }
        else if (subNodeName.substring(0, 4) === "file")
        {
            let resourceId = subNode.getProperty("resourceId").getString();
            let name = subNode.getProperty("name").getString();
            let oldName = null;
            if (subNode.hasProperty("oldName")) {
                oldName = subNode.getProperty("oldName").getString();
            }
            let mimeType = subNode.getProperty("mimeType").getString();
            let version = null;
            if (subNode.hasProperty("baseVersionName")) {
                version = subNode.getProperty("baseVersionName").getString();
            }
            resourceReferences.push(new org.ametys.plugins.workspaces.activities.documents.AbstractResourceReferenceElementType.ResourceReference(resourceId, name, oldName, mimeType, version));
        }
    }

    if (resourceReferences.length > 0)
    {
        let eventType = eventNode.getProperty("ametys:type").getString();
        // resource rename is not a multiple value unlike other resource activities
        if (eventType == "resource.renamed")
        {
            activity.setValue("file", resourceReferences[0]);
        }
        else
        {
            Repository.helper.setModelLessValue(activity, "files", resourceReferences, "resourceReference");
        }
    }
}

function isCorruptedEvent(eventNode)
{
    // an event node is considered corrupted if doesn't have a date, a type and an author
    var corrupted = false;
    if (!eventNode.hasProperty("ametys:type"))
    {
        logger.error(`node '${eventNode.getPath()} has no type`);
        corrupted = true;
    }
    if (!eventNode.hasProperty("ametys:date"))
    {
        logger.error(`node '${eventNode.getPath()} has no date`);
        corrupted = true;
    }
    if (!eventNode.hasNode("ametys:author"))
    {
        logger.error(`node '${eventNode.getPath()} has no author`);
        corrupted = true;
    }
    else
    {
        var authorNode = eventNode.getNode("ametys:author");
        if (!authorNode.hasProperty("ametys:login"))
        {
            logger.error(`node '${eventNode.getPath()} has no author's login`);
            corrupted = true;
        }
        if (!authorNode.hasProperty("ametys:population"))
        {
            logger.error(`node '${eventNode.getPath()} has no author's population`);
            corrupted = true;
        }
    }
    let nodeIterator = eventNode.getNodes();
    while (nodeIterator.hasNext())
    {
        let subNode = nodeIterator.next();
        let subNodeName = subNode.getName()
        if (subNodeName.substring(0, 4) === "file")
        {
            if (!subNode.hasProperty("resourceId") || !subNode.hasProperty("name") || !subNode.hasProperty("mimeType"))
            {
                logger.warn(`node '${eventNode.getPath()} has a corrupted file reference (missing resourceId, name, or mimeType)`);
                corrupted = true;
            }
        }
    }
    return corrupted;
}
