/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import Vue from 'vue';
import AmetysFront from 'AmetysFront';
import i18n from 'i18n';

import {callMethod} from '../../helper/ServerCommHelper';

export const namespaced = true;
export const state = {
  folder: {},
  folders: [],
  isFilter: false,
  currentFolder: {},
  treeOpenedFolders: [],
  treeActiveFolder: []
};

export const mutations = {
  SET_CURRENT_FOLDER(state, folder) {
    state.currentFolder = folder;
  },
  SET_FOLDERS(state, folders) {
    state.folders = folders;
  },
  UPDATE_FOLDERS_AFTER_MOVE(state, {selectedFoldersIds, toFolderId}) {
    function getFolder(folder, id) {
        if (folder.id == id)
        {
            return folder;
        }
        
        if (folder.children)
        {
            for (let child of folder.children)
            {
                let childF = getFolder(child, id);
                if (childF != null)
                {
                    return childF;
                }
            }
        }
        
        return null;
    }
    
    function getMovedFolders(folders, selectedFoldersIds) {
        let movedFolders = folders.filter((f) => selectedFoldersIds.includes(f.id));
        for (let f of folders)
        {
            if (f.children && f.children.length)
            {
                for (let childF of getMovedFolders(f.children, selectedFoldersIds))
                {
                    movedFolders.push(childF);                
                }
            }
        }
        
        return movedFolders;
    }
    
    let toFolder = getFolder(state.folder, toFolderId);
    let path = toFolder.path;
    
    let movedFolders = getMovedFolders(state.folder.children, selectedFoldersIds);
    for (let f of movedFolders)
    {
        f.path = path;
        f.parentId = toFolderId;
    }
    
    /* Remove Moved Folders */
    function removedFolders(data, ids) {
      const items = data.filter((d) => !ids.includes(d.id));
      if (items.length > 0) 
      {
        items.forEach((item) => { 
          if (item.children != undefined) item.children = removedFolders(item.children, ids);
          if (item.id == toFolderId)
          {
            if (item.children == undefined) 
            {
                item.children = [];
            }
            
            for (let movedF of movedFolders)
            {
                item.children.push(movedF);
            }
          }
        });
      }
      return items.length > 0 ? items : undefined;
    }
    
    state.folder.children = removedFolders(state.folder.children, selectedFoldersIds).sort(function(f1, f2) {
        return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
    });
    if (state.folder.id == toFolderId)
    {
        for (let movedF of movedFolders)
        {
            state.folder.children.push(movedF);
        }
        state.folder.children = state.folder.children.sort(function(f1, f2) {
            return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
        });
    }
    
    if(state.currentFolder.id == toFolderId){
      this.dispatch('folders/openFolder', state.currentFolder);
    }
    
    let newFolders = removedFolders(state.folders, selectedFoldersIds);
    state.folders = newFolders != undefined ? newFolders : [];
    if (state.treeActiveFolder)
    {
        let activeFolder = removedFolders(state.treeActiveFolder, selectedFoldersIds);
        state.treeActiveFolder = activeFolder != undefined ? newFolders : [];
    }
    let treeOpenedFolders = removedFolders(state.treeOpenedFolders, selectedFoldersIds);
    state.treeOpenedFolders = treeOpenedFolders != undefined ? treeOpenedFolders : [];
    
    let currentF = movedFolders.filter((f) => f.id == state.currentFolder.id);
    if (currentF.length > 0)
    {
        state.currentFolder = currentF[0];
    }
    
    AmetysFront.Event.fire('rerender-file-tree');
 
  },
  SET_FOLDER(state, folder) {
    state.folder = folder;
  },
  CREATE_FOLDER(state, folder) {
    if (state.currentFolder.id == folder.parentId)
    {
        if (state.currentFolder.hasOwnProperty('children')) {
          state.currentFolder.children.push(folder)
        } else {
          Vue.set(state.currentFolder, 'children', [folder])
        }
    
        state.folders = state.currentFolder.children
        
        AmetysFront.Event.fire('rerender-file-tree');        
    }
  },
  RENAME_FOLDER(state, payload) {
    payload.folder.name = payload.name;
  },
  UPDATE_FOLDER_DESC(state, payload) {
    payload.folder.description = payload.description;
  },
  DELETE_FOLDER(state, payload) {
    let index = state.folders.findIndex(folder => folder.id == payload.folderId);
    if (index != -1)
    {
        state.folders.splice(index, 1);
    }        
    let indexOpenFolder = state.treeOpenedFolders.findIndex(folder => folder.id == payload.folderId);
    if (indexOpenFolder != -1)
    {
        state.treeOpenedFolders.splice(indexOpenFolder, 1);
    }
    
    if (state.treeActiveFolder.id == payload.folderId)
    {
        state.treeActiveFolder = [];
    }        
  },
  SET_TREE_OPENED_FOLDER(state, folder) {
    state.treeOpenedFolders = folder;
  },
  ADD_TREE_OPENED_FOLDER(state, folder) {
    let index = state.treeOpenedFolders.findIndex(f => f.id == folder.id);
    if (index == -1)
    {
        state.treeOpenedFolders.push(folder);
    }
    else
    {
        state.treeOpenedFolders.splice(index, 1, folder)
    }
  },
  SET_TREE_ACTIVE_FOLDER(state, folder) {
    if (folder)
    {
        state.treeActiveFolder = [folder];
    }
    else
    {
        state.treeActiveFolder = [];
    }
  }
}

export const getters = {
  /**
   * Get breadcrumb for current opened folder
   */
  getPath: (state /*, getters, rootState */) => {
    if (!state.currentFolder.hasOwnProperty('id')) return false
    let paths = []
    let obj = [state.folder]
    let pathKey = 0

    let collectPath = function () {
      if (!state.currentFolder.path.length) {
        return paths
      }

      obj.forEach(child => {
        if (child.id === state.currentFolder.path[pathKey]) {
          paths.push(child)

          pathKey++

          if (pathKey < state.currentFolder.path.length) {
            obj = child.children
            return obj != undefined ? collectPath() : [];               
          }
        }
      })

      return paths
    }

    return collectPath()
  },

  /**
   * Get folders of the current opened folder
   */
  getFolders: (state, getters, rootState) => {
    if (rootState.filters.hideFolders) {
      return []
    }
    else {
        return state.folders.sort(function(f1, f2) {
            return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
        });
    }
  }
}

export const actions = {

  /**
   * Reload the current folder (folders and files)
   */
  async reload({commit, /* state,  dispatch */}) {
    try
    {
        var data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [state.currentFolder.id]
        });

        state.currentFolder.children = data.children;

        commit('SET_FOLDERS', data.children);
        commit('files/SET_FILES', data.files, {root: true});
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },

  /**
   * Get folder's data with child folders and files
   */
  async getFolder({commit, /* state, */ dispatch}, id) {
    commit('filters/SET_HIDE_LOADING', true, {root: true});

    try
    {
        var data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [id]
        });

        if (!id)
            data.root = true;

        commit('SET_FOLDER', data);
        commit('SET_CURRENT_FOLDER', data)
        commit('SET_FOLDERS', data.children);

        commit('files/SET_FILES', data.files, {root: true});

        commit('filters/SET_HIDE_LOADING', false, {root: true});
        dispatch('filters/resetFilters', false, {root: true});
        dispatch('selectedItems/unselectAll', null, {root: true});

        dispatch('info/setItem', {item: data, notOpen: true}, {root: true});

        if (!data.root)
        {
            AmetysFront.Event.fire('hideMobileHeader');
        }
        else
        {
            AmetysFront.Event.fire('showMobileHeader', true);
        }
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },

  /**
   * Expand folder in tree
   */
  async expandFolder(obj /*{commit,state,  dispatch}*/, folder) {

    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });
        
        folder.children = data.children.length ? data.children.sort(function(f1, f2) {
            return f1.name.toLowerCase() < f2.name.toLowerCase() ? -1 : (f1.name.toLowerCase() > f2.name.toLowerCase() ? 1 : 0);
        }) : undefined;
    }
    catch(e)
    {
       AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },

  /**
   * Open folder from breadcrumb
   */
  async breadOpenFolder({commit, /*state,*/  dispatch}, folder) {

    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });

        commit('filters/SET_HIDE_LOADING', true, {root: true});

        commit('SET_CURRENT_FOLDER', folder)
        commit('SET_FOLDERS', folder.children ? folder.children : []);
        commit('ADD_TREE_OPENED_FOLDER', folder);
        commit('SET_TREE_ACTIVE_FOLDER', folder);

        commit('files/SET_FILES', data.files, {root: true});

        commit('filters/SET_HIDE_LOADING', false, {root: true});
        dispatch('filters/resetFilters', false, {root: true});
        dispatch('selectedItems/unselectAll', null, {root: true});

        dispatch('info/setItem', {item: folder, notOpen: true}, {root: true});

        if (folder.root)
        {
            AmetysFront.Event.fire('showMobileHeader', true);
        }
        else
        {
            AmetysFront.Event.fire('hideMobileHeader');
        }

    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },
  
  /**
   * Open the parent folder of the given file
   */
  async openFileParentFolder({commit, /*state,*/  dispatch}, fileId) 
  {
    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFileParentInfo',
            parameters: [fileId]
        });
        
        if (!data.error)
        {
            commit('filters/SET_HIDE_LOADING', true, {root: true});
    
            commit('SET_CURRENT_FOLDER', data)
            commit('SET_FOLDERS', data.children ? data.children : []);
            commit('ADD_TREE_OPENED_FOLDER', data);
            commit('SET_TREE_ACTIVE_FOLDER', data);
    
            commit('files/SET_FILES', data.files, {root: true});
    
            commit('filters/SET_HIDE_LOADING', false, {root: true});
            dispatch('filters/resetFilters', false, {root: true});
            dispatch('selectedItems/unselectAll', null, {root: true});
    
            dispatch('info/setItem', {item: data, notOpen: true}, {root: true});
    
            if (data.root)
            {
                AmetysFront.Event.fire('showMobileHeader', true);
            }
            else
            {
                AmetysFront.Event.fire('hideMobileHeader');
            }
        }
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },

  /**
   * Open folder
   */
  async openFolder({commit, /* state,*/ dispatch}, folder) {
    commit('filters/SET_HIDE_LOADING', true, {root: true});

    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });

        if (data.hasOwnProperty('children')) {
            folder.children = data.children;
        } else {
            Vue.delete(folder, 'children')
        }

        commit('SET_CURRENT_FOLDER', folder)
        commit('SET_FOLDERS', folder.children ? folder.children : []);
        commit('ADD_TREE_OPENED_FOLDER', folder);
        commit('SET_TREE_ACTIVE_FOLDER', folder);

        commit('files/SET_FILES', data.files, {root: true});

        commit('filters/SET_HIDE_LOADING', false, {root: true});
        dispatch('filters/resetFilters', false, {root: true});
        dispatch('selectedItems/unselectAll', null, {root: true});

        dispatch('info/setItem', {item: folder, notOpen: true}, {root: true});

        AmetysFront.Event.fire('reset-filter');
        AmetysFront.Event.fire('hideMobileHeader');
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
          title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
          text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
          details: null
        });

        commit('filters/SET_HIDE_LOADING', false, {root: true});
        dispatch('filters/resetFilters', false, {root: true});

        AmetysFront.Event.fire('reset-filter')
    }
  },

  /**
   * Create a new folder
   * @param {String} name the folder's name
   * @param {String} description the folder's description
   */
  async createFolder({commit, state/*, dispatch*/}, {folder, name, description}) {

    let parentFolder = folder ? folder : state.currentFolder;
    try
    {
        let result = await callMethod({
          role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
          methodName: 'addFolder',
          parameters: [parentFolder.id /* parentId */, name /* name */, description]
        });

        if (result.error)
        {
            var text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR;
            if (result.message == 'locked')
            {
                text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED;
            }
            else if (result.message == 'already-exist')
            {
                text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ALREADY_EXIST;
            }

            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ADD,
                text : text,
                details: null
            });
        }
        else
        {
            commit('CREATE_FOLDER', result);
            commit('ADD_TREE_OPENED_FOLDER', parentFolder);
            commit('moveFoldersAndFiles/CREATE_FOLDER', result, {root:true});
            commit('moveFoldersAndFiles/ADD_TREE_OPENED_FOLDER', parentFolder, {root:true});
            commit('moveFoldersAndFiles/SET_CURRENT_FOLDER', result, {root:true});
        }
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
          title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ADD,
          text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
          details: null
        });
    }
  },

  /**
   * Delete a folder
   */
  async deleteFolder({commit, state}, payload) {

    AmetysFront.Event.fire('loaderStart', {
      text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETING
    });

    if (state.currentFolder.children.length === 1) {
      state.treeOpenedFolders = state.treeOpenedFolders.filter(f => f.id !== state.currentFolder.id)
    }

    try
    {
        let result = await callMethod({
          role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
          methodName: 'deleteFolder',
          parameters: [payload.id]
        });

        if (!result.error)
        {
            await commit('DELETE_FOLDER', {folderId: payload.id});

            let data = await callMethod({
                role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                methodName: 'getFoldersAndFiles',
                parameters: [payload.parentId]
            });
            
            if (state.currentFolder.id == payload.id)
            {
                await commit('files/SET_FILES', data.files, {root: true});
                await commit('SET_CURRENT_FOLDER', data)
                await commit('SET_TREE_ACTIVE_FOLDER', data);
                await commit('SET_FOLDERS', data.children);

                // Parent is root
                if (data.root)
                {
                  await commit('SET_FOLDER', data)
                }
                else
                {
                    let indexOpenFolder = state.treeOpenedFolders.findIndex(folder => folder.id == payload.parentId);
                    if (indexOpenFolder != -1)
                    {
                        state.treeOpenedFolders.splice(indexOpenFolder, 1);
                    }
                    
                    let folder = state.folder;
                    removeFolder(folder);
                    AmetysFront.Event.fire('rerender-file-tree');
                }
            }
            else if (!state.currentFolder.children.length)
            {
               Vue.delete(state.currentFolder, 'children')
            }

            await commit('info/SET_ITEM', state.currentFolder, {root: true});

            setTimeout(() => {
              AmetysFront.Event.fire('loaderEnd', {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_FOLDER_SUCCESS
              });
            }, 500);
        }
        else
        {
            var errorMsg = result.message;
            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_DELETE,
                text : errorMsg == 'locked' ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                details: null
            });
        }

    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', {
          title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_DELETE,
          text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
          details: null
      });
    }
    
    function removeFolder(folder)
    {
        if (folder.children)
        {
            let index = folder.children.findIndex(folder => folder.id == payload.id);
            if (index != -1)
            {
                folder.children.splice(index, 1);
            }
            else
            {
                for (let f of folder.children)
                {
                    removeFolder(f);
                }
            }
        }
    }
  },

  /**
   * Rename a folder
   */
  async renameFolder({commit, /*state , dispatch*/}, payload) {

    let folder = payload.folder;
    let success = false;

    try
    {
        let result = await callMethod({
          role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
          methodName: 'renameFolder',
          parameters: [folder.id, payload.name]
        });

        if (!result.success)
        {
            var text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR;
            if (result.message == 'locked')
            {
                text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED;
            }
            else if (result.message == 'already-exist')
            {
                text = i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_ALREADY_EXIST;
            }

            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_RENAME,
                text : text,
                details: null
            });
          }
          else
          {
            success = true;
            commit('RENAME_FOLDER', {folder: folder, name: payload.name});
          }

          return success;
    }
    catch(e)
    {
         AmetysFront.Event.fire('loaderFail', {
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_RENAME,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }

    return success;
  },
  
  /**
   * Edit a folder
   */
  async updateFolder({commit, /*state , dispatch*/}, payload) {

    let folder = payload.folder;
    let success = false;

    try
    {
        let result = await callMethod({
          role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
          methodName: 'editFolder',
          parameters: [folder.id, payload.name, payload.description]
        });
        
        if (!result.success)
        {
            var text = result.message == 'locked'
                ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_LOCKED
                : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR;

            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_UPDATE,
                text : text,
                details: null
            });
        }
        else
        {
            success = true;
            if (payload.name)
            {
                commit('RENAME_FOLDER', {folder: folder, name: payload.name});
            }
            if (payload.description)
            {
                commit('UPDATE_FOLDER_DESC', {folder: folder, description: payload.description});
            }
        }
    }
    catch(e)
    {
         AmetysFront.Event.fire('loaderFail', { 
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_UPDATE,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
    
    return success;
  }
}
