/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * Singleton class to handle actions on project tags.
 * @private
 */
Ext.define('Ametys.plugins.workspaces.project.tag.ProjectTagActions', {
    singleton : true,

    /**
     * Function called to add a new tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    add: function(controller)
    {
    	var targets = controller.getMatchingTargets();
    	if (targets.length > 0)
        {
    		this._initFormPanel();
    		
    		var tagId = targets[0].getParameters().id;
    		Ametys.plugins.cms.tag.TagHelper.add(
    				tagId,
    				this._formPanel,
    				Ametys.message.MessageTarget.PROJECT_TAG,
    				"org.ametys.plugins.workspaces.tags.ProjectJCRTagsDAO",
    				"{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_CREATE}}",
    				"ametysicon-tag25 decorator-ametysicon-add64",
    				null
    		);
        }  
    },
    
    /**
     * Function called to edit a tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    edit: function (controller)
    {
    	var targets = controller.getMatchingTargets();
    	if (targets.length > 0)
        {
    		this._initFormPanel();
    		
    		var tagId = targets[0].getParameters().id;
    		Ametys.plugins.cms.tag.TagHelper.edit(
            		tagId,
            		this._formPanel,
            		Ametys.message.MessageTarget.PROJECT_TAG,
            		"org.ametys.plugins.workspaces.tags.ProjectJCRTagsDAO",
            		"{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_EDIT}}",
            		"ametysicon-tag25 decorator-ametysicon-edit45",
            		null
    		);
        }  
    },
    
    /**
     * Function called to remove a tag
     * @param {Ametys.ribbon.element.ui.ButtonController} controller The controller calling the function
     */
    remove: function (controller)
    {
    	var targets = controller.getMatchingTargets();
        if (targets.length > 0)
        {
            var tagId = targets[0].getParameters().id;
            Ametys.plugins.cms.tag.TagHelper.remove(
            		tagId, 
            		Ametys.message.MessageTarget.PROJECT_TAG,
            		"org.ametys.plugins.workspaces.tags.ProjectJCRTagsDAO",
            		"{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DELETE_TAG_LABEL}}",
            		"{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DELETE_TAG_CONFIRM}}"
    		);
        }
    },
    
    /**
     * Create a new tag, whose informations are returned to the callback afterwards. 
     * @param {String} parentId The parent id for tag creation
     * @param {Function} callback The callback function to called after tag creation or edition. Can be null. The arguments are:
     * @param {String} callback.tagId The id of created or edited tag
     * @param {String} callback.tagName The name of created or edited tag
     * @param {String} callback.tagParentId The id of parent tag. Null in edition mode.
     */
    create: function (parentId, callback)
    {
    	Ametys.plugins.workspaces.project.tag.ProjectTagActions._initFormPanel();
		
    	Ametys.plugins.cms.tag.TagHelper.add(
    			parentId,
    			Ametys.plugins.workspaces.project.tag.ProjectTagActions._formPanel,
				Ametys.message.MessageTarget.PROJECT_TAG,
				"org.ametys.plugins.workspaces.tags.ProjectJCRTagsDAO",
				null
		);
    },
    
	/**
	 * Init the form panel
	 * @private
	 */
	_initFormPanel: function()
    {
    	if (!this._formPanel)
    	{
    		this._formPanel = Ext.create('Ext.form.Panel', {
                border: false,
                defaultType :'textfield',
                
                defaults: {
                    cls: 'ametys',
                    labelWidth: 90,
                    labelAlign: 'right',
                    labelSeparator: '',
                    msgTarget: 'side',
                    width: 340
                },
                
                items: [{
                            xtype: 'hidden',
                            name: 'id'
                        },
                        {
                            xtype: 'hidden',
                            name: 'parentID'
                        },
                        {
                            xtype:'textfield',
                            name: 'title',
                            itemId: 'title',
                            fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_TITLE}}",
                            ametysDescription: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_TITLE_DESCRIPTION}}",
                            allowBlank: false,
                            enableKeyEvents: true,
                            listeners: {
                                'keyup': Ext.bind(Ametys.plugins.cms.tag.TagHelper.autoFillKey, this)
                            }
                        },
                        {
                            xtype: 'hidden',
                            name: 'name',
                            fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_NAME}}",
                            allowBlank: false,
                            regex: /^[A-Za-z0-9_-]+$/,
                            regexText: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_INVALID_REGEXP}}"
                        },
                        {
                        	xtype:'textarea',
                        	name: 'description',
                        	fieldLabel: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DESC}}",
                        	ametysDescription: "{{i18n plugin.cms:PLUGINS_CMS_HANDLE_TAGS_DESC_DESCRIPTION}}",
                        	height: 70
                        }
                        ]
    		});
    	}
    }
});