/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import Vue from 'vue';
import AmetysFront from 'AmetysFront';
import i18n from 'i18n';

import {callMethod} from '../../helper/ServerCommHelper';

export const namespaced = true;
export const state = {
  folder: {},
  folders: [],
  currentFolder: {},
  treeOpenedFolders: [],
  treeActiveFolder: []
};

export const mutations = {
  CREATE_FOLDER(state, folder) {
    if (state.currentFolder.id == folder.parentId)
    {
        if (state.currentFolder.hasOwnProperty('children')) {
          state.currentFolder.children.push(folder)
        } else {
          Vue.set(state.currentFolder, 'children', [folder])
        }
    
        state.folders = state.currentFolder.children
        
        AmetysFront.Event.fire('rerender-file-tree');
    }
  },
  SET_CURRENT_FOLDER(state, folder) {
    state.currentFolder = folder;
  },
  SET_FOLDERS(state, folders) {
    state.folders = folders;
  },
  SET_FOLDER(state, folder) {
    state.folder = folder;
  },
  SET_TREE_OPENED_FOLDER(state, folder) {
    state.treeOpenedFolders = folder;
  },
  SET_TREE_OPENED_FOLDERS(state, folders) {
    state.treeOpenedFolders = JSON.parse(JSON.stringify(folders));
  },
  ADD_TREE_OPENED_FOLDER(state, folder) {
    state.treeOpenedFolders.push(folder);
  },
  SET_TREE_ACTIVE_FOLDER(state, folder) {
    state.treeActiveFolder = [folder];
  },
  SET_TREE_ACTIVE_FOLDERS(state, folders) {
    state.treeActiveFolder =folders;
  }
}

export const actions = {

  /**
   * Expand folder in tree
   */
  async expandFolder(obj /*{commit , state, dispatch}*/, folder) {
    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });

        folder.children = data.children
    }
    catch(e)
    {
       AmetysFront.Event.fire('loaderFail', { 
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        }); 
    }
  },

  /**
   * Open folder from breadcrumb
   */
  async breadOpenFolder({commit, /*state,*/  }, folder) {

    try
    {
         await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });

        commit('SET_CURRENT_FOLDER', folder)
        commit('SET_FOLDERS', folder.children ? folder.children : []);
        commit('ADD_TREE_OPENED_FOLDER', folder);
        commit('SET_TREE_ACTIVE_FOLDER', folder);
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', { 
            title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
            text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
            details: null
        });
    }
  },

  /**
   * Open folder
   */
  async openFolder({commit, /* state,*/}, folder) {
    try
    {
        let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'getFoldersAndFiles',
            parameters: [folder.id]
        });

        if (data.hasOwnProperty('children')) 
        {
            let children = data.children;
            for (let child of children)
            {
                let f = folder.children.find(f => f.id == child.id);
                if (f)
                {
                    child.children = f.children;
                }
            }
            folder.children = children;
        } 
        else 
        {
            Vue.delete(folder, 'children')
        }

        commit('SET_CURRENT_FOLDER', folder)
        commit('SET_FOLDERS', folder.children ? folder.children : []);
        commit('ADD_TREE_OPENED_FOLDER', folder);
        commit('SET_TREE_ACTIVE_FOLDER', folder);

        AmetysFront.Event.fire('reset-filter');
        AmetysFront.Event.fire('hideMobileHeader');
    }
    catch(e)
    {
        AmetysFront.Event.fire('loaderFail', { 
          title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_ERROR_GET,
          text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
          details: null
        });

    }
  },

   /**
   * Move a folder
   */
    async moveFoldersAndFiles({commit, dispatch}, payload) {
      let success = false;

      AmetysFront.Event.fire('loaderStart', {
        text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_STARTED
      });
      
      try
      {
          let objectIds = payload.selectedFilesIds;
          for (let folderId of payload.selectedFoldersIds)
          {
              objectIds.push(folderId);
          }
          
          let data = await callMethod({
            role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
            methodName: 'moveObjects',
            parameters: [objectIds, payload.toFolderId]
          });
        
          if (data.message == 'locked')
          {
              AmetysFront.Event.fire('loaderFail', { 
                  title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                  text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_LOCKED_MESSAGE,
                  details: null
              });
          }
          else if (data.message == "already-exist")
          {
              AmetysFront.Event.fire('loaderFail', { 
                  title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                  text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_ALREADY_EXIST_MESSAGE + " " + data["unmoved-objects"],
                  details: null
              });
          }
          
          let movedObjects = data["moved-objects"];
          if (movedObjects && movedObjects.length > 0)
          {
              success = true;
              let params = {
                selectedFoldersIds : data["moved-folders"],
                selectedFilesIds : data["moved-files"],
                toFolderId : payload.toFolderId
              }
              
              setTimeout(() => {
                commit('folders/UPDATE_FOLDERS_AFTER_MOVE', params, {root:true});
                commit('files/UPDATE_FILES_AFTER_MOVE', data["moved-files"], {root:true});
                dispatch('selectedItems/unselectAll', null, {root:true});
                AmetysFront.Event.fire('loaderEnd', {
                  text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ENDED
                });
                
                return success;
              }, 1000);
          }
          else if (!data.message)
          {
              AmetysFront.Event.fire('loaderFail', { 
                  title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
                  text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_MESSAGE,
                  details: null
              });
          }
      }
      catch(e)
      {
           AmetysFront.Event.fire('loaderFail', { 
              title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_TITLE,
              text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FOLDER_AND_FILE_MOVE_ERROR_MESSAGE,
              details: null
          });
      }

      return success;
    },
    
    /**
     * Move a folder
     */
    async showCurrentFoldersStructure({commit, rootState}) {
        commit('SET_FOLDER', rootState.folders.folder)
        commit('SET_CURRENT_FOLDER', rootState.folders.currentFolder)
        commit('SET_TREE_OPENED_FOLDERS', rootState.folders.treeOpenedFolders);
        commit('SET_FOLDERS', rootState.folders.folders);
        commit('SET_TREE_ACTIVE_FOLDERS', rootState.folders.treeActiveFolder);
    }
}