/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.servlet.multipart.Part;

import org.ametys.cms.data.Binary;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.AttachableAmetysObject;
import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.ReactionableObject.ReactionType;
import org.ametys.cms.repository.comment.AbstractComment;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.cms.transformation.RichTextTransformer;
import org.ametys.cms.transformation.docbook.DocbookTransformer;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.right.RightManager.RightResult;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;
import org.ametys.plugins.workflow.support.WorkflowHelper;
import org.ametys.plugins.workflow.support.WorkflowProvider;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tags.ProjectTagsDAO;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Abstract class for workspace modules DAO's
 *
 */
public abstract class AbstractWorkspaceDAO extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{

    /** Ametys resolver */
    protected AmetysObjectResolver _resolver;

    /** Observer manager. */
    protected ObservationManager _observationManager;

    /** The current user provider. */
    protected CurrentUserProvider _currentUserProvider;

    /** The rights manager */
    protected RightManager _rightManager;

    /** User manager */
    protected UserManager _userManager;

    /** The workflow provider */
    protected WorkflowProvider _workflowProvider;

    /** The workflow helper */
    protected WorkflowHelper _workflowHelper;

    /** Workspaces project manager */
    protected ProjectManager _projectManager;

    /** The avalon context */
    protected Context _context;

    /** The workspace module EP */
    protected WorkspaceModuleExtensionPoint _workspaceModuleEP;

    /** The project tags DAO */
    protected ProjectTagsDAO _projectTagsDAO;

    /** The Workspaces helper */
    protected WorkspacesHelper _workspaceHelper;
    /** Rich text transformer */
    protected RichTextTransformer _richTextTransformer;

    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _workflowProvider = (WorkflowProvider) manager.lookup(WorkflowProvider.ROLE);
        _workflowHelper = (WorkflowHelper) manager.lookup(WorkflowHelper.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _projectTagsDAO = (ProjectTagsDAO) manager.lookup(ProjectTagsDAO.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
        _richTextTransformer = (RichTextTransformer) manager.lookup(DocbookTransformer.ROLE);
    }

    /**
     * Get the project name
     * @return the project name
     */
    protected String _getProjectName()
    {
        Request request = ContextHelper.getRequest(_context);
        return (String) request.getAttribute("projectName");
    }

    /**
     * Get the project
     * @return the project
     */
    protected Project _getProject()
    {
        return _projectManager.getProject(_getProjectName());
    }

    /**
     * Get the sitemap language
     * @return the sitemap language
     */
    protected String _getSitemapLanguage()
    {
        Request request = ContextHelper.getRequest(_context);
        return (String) request.getAttribute("sitemapLanguage");
    }

    /**
     * Get the site name
     * @return the site name
     */
    protected String _getSiteName()
    {
        Request request = ContextHelper.getRequest(_context);
        return (String) request.getAttribute("siteName");
    }

    /**
     * Check user rights
     * @param objectToCheck the object to check
     * @param rightId the right id
     * @throws IllegalAccessException if a right error occurred
     */
    protected void _checkUserRights(AmetysObject objectToCheck, String rightId) throws IllegalAccessException
    {
        if (_rightManager.hasRight(_currentUserProvider.getUser(), rightId, objectToCheck) != RightResult.RIGHT_ALLOW)
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to do operation without convenient right [" + rightId + "]");
        }
    }

    /**
     * Check user reading rights
     * @param objectToCheck the object to check
     * @throws IllegalAccessException if a right error occurred
     */
    protected void _checkUserReadingRights(AmetysObject objectToCheck) throws IllegalAccessException
    {
        if (!_rightManager.currentUserHasReadAccess(objectToCheck))
        {
            throw new IllegalAccessException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
    }

    /**
     * Handle tags for the edition of a TaggableAmetysObject
     * @param taggableAmetysObject the object to edit
     * @param tags the tags
     * @return the created tags
     */
    protected List<Map<String, Object>> _handleTags(TaggableAmetysObject taggableAmetysObject, List<Object> tags)
    {
        return _workspaceHelper.handleTags(taggableAmetysObject, tags);
    }

    /**
     * Edit the attachments of an AttachableAmetysObject
     * @param attachableAmetysObject the ametys object to edit
     * @param newFiles list of new files
     * @param newFileNames list of new files names
     * @param deleteFiles list of names of old files to delete
     */
    protected void _setAttachments(AttachableAmetysObject attachableAmetysObject, List<Part> newFiles, List<String> newFileNames, List<String> deleteFiles)
    {
        if (!newFiles.isEmpty() || !deleteFiles.isEmpty())
        {
            List<Binary> attachments = attachableAmetysObject.getAttachments()
                .stream()
                .filter(b -> !deleteFiles.contains(b.getName()))
                .collect(Collectors.toList());

            List<String> fileNames = attachments.stream()
                    .map(Binary::getFilename)
                    .collect(Collectors.toList());

            int i = 0;
            for (Part newPart : newFiles)
            {
                String newName = newFileNames.get(i);
                fileNames.add(newName);
                Binary newBinary = _partToBinary(newPart, newName);
                if (newBinary != null)
                {
                    attachments.add(newBinary);
                }
                i++;
            }
            attachableAmetysObject.setAttachments(attachments);
        }
    }

    private Binary _partToBinary(Part part, String name)
    {
        if (part.isRejected())
        {
            getLogger().error("Part {} will not be uploaded because it's rejected", part.getFileName());
            return null;
        }

        try (InputStream is = part.getInputStream())
        {
            Binary binary = new Binary();

            binary.setFilename(name);
            binary.setInputStream(is);
            binary.setLastModificationDate(ZonedDateTime.now());
            binary.setMimeType(part.getMimeType());

            return binary;
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred getting binary from part {}", part.getFileName(), e);
        }

        return null;
    }

    /**
     * Comment a commentableAmetysObject
     * @param <T> type of the value to retrieve
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentText the comment text
     * @param moduleRoot the module root
     * @return The commentableAmetysObject
     */
    public <T extends AbstractComment> T createComment(CommentableAmetysObject<T>  commentableAmetysObject, String commentText, ModifiableTraversableAmetysObject moduleRoot)
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();

        T comment = commentableAmetysObject.createComment();

        _setComment(comment, userIdentity, commentText);

        moduleRoot.saveChanges();

        return comment;
    }

    /**
     * Edit a commentableAmetysObject comment
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentId the comment Id
     * @param commentText the comment text
     * @param moduleRoot the module root
     * @return The commentableAmetysObject
     * @throws IllegalAccessException If an error occurs when checking the rights
     */
    public CommentableAmetysObject editComment(CommentableAmetysObject commentableAmetysObject, String commentId, String commentText, ModifiableTraversableAmetysObject moduleRoot) throws IllegalAccessException
    {
        UserIdentity userIdentity = _currentUserProvider.getUser();
        User user = _userManager.getUser(userIdentity);
        AbstractComment comment = commentableAmetysObject.getComment(commentId);
        String authorEmail = comment.getAuthorEmail();
        if (!authorEmail.equals(user.getEmail()))
        {
            throw new IllegalAccessException("User '" + userIdentity + "' tried to edit an other user's comment");
        }

        if (comment.getContent().equals(commentText))
        {
            return commentableAmetysObject;
        }

        comment.setContent(commentText);
        if (comment instanceof RichTextComment richTextComment)
        {
            _setRichTextContent(commentText, richTextComment);
        }
        comment.setEdited(true);

        moduleRoot.saveChanges();
        return commentableAmetysObject;
    }

    private void _setRichTextContent(String commentText, RichTextComment richTextComment)
    {
        RichText richText = richTextComment.getRichTextContent();
        if (richText == null)
        {
            richText = new RichText();
        }

        try
        {
            _richTextTransformer.transform(commentText, richText);
        }
        catch (AmetysRepositoryException | IOException e)
        {
            throw new AmetysRepositoryException("Unable to transform the text " + commentText + " into a rich text for comment " + richTextComment.getId(), e);
        }

        richTextComment.setRichTextContent(richText);
    }

    /**
     * Edit a commentableAmetysObject comment
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentId the comment Id
     * @param moduleRoot the module root
     * @return The commentableAmetysObject
     */
    public CommentableAmetysObject deleteComment(CommentableAmetysObject commentableAmetysObject, String commentId, ModifiableTraversableAmetysObject moduleRoot)
    {
        AbstractComment comment = commentableAmetysObject.getComment(commentId);

        if (comment.isSubComment())
        {
            AbstractComment parentComment = comment.getCommentParent();
            List<AbstractComment> subComments = parentComment.getSubComment(true, true);
            boolean hasAfterSubComments = _hasAfterSubComments(comment, subComments);
            if (comment.hasSubComments() || hasAfterSubComments)
            {
                comment.setDeleted(true);
                comment.setAccepted(false);
            }
            else
            {
                comment.remove();
            }

            // Sort comment by creation date (Recent creation date in first)
            List<AbstractComment> currentSubComments = parentComment.getSubComment(true, true);
            Collections.sort(currentSubComments, (c1, c2) ->
            {
                return c2.getCreationDate().compareTo(c1.getCreationDate());
            });

            // Remove already deleted sub comment if no recent sub comment is present
            for (AbstractComment subCom : currentSubComments)
            {
                if (subCom.isDeleted() && !subCom.hasSubComments())
                {
                    subCom.remove();
                }
                else
                {
                    break;
                }
            }

            // Remove parent comment
            if (parentComment.isDeleted())
            {
                deleteComment(commentableAmetysObject, parentComment.getId(), moduleRoot);
            }
        }
        else
        {
            List<AbstractComment> subComment = comment.getSubComment(true, true);
            boolean hasSubComments = subComment.stream()
                .filter(c -> !c.isDeleted()) // Ignore alreay deleted sub comment
                .findAny()
                .isPresent();
            if (hasSubComments)
            {
                comment.setDeleted(true);
                comment.setAccepted(false);
            }
            else
            {
                comment.remove();
            }
        }

        moduleRoot.saveChanges();

        return commentableAmetysObject;
    }

    /**
     * Check if a given comment should be deleted or removed
     * @param comment the comment
     * @param subComments the subcomments
     * @return true if the comment should be deleted, false if the comment should be removed instead
     */
    protected boolean _hasAfterSubComments(AbstractComment comment, List<AbstractComment> subComments)
    {
        boolean hasAfterSubComments = subComments.stream()
            .filter(c -> !c.getId().equals(comment.getId())) //Don't take current sub comment
            .filter(c -> !c.isDeleted()) // Ignore alreay deleted sub comment
            .filter(c -> c.getCreationDate().isAfter(comment.getCreationDate())) // Just take sub comment after current comment
            .findAny()
            .isPresent();
        return hasAfterSubComments;
    }

    /**
     * Answer to a commentableAmetysObject comment
     * @param <T> type of the value to retrieve
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentId the parent comment Id
     * @param commentText the comment text
     * @param moduleRoot the module root
     * @return The commentableAmetysObject
     */
    public <T extends AbstractComment> T answerComment(CommentableAmetysObject<T> commentableAmetysObject, String commentId, String commentText, ModifiableTraversableAmetysObject moduleRoot)
    {
        AbstractComment comment = commentableAmetysObject.getComment(commentId);

        UserIdentity userIdentity = _currentUserProvider.getUser();
        T subComment = comment.createSubComment();

        _setComment(subComment, userIdentity, commentText);

        moduleRoot.saveChanges();
        return subComment;
    }

    private void _setComment(AbstractComment comment, UserIdentity userIdentity, String commentText)
    {
        User user = _userManager.getUser(userIdentity);
        comment.setAuthorName(user.getFullName());
        comment.setAuthorEmail(user.getEmail());
        comment.setAuthor(userIdentity);
        comment.setEmailHiddenStatus(true);
        comment.setContent(commentText);
        if (comment instanceof RichTextComment richTextComment)
        {
            _setRichTextContent(commentText, richTextComment);
        }
        comment.setValidated(true);
    }

    /**
     * Answer to a commentableAmetysObject comment
     * @param commentableAmetysObject the commentableAmetysObject
     * @param commentId the parent comment Id
     * @param liked true if the comment is liked, otherwise the comment is unliked
     * @param moduleRoot the module root
     * @return The commentableAmetysObject
     */
    public CommentableAmetysObject likeOrUnlikeComment(CommentableAmetysObject commentableAmetysObject, String commentId, Boolean liked, ModifiableTraversableAmetysObject moduleRoot)
    {
        AbstractComment comment = commentableAmetysObject.getComment(commentId);

        UserIdentity user = _currentUserProvider.getUser();
        if (Boolean.FALSE.equals(liked)
            || liked == null && comment.getReactionUsers(ReactionType.LIKE).contains(user))
        {
            comment.removeReaction(user, ReactionType.LIKE);
        }
        else
        {
            comment.addReaction(user, ReactionType.LIKE);
        }

        moduleRoot.saveChanges();
        return commentableAmetysObject;
    }
}
