/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.documents;

import org.apache.commons.lang3.StringUtils;

import org.ametys.plugins.repository.data.repositorydata.ModifiableRepositoryData;
import org.ametys.plugins.repository.data.repositorydata.RepositoryData;
import org.ametys.plugins.repository.data.type.ComplexRepositoryElementType;
import org.ametys.plugins.workspaces.activities.documents.AbstractResourceReferenceElementType.ResourceReference;

/**
 * Implementation for storing ResourceReference in the repository
 */
public class ResourceReferenceRepositoryElementType extends AbstractResourceReferenceElementType implements ComplexRepositoryElementType<ResourceReference>
{
    private static final String RESOURCE_REFERENCE_NODETYPE = "ametys:resourceReference";

    public String getRepositoryDataType()
    {
        return RESOURCE_REFERENCE_NODETYPE;
    }

    public boolean isSingleValueEmpty(RepositoryData singleValueData)
    {
        return _isStringValueEmpty(singleValueData, ResourceReference.ID) 
            || _isStringValueEmpty(singleValueData, ResourceReference.NAME) 
            || _isStringValueEmpty(singleValueData, ResourceReference.MIME_TYPE);
    }
    
    private boolean _isStringValueEmpty(RepositoryData referenceData, String name)
    {
        return !referenceData.hasValue(name) || StringUtils.isEmpty(referenceData.getString(name));
    }

    public ResourceReference readSingleValue(RepositoryData singleValueData)
    {
        String id = singleValueData.getString(ResourceReference.ID);
        String name = singleValueData.getString(ResourceReference.NAME);
        String oldName = null;
        if (singleValueData.hasValue(ResourceReference.OLD_NAME))
        {
            oldName = singleValueData.getString(ResourceReference.OLD_NAME);
        }
        String mimeType = singleValueData.getString(ResourceReference.MIME_TYPE);
        String version = null;
        if (singleValueData.hasValue(ResourceReference.VERSION))
        {
            version = singleValueData.getString(ResourceReference.VERSION);
        }
        return new ResourceReference(id, name, oldName, mimeType, version);
    }

    public void writeSingleValue(ModifiableRepositoryData parentData, String name, ResourceReference value)
    {
        if (value != null)
        {
            ModifiableRepositoryData referenceData = parentData.addRepositoryData(name, getRepositoryDataType());
            referenceData.setValue(ResourceReference.ID, value.id());
            referenceData.setValue(ResourceReference.NAME, value.name());
            String oldName = value.oldName();
            if (oldName != null)
            {
                referenceData.setValue(ResourceReference.OLD_NAME, value.oldName());
            }
            referenceData.setValue(ResourceReference.MIME_TYPE, value.mimeType());
            String baseVersionName = value.baseVersionName();
            if (baseVersionName != null)
            {
                referenceData.setValue(ResourceReference.VERSION, baseVersionName);
            }
            
        }
        
    }

}
