/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.RepositoryException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.plugins.messagingconnector.MessagingConnector;
import org.ametys.plugins.messagingconnector.MessagingConnector.AttendeeInformation;
import org.ametys.plugins.messagingconnector.MessagingConnector.FreeBusyStatus;
import org.ametys.plugins.messagingconnector.MessagingConnector.ResponseType;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.workspaces.calendars.events.CalendarEventAttendee;
import org.ametys.plugins.workspaces.calendars.jcr.JCRCalendarEvent;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;

/**
 * Manager for manipulating messaging connector linked to calendars event of a project
 *
 */
public class MessagingConnectorCalendarManager extends AbstractLogEnabled implements Serviceable, Component, Contextualizable
{
    /** Avalon Role */
    public static final String ROLE = MessagingConnectorCalendarManager.class.getName();
    
    /** Property's name for synchronized id id */
    public static final String PROPERTY_SYNCHRONIZED_ID = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":synchronizedId";
    
    private MessagingConnector _messagingConnector;
    private UserManager _userManager;
    private AmetysObjectResolver _resolver;
    private ObservationManager _observationManager;
    private CurrentUserProvider _currentUserProvider;
    private ProjectManager _projectManager;
    private I18nUtils _i18nUtils;
    private Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        boolean hasService = manager.hasService(MessagingConnector.ROLE);
        _messagingConnector = hasService ? (MessagingConnector) manager.lookup(MessagingConnector.ROLE) : null;

        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
    }
    
    /**
     * Set the synchronized id (for messaging connector) to the event
     * @param event the event
     * @param synchronizedId the synchronized id
     */
    public void setSynchronizedId(JCRCalendarEvent event, String synchronizedId)
    {
        try
        {
            event.getNode().setProperty(PROPERTY_SYNCHRONIZED_ID, synchronizedId);
            event.saveChanges();
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred setting synchronized id for event " + event.getId());
        }
    }
    
    /**
     * Get the synchronized id (for messaging connector) of the event
     * @param event the event
     * @return the synchronized id
     */
    public String getSynchronizedId(JCRCalendarEvent event)
    {
        try
        {
            Node eventNode = event.getNode();
            if (eventNode.hasProperty(PROPERTY_SYNCHRONIZED_ID))
            {
                return eventNode.getProperty(PROPERTY_SYNCHRONIZED_ID).getString();
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting synchronized id for event " + event.getId());
        }
        
        return null;
    }
    
    /**
     * Set attendees to the event 
     * @param eventId the event id
     * @param attendees the list of attendees
     */
    public void setAttendees(String eventId, List<CalendarEventAttendee> attendees)
    {
        if (attendees != null)
        {
            JCRCalendarEvent event = (JCRCalendarEvent) _resolver.resolveById(eventId);
            try
            {
                event.setAttendees(attendees);
            }
            catch (RepositoryException e)
            {
                getLogger().error("An error occurred setting attendees for event " + eventId, e);
            }
        }
    }
    
    /**
     * Set organiser to the event 
     * @param eventId the event id
     * @param organiser the organiser
     */
    public void setOrganiser(String eventId, UserIdentity organiser)
    {
        JCRCalendarEvent event = (JCRCalendarEvent) _resolver.resolveById(eventId);
        if (organiser != null)
        {
            event.setOrganiser(organiser);
        }
    }
    
    /**
     * Add event invitation parameters (attendees and organiser)
     * @param parameters the map of parameters
     * @param eventId the event id
     */
    public void addEventInvitation(Map<String, Object> parameters, String eventId)
    {
        @SuppressWarnings("unchecked")
        List<Map<String, Object>> attendeesList = (List<Map<String, Object>>) parameters.get("attendees");
        List<CalendarEventAttendee> attendees = _getAttendeeListFromParameter(attendeesList);
        setAttendees(eventId, attendees);
        
        @SuppressWarnings("unchecked")
        Map<String, Object> organiserMap = (Map<String, Object>) parameters.get("organiser");
        UserIdentity organiser = _getUserFromParameter(organiserMap);
        setOrganiser(eventId, organiser);
        
        if (organiser != null)
        {
            JCRCalendarEvent event = _resolver.resolveById(eventId);
            _createEvent(event, organiser, attendees);
        }
    }
    
    /**
     * Edit event invitation parameters (attendees and organiser)
     * @param parameters the map of parameters
     * @param eventId the event id
     */
    public void editEventInvitation(Map<String, Object> parameters, String eventId)
    {
        JCRCalendarEvent event = _resolver.resolveById(eventId);

        @SuppressWarnings("unchecked")
        List<Map<String, Object>> attendeesList = (List<Map<String, Object>>) parameters.get("attendees");
        List<CalendarEventAttendee> attendees = _getAttendeeListFromParameter(attendeesList);
        try
        {
            if (attendees == null)
            {
                attendees = event.getAttendees();
            }
        }
        catch (RepositoryException e)
        {
            getLogger().error("An error occurred getting attendees from repository for event " + event.getId());
        }
        setAttendees(eventId, attendees);

        @SuppressWarnings("unchecked")
        Map<String, Object> organiserMap = (Map<String, Object>) parameters.get("organiser");
        UserIdentity organiserFromMap = _getUserFromParameter(organiserMap);
        UserIdentity organiser = organiserFromMap != null ? organiserFromMap : event.getOrganiser();
        setOrganiser(eventId, organiser);
        
        if (_messagingConnector != null)
        {
            String synchronizedId = getSynchronizedId(event);
            if (StringUtils.isNotBlank(synchronizedId) && _messagingConnector.isEventExist(synchronizedId, organiser))
            {
                _editEvent(event, organiser, attendees);
            }
            else if (organiser != null)
            {
                _createEvent(event, organiser, attendees);
            }
        }
    }
    
    /**
     * Create event in the messaging connector
     * @param event the event
     * @param organiser the organiser
     * @param attendees the list of attendee
     */
    protected void _createEvent(JCRCalendarEvent event, UserIdentity organiser, List<CalendarEventAttendee> attendees)
    {
        Map<String, Boolean> attendeesForMessagingConnector = new HashMap<>();
        for (CalendarEventAttendee attendee : attendees)
        {
            if (attendee.isExternal())
            {
                attendeesForMessagingConnector.put(attendee.getEmail(), attendee.isMandatory());
            }
            else
            {
                User user = _userManager.getUser(new UserIdentity(attendee.getLogin(), attendee.getPopulationId()));
                attendeesForMessagingConnector.put(user.getEmail(), attendee.isMandatory());
            }
        }
        
        String synchronizedId = _messagingConnector.createEvent(_getEventTitle(event), event.getDescription(), event.getLocation(), event.getFullDay(), event.getStartDate(), event.getEndDate(), event.getRecurrenceType(), event.getRepeatUntil(), attendeesForMessagingConnector, event.getOrganiser());
        setSynchronizedId(event, synchronizedId);
    }
    
    /**
     * Delete event in the messaging connector
     * @param event the event
     */
    public void deleteEvent(JCRCalendarEvent event)
    {
        if (isEventSynchronized(event.getId()))
        {
            String synchronizedId = getSynchronizedId(event);
            _messagingConnector.deleteEvent(synchronizedId, event.getOrganiser());
        }
    }
    
    /**
     * Edit event in the messaging connector
     * @param event the event 
     * @param organiser the organiser
     * @param attendees the list of attendee
     */
    protected void _editEvent(JCRCalendarEvent event, UserIdentity organiser, List<CalendarEventAttendee> attendees)
    {
        String synchronizedId = getSynchronizedId(event);
        Map<String, Boolean> attendeesForMessagingConnector = null;
        if (attendees != null)
        {
            attendeesForMessagingConnector = new HashMap<>();
            for (CalendarEventAttendee attendee : attendees)
            {
                if (attendee.isExternal())
                {
                    attendeesForMessagingConnector.put(attendee.getEmail(), attendee.isMandatory());
                }
                else
                {
                    User user = _userManager.getUser(new UserIdentity(attendee.getLogin(), attendee.getPopulationId()));
                    attendeesForMessagingConnector.put(user.getEmail(), attendee.isMandatory());
                }
            }
        }
        
        _messagingConnector.updateEvent(synchronizedId, _getEventTitle(event), event.getDescription(), event.getLocation(), event.getFullDay(), event.getStartDate(), event.getEndDate(), event.getRecurrenceType(), event.getRepeatUntil(), attendeesForMessagingConnector, event.getOrganiser());
    }
    
    /**
     * Get the computed title of the event
     * @param event the event
     * @return the computed title of the event
     */
    protected String _getEventTitle(JCRCalendarEvent event)
    {
        Calendar calendar = event.getParent();
        Project project = _projectManager.getParentProject(calendar);
        
        List<String> parameters = new ArrayList<>();
        parameters.add(project.getTitle());
        parameters.add(event.getTitle());
        I18nizableText titleI18n = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_EVENT_MESSAGING_CONNECTOR_TITLE", parameters);
        
        Request request = ContextHelper.getRequest(_context);
        String lang = (String) request.getAttribute("sitemapLanguage");
        
        return _i18nUtils.translate(titleI18n, lang);
    }
    
    /**
     * True if messaging connector support invitation
     * @return true if messaging connector support invitation
     */
    @Callable
    public boolean supportInvitation()
    {
        return _messagingConnector != null && _messagingConnector.supportInvitation();
    }
    
    /**
     * True if user exists in the messaging connector
     * @param userAsMap the user
     * @return true if user exists in the messaging connector
     */
    @Callable
    public boolean isValidUser(Map<String, Object> userAsMap)
    {
        UserIdentity user = _getUserFromParameter(userAsMap);
        return _messagingConnector != null && _messagingConnector.isUserExist(user);
    }
        
    /**
     * True if the event is synchronized in the messaging connector
     * @param eventId the event id
     * @return true if the event exist
     */
    @Callable
    public boolean isEventSynchronized(String eventId)
    {
        if (_messagingConnector == null)
        {
            return false;
        } 
        JCRCalendarEvent event = _resolver.resolveById(eventId);
        String synchronizedId = getSynchronizedId(event);
        
        if (StringUtils.isBlank(synchronizedId))
        {
            return false;
        }
        
        try
        {
            return _messagingConnector.isEventExist(synchronizedId, event.getOrganiser());
        }
        catch (UnsupportedOperationException e)
        {
            getLogger().error("An error occurred while checking if event " + eventId + " exists", e);
            return false;
        }
        
        
    }
    
    /**
     * Get the list of attendees email with their free/busy status
     * @param startDate the start date
     * @param endDate the end date
     * @param isAllDay true if is all day
     * @param organiserMap the organiser map
     * @param attendeesList the attendee list
     * @return the list of attendees email with their free/busy status
     */
    @Callable
    public Map<String, FreeBusyStatus> getFreeBusy(String startDate, String endDate, boolean isAllDay, Map<String, Object> organiserMap, Map<String, String> attendeesList)
    {
        Map<String, FreeBusyStatus> userIdFreeBusy = new HashMap<>();

        if (_messagingConnector != null)
        {
            Date start = DateUtils.parse(startDate);
            Date end = DateUtils.parse(endDate);
            
            UserIdentity organiser = _getUserFromParameter(organiserMap);
            
            Map<String, FreeBusyStatus> freeBusy = _messagingConnector.getFreeBusy(start, end, isAllDay, attendeesList.keySet(), organiser);
            for (String email : freeBusy.keySet())
            {
                userIdFreeBusy.put(attendeesList.get(email), freeBusy.get(email));
            }
        }
        
        return userIdFreeBusy;
    }
    
    /**
     * Set invitation to the event 
     * @param eventId the event id
     * @param organiserMap the organiser map
     * @param attendeesList the attendees list 
     */
    @Callable
    public void setInvitations(String eventId, Map<String, Object> organiserMap, List<Map<String, Object>> attendeesList)
    {
        UserIdentity organiser = _getUserFromParameter(organiserMap);
        setOrganiser(eventId, organiser);

        List<CalendarEventAttendee> attendees = _getAttendeeListFromParameter(attendeesList);
        setAttendees(eventId, attendees);
        
        JCRCalendarEvent event = _resolver.resolveById(eventId);
        event.saveChanges();
        
        if (_messagingConnector != null)
        {
            String synchronizedId = getSynchronizedId(event);
            if (StringUtils.isNotBlank(synchronizedId) && _messagingConnector.isEventExist(synchronizedId, organiser))
            {
                
                Map<String, Boolean> attendeesForMessagingConnector = new HashMap<>();
                for (CalendarEventAttendee attendee : attendees)
                {
                    if (attendee.isExternal())
                    {
                        attendeesForMessagingConnector.put(attendee.getEmail(), attendee.isMandatory());
                    }
                    else
                    {
                        User user = _userManager.getUser(new UserIdentity(attendee.getLogin(), attendee.getPopulationId()));
                        attendeesForMessagingConnector.put(user.getEmail(), attendee.isMandatory());
                    }
                }
                _messagingConnector.setAttendees(synchronizedId, attendeesForMessagingConnector, organiser);
            }
            else
            {
                _createEvent(event, organiser, attendees);
            }
        }
        
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_CALENDAR, event.getParent());
        eventParams.put(ObservationConstants.ARGS_CALENDAR_EVENT, event);
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_ID, event.getId());
        eventParams.put(org.ametys.plugins.explorer.ObservationConstants.ARGS_PARENT_ID, event.getParent().getId());
        
        _observationManager.notify(new Event(ObservationConstants.EVENT_CALENDAR_EVENT_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
    
    /**
     * Retrieves attendees to the event 
     * @param eventId the event id
     * @return The map of attendees 
     */
    @Callable
    public Map<String, Object> getAttendees(String eventId)
    {
        JCRCalendarEvent event = _resolver.resolveById(eventId);
        Map<String, AttendeeInformation> attendees = new HashMap<>();
        
        String synchronizedId = getSynchronizedId(event);
        if (_messagingConnector != null && StringUtils.isNotBlank(synchronizedId))
        {
            attendees = _messagingConnector.getAttendees(synchronizedId, event.getOrganiser());
        }
        
        Map<String, Object> result = new HashMap<>();
        List<Map<String, Object>> attendeesMap = new ArrayList<>();
        
        try
        {
            for (CalendarEventAttendee attendee : event.getAttendees())
            {
                Map<String, Object> attendee2Json = new HashMap<>();
                if (!attendee.isExternal())
                {
                    UserIdentity userIdentity = new UserIdentity(attendee.getLogin(), attendee.getPopulationId());
                    User user = _userManager.getUser(userIdentity);
                    
                    attendee2Json.put("login", attendee.getLogin());
                    attendee2Json.put("populationId", attendee.getPopulationId());
                    
                    attendee2Json.put("sortablename", user.getSortableName());
                    attendee2Json.put("fullname", user.getFullName());
                    
                    String email = user.getEmail();
                    attendee2Json.put("email", email);
                    AttendeeInformation attendeeInformation = attendees.get(email);
                    if (attendeeInformation != null)
                    {
                        attendee2Json.put("status", attendeeInformation.getResponseType());
                    }
                    else
                    {
                        attendee2Json.put("status", ResponseType.Unknown);
                    }
                }
                else
                {
                    String email = attendee.getEmail();
                    attendee2Json.put("email", email);
                    attendee2Json.put("sortablename", email);
                    attendee2Json.put("fullname", email);
                    AttendeeInformation attendeeInformation = attendees.get(email);
                    if (attendeeInformation != null)
                    {
                        attendee2Json.put("status", attendeeInformation.getResponseType());
                    }
                    else
                    {
                        attendee2Json.put("status", ResponseType.Unknown);
                    }
                }
                
                attendee2Json.put("external", attendee.isExternal());
                attendee2Json.put("mandatory", attendee.isMandatory());
                
                attendeesMap.add(attendee2Json);
            }
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting attendees for event " + eventId, e);
        }
        
        result.put("attendees", attendeesMap);
        
        return result;
    }
    
    /**
     * Retrieves organiser to the event 
     * @param eventId the event id
     * @return The organiser 
     */
    @Callable
    public Map<String, Object> getOrganiser(String eventId)
    {
        Map<String, Object> result = new HashMap<>();
        Map<String, Object> organiserMap = new HashMap<>();
        
        try
        {
            JCRCalendarEvent event = (JCRCalendarEvent) _resolver.resolveById(eventId);
            UserIdentity organiserIdentity = event.getOrganiser();
            
            organiserMap.put("login", organiserIdentity.getLogin());
            organiserMap.put("populationId", organiserIdentity.getPopulationId());
            
            User organiser = _userManager.getUser(organiserIdentity);
            organiserMap.put("sortablename", organiser.getSortableName());
            organiserMap.put("fullname", organiser.getFullName());
        }
        catch (Exception e) 
        {
            getLogger().error("An error occurred getting organiser for event " + eventId, e);
        }
        
        result.put("user", organiserMap);
        
        return result;
    }
    
    /**
     * Get user from the user parameter
     * @param userMap the user map from parameters
     * @return the user
     */
    protected UserIdentity _getUserFromParameter(Map<String, Object> userMap)
    {
        if (userMap != null)
        {
            String login = (String) userMap.get("login");
            String populationId = (String) userMap.get("populationId");
            
            return new UserIdentity(login, populationId);
        }
        
        return null;
    }
    
    /**
     * Get attendees list from the attendees parameter
     * @param attendeesList the attendees list from parameters
     * @return the attendees list
     */
    protected List<CalendarEventAttendee> _getAttendeeListFromParameter(List<Map<String, Object>> attendeesList)
    {
        if (attendeesList != null)
        {
            List<CalendarEventAttendee> attendees = new ArrayList<>();
            for (Map<String, Object> attendee : attendeesList)
            {
                CalendarEventAttendee calendarEventAttendee = new CalendarEventAttendee();
                boolean isExternal = (boolean) attendee.get("external");
                if (isExternal)
                {
                    calendarEventAttendee.setEmail((String) attendee.get("email"));
                }
                else
                {
                    calendarEventAttendee.setPopulationId((String) attendee.get("populationId"));
                    calendarEventAttendee.setLogin((String) attendee.get("login"));
                }
                
                calendarEventAttendee.setIsExternal(isExternal);
                calendarEventAttendee.setIsMandatory((boolean) attendee.get("mandatory"));
                
                attendees.add(calendarEventAttendee);
            }
            return attendees;
        }
        
        return null;
    }
    
}
