/*
 *  Copyright 2015 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.helper;

import java.time.LocalDate;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.runtime.config.Config;

/**
 * Helper for recurrent event operation
 */
public final class RecurrentEventHelper
{
    private RecurrentEventHelper()
    {
        // Helper class
    }
    
    /**
     * Get the next occurrence date
     * @param recurrenceType recurrence type
     * @param eventStartDate the event start date
     * @param date the occurrence date
     * @return the next occurrence date
     */
    public static ZonedDateTime getNextDate(EventRecurrenceTypeEnum recurrenceType, ZonedDateTime eventStartDate, ZonedDateTime date)
    {
        ZonedDateTime nextDate = null;
        switch (recurrenceType)
        {
            case ALL_DAY:
                nextDate = _nextAllDayDate(date);
                break;
            case ALL_WORKING_DAY:
                nextDate = _nextAllWorkingDayDate(date);
                break;
            case WEEKLY:
                nextDate = _nextWeeklyDate(date);
                break;
            case BIWEEKLY:
                nextDate = _nextBiweeklyDate(date);
                break;
            case MONTHLY:
                nextDate = _nextMonthlyDate(date, eventStartDate);
                break;
            default:
                break;
        }
        
        return nextDate;
    }
    
    private static ZonedDateTime _nextAllDayDate(ZonedDateTime date)
    {
        return date.plusDays(1);
    }
    
    /**
     * get the list of working days
     * @return a list of integer, matching {@link java.util.Calendar} days
     */
    public static List<Integer> getWorkingDays()
    {
        List<Integer> result = new ArrayList<>();
        String workingDayAsString = Config.getInstance().getValue("workspaces.calendar.event.working.day");
        List<String> workingDaysStrings = Arrays.asList(workingDayAsString.split(","));
        for (String workkingDayString : workingDaysStrings)
        {
            result.add(Integer.valueOf(workkingDayString));
        }
        return result;
    }

    private static ZonedDateTime _nextAllWorkingDayDate(ZonedDateTime date)
    {
        String workingDayAsString = Config.getInstance().getValue("workspaces.calendar.event.working.day");
        
        int dayId = date.getDayOfWeek().getValue() + 1;
        
        int nbDay = 1;
        int nextDayId = (dayId % 7) + 1;
        List<String> workingDay = Arrays.asList(workingDayAsString.split(","));
        while (!workingDay.contains(String.valueOf(nextDayId)))
        {
            nextDayId = (nextDayId % 7) + 1;
            nbDay++;
        }
        
        return date.plusDays(nbDay);
    }
    
    private static ZonedDateTime _nextWeeklyDate(ZonedDateTime date)
    {
        return  date.plusWeeks(1);
    }
    
    private static ZonedDateTime _nextBiweeklyDate(ZonedDateTime date)
    {
        return  date.plusWeeks(2);
    }
    
    private static ZonedDateTime _nextMonthlyDate(ZonedDateTime date, ZonedDateTime eventStartDate)
    {
        int dayOfMonth = eventStartDate.getDayOfMonth();
        
        ZonedDateTime nextDate = date.plusMonths(1);
        int nextDayOfMonth = nextDate.getDayOfMonth();
        LocalDate nextDateAsLocal = nextDate.toLocalDate();
        int nextDayMaxOfMonth = nextDateAsLocal.lengthOfMonth();

        if (nextDayOfMonth < dayOfMonth)
        {
            if (dayOfMonth < nextDayMaxOfMonth)
            {
                nextDate = nextDate.withDayOfMonth(dayOfMonth);
            }
            else
            {
                nextDate = nextDate.withDayOfMonth(nextDayMaxOfMonth);
            }
        }
        
        return nextDate;
    }
    
    /**
     * Compute the occurrence of an event
     * @param startDate the start of the range to compute occurrences
     * @param endDate the end of the range to compute occurrences
     * @param eventStartDate the start date of the occurrence
     * @param originalOccurrenceStartDate the original start date of the occurrence (different as the eventStartDate when editing an occurrence)
     * @param recurrenceType the recurrence type
     * @param excludedOccurences the occurrences to exclude
     * @param zoneId the zoneId used for the dates
     * @param untilDate until date of the recurring event
     * @return a list of occurrence start dates
     */
    public static List<ZonedDateTime> getOccurrences(ZonedDateTime startDate, ZonedDateTime endDate, ZonedDateTime eventStartDate, ZonedDateTime originalOccurrenceStartDate, EventRecurrenceTypeEnum recurrenceType, List<ZonedDateTime> excludedOccurences, ZoneId zoneId, ZonedDateTime untilDate)
    {
        long diffInSeconds = ChronoUnit.SECONDS.between(originalOccurrenceStartDate, eventStartDate);
        List<ZonedDateTime> occurences = new ArrayList<>();
        
        ZonedDateTime firstDate = startDate.equals(eventStartDate) ? startDate : startDate.plusSeconds(diffInSeconds);
        firstDate = firstDate.withZoneSameInstant(zoneId);

        if (firstDate.isAfter(endDate))
        {
            return occurences;
        }
        
        ZonedDateTime firstDateCalendar = firstDate.withZoneSameInstant(zoneId).truncatedTo(ChronoUnit.DAYS);
        
        if (excludedOccurences.stream().noneMatch(excludedOccurrence -> excludedOccurrence.isEqual(firstDateCalendar)))
        {
            occurences.add(firstDate);
        }
        
        ZonedDateTime nextDate = getNextDate(recurrenceType, eventStartDate.withZoneSameInstant(zoneId), firstDate.withZoneSameInstant(zoneId));
        while (nextDate != null && nextDate.isBefore(endDate) && (untilDate == null || untilDate.isAfter(nextDate)))
        {
            ZonedDateTime nextDateCalendar = nextDate.truncatedTo(ChronoUnit.DAYS);
            
            if (excludedOccurences.stream().noneMatch(excludedOccurrence -> excludedOccurrence.isEqual(nextDateCalendar)))
            {
                occurences.add(nextDate);
            }
            nextDate = getNextDate(recurrenceType, eventStartDate.withZoneSameInstant(zoneId), nextDate.withZoneSameInstant(zoneId));
        }
        
        return occurences;
    }
}
