/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.chat;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.charset.StandardCharsets;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.codec.digest.Sha2Crypt;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.http.Consts;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.entity.ContentType;
import org.apache.http.entity.StringEntity;
import org.apache.http.entity.mime.HttpMultipartMode;
import org.apache.http.entity.mime.MultipartEntityBuilder;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.apache.poi.util.IOUtils;
import org.apache.tika.Tika;

import org.ametys.cms.repository.Content;
import org.ametys.core.authentication.AuthenticateAction;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.CryptoHelper;
import org.ametys.core.util.JSONUtils;
import org.ametys.core.util.URIUtils;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for chat
 */
public class ChatHelper extends AbstractLogEnabled implements Component, Serviceable, Contextualizable
{
    /** The Avalon role */
    public static final String ROLE = ChatHelper.class.getName();
    
    private static final String __USERPREF_PREF_PASSWORD = "workspaces.chat-connector-password";
    
    private static final String __USERPREF_PREF_TOKEN = "workspaces.chat-connector-token";

    private static final String __USERPREF_PREF_ID = "workspaces.chat-connector-id";

    private static final String __USERPREF_CONTEXT = "/workspaces.chat-connector";

    /** Rocket.Chat admin ID */
    private static final String CONFIG_ADMIN_ID = "workspaces.chat.rocket.admin.id";

    /** Rocket.Chat admin Token */
    private static final String CONFIG_ADMIN_TOKEN = "workspaces.chat.rocket.admin.token";
    
    /** Rocket.Chat URL */
    private static final String CONFIG_URL = "workspaces.chat.rocket.url";
    
    /** JSON Utils */
    protected JSONUtils _jsonUtils;
    
    /** User Manager */
    protected UserManager _userManager;
    
    /** User Preferences */
    protected UserPreferencesManager _userPreferencesManager;
    
    /** Cryptography */
    protected CryptoHelper _cryptoHelper;

    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;

    private SourceResolver _sourceResolver;

    private UserDirectoryHelper _userDirectoryHelper;

    private Context _context;
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _userPreferencesManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _cryptoHelper = (CryptoHelper) manager.lookup("org.ametys.plugins.workspaces.chat.cryptoHelper");
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) manager.lookup(UserDirectoryHelper.ROLE);
    }

    private Map<String, Object> _doGet(String api, Map<String, String> parameters) throws IOException
    {
        return _doGet(api, parameters, Config.getInstance().getValue(CONFIG_ADMIN_TOKEN), Config.getInstance().getValue(CONFIG_ADMIN_ID));
    }
    
    private Map<String, Object> _doGet(String api, Map<String, String> parameters, String authToken, String userId) throws IOException
    {
        String path = Config.getInstance().getValue(CONFIG_URL) + "/api/" + api;

        String uri = URIUtils.encodeURI(path, parameters);
        HttpGet request = new HttpGet(uri);
        request.setHeader("Content-Type", "application/json");

        return _execRequest(request, authToken, userId);
    }

    private Map<String, Object> _doPOST(String api, Map<String, Object> parameters) throws IOException
    {
        return _doPOST(api, parameters, Config.getInstance().getValue(CONFIG_ADMIN_TOKEN), Config.getInstance().getValue(CONFIG_ADMIN_ID));
    }
    
    private Map<String, Object> _doPOST(String api, Map<String, Object> parameters, String authToken, String userId) throws IOException
    {
        String path = Config.getInstance().getValue(CONFIG_URL) + "/api/" + api;
        
        HttpPost request = new HttpPost(path);
        
        String json = _jsonUtils.convertObjectToJson(parameters);
        request.setEntity(new StringEntity(json, ContentType.create("application/json", StandardCharsets.UTF_8)));
        request.setHeader("Content-Type", "application/json");
        
        return _execRequest(request, authToken, userId);
    }
    
    private Map<String, Object> _doMultipartPOST(String api, Map<String, Object> parameters) throws IOException
    {
        return _doMultipartPOST(api, parameters, Config.getInstance().getValue(CONFIG_ADMIN_TOKEN), Config.getInstance().getValue(CONFIG_ADMIN_ID));
    }
    
    private Map<String, Object> _doMultipartPOST(String api, Map<String, Object> parameters, String authToken, String userId) throws IOException
    {
        String path = Config.getInstance().getValue(CONFIG_URL) + "/api/" + api;
        
        HttpPost request = new HttpPost(path);
        
        MultipartEntityBuilder builder = MultipartEntityBuilder.create();
        builder.setMode(HttpMultipartMode.RFC6532);
        
        for (Entry<String, Object> p : parameters.entrySet())
        {
            if (p.getValue() instanceof String)
            {
                builder.addTextBody(p.getKey(), (String) p.getValue(),  ContentType.create("text/plain", Consts.UTF_8));
            }
            else if (p.getValue() instanceof InputStream is)
            {
                byte[] imageAsBytes = IOUtils.toByteArray(is);
                ByteArrayInputStream bis = new ByteArrayInputStream(imageAsBytes);
                Tika tika = new Tika();
                String mimeType = tika.detect(imageAsBytes);
                
                builder.addBinaryBody(p.getKey(), bis, ContentType.create(mimeType), p.getKey());
            }
            else
            {
                throw new UnsupportedOperationException("Cannot post the type " + p.getValue().getClass().getName() + " for parameter " + p.getKey());
            }
        }

        HttpEntity multipart = builder.build();
        request.setEntity(multipart);

        return _execRequest(request, authToken, userId);
    }
    

    private boolean _isPOSTSucessful(String api, Map<String, Object> parameters) throws IOException
    {
        return _isOperationSuccessful(_doPOST(api, parameters));
    }

    private Map<String, Object> _execRequest(HttpUriRequest request, String authToken, String userId) throws IOException
    {
        request.setHeader("X-Auth-Token", authToken);
        request.setHeader("X-User-Id", userId);

        try (CloseableHttpClient httpClient = HttpClients.createDefault();
             CloseableHttpResponse response = httpClient.execute(request))
        {
            Map<String, Object> convertJsonToMap = _jsonUtils.convertJsonToMap(EntityUtils.toString(response.getEntity()));
            return convertJsonToMap;
        }
    }
    
    private String _getError(Map<String, Object> info)
    {
        if (info.containsKey("error"))
        {
            return (String) info.get("error");
        }
        else if (info.containsKey("message"))
        {
            return (String) info.get("message");
        }
        else
        {
            return "";
        }
    }
    
    /**
     * Creates a new private group. If the group already exists the operation will succeed.
     * @param roomName name of the group
     * @param create Create if necessary
     * @return The group info. Can be null if no creation allowed
     * @throws IOException something went wrong
     */
    @SuppressWarnings("unchecked")
    public Map<String, Object> getRoom(String roomName, boolean create) throws IOException
    {
        Map<String, Object> groupInfo = _doGet("v1/groups.info", Map.of("roomName", roomName));
        if (!_isOperationSuccessful(groupInfo))
        {
            if (create)
            {
                groupInfo = _doPOST("v1/groups.create", Map.of("name", roomName));
                if (!_isOperationSuccessful(groupInfo))
                {
                    throw new IOException("Could not create room " + roomName + ", because " + _getError(groupInfo));
                }
            }
            else
            {
                return null;
            }
        }
        return (Map<String, Object>) groupInfo.get("group");
    }
    
    /**
     * Remove a private group. If the group does not exist the operation will succeed.
     * @param roomName name of the group
     * @throws IOException something went wrong
     */
    public void deleteRoom(String roomName) throws IOException
    {
        if (getRoom(roomName, false) != null)
        {
            Map<String, Object> deleteInfos = _doPOST("v1/groups.delete", Map.of("roomName", roomName));
            if (!_isOperationSuccessful(deleteInfos))
            {
                throw new IOException("Could not delete room " + roomName + ", because " + _getError(deleteInfos));
            }
        }
    }
    
    /**
     * Get (or create) a new user.
     * @param userIdentity the user that will be mirrored into chat
     * @param create Create if missing
     * @return the user info or null if user does not exist in chat and create was not required
     * @throws IOException something went wrong
     * @throws UserPreferencesException error while reading the user preferences
     */
    @SuppressWarnings("unchecked")
    public Map<String, Object> getUser(UserIdentity userIdentity, boolean create) throws IOException, UserPreferencesException
    {
        User user = _userManager.getUser(userIdentity);
        if (user == null)
        {
            throw new IllegalStateException("Cannot create user in Rocket.Chat for unexisting user " + UserIdentity.userIdentityToString(userIdentity));
        }
        
        Map<String, Object> userInfo = _doGet("v1/users.info", Map.of("username", userIdentitytoUserName(userIdentity)));
        if (!_isOperationSuccessful(userInfo))
        {
            if (create)
            {
                Map<String, String> ametysUserInfo = _getAmetysUserInfo(user);
                String userName = ametysUserInfo.get("userName");
                String userEmail = ametysUserInfo.get("userEmail");
                
                userInfo = _doPOST("v1/users.create", Map.of("username", userIdentitytoUserName(userIdentity),
                                                            "email", userEmail,
                                                            "name", userName,
                                                            "verified", true,
                                                            "password", _getUserPassword(userIdentity)));
                if (!_isOperationSuccessful(userInfo))
                {
                    throw new IllegalStateException("Cannot create user in Rocket.Chat for " + UserIdentity.userIdentityToString(userIdentity) + ": " +  _getError(userInfo));
                }
                
                getLogger().debug("User " + UserIdentity.userIdentityToString(userIdentity) + " created on the chat server");
                
                _updateAvatar(userIdentity);
            }
            else
            {
                return null;
            }
        }
        
        Map<String, Object> userMap = (Map<String, Object>) userInfo.get("user");
        
        _userPreferencesManager.addUserPreference(userIdentity, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_ID, (String) userMap.get("_id"));
        
        return userMap;
    }
    
    private Map<String, String> _getAmetysUserInfo(User user)
    {
        String userName = user.getFullName();
        String userEmail = user.getEmail();
        
        Content userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), null);
        if (userContent != null)
        {
            userName = StringUtils.defaultIfBlank(StringUtils.join(new String[] {userContent.getValue("firstname"), userContent.getValue("lastname")}, " "), userName);
            userEmail = StringUtils.defaultIfBlank(userContent.getValue("email"), userEmail);
        }

        return Map.of("userName", userName,
                      "userEmail", userEmail);
    }
    
    /**
     * Update user name, email, avatar... on chat server
     * @param userIdentity The user to update
     * @param changePassword Update the user password (slow)
     * @throws UserPreferencesException If an error occurred while getting user infos
     * @throws IOException If an error occurred while updating
     * @throws InterruptedException If an error occurred while updating
     */
    public void updateUserInfos(UserIdentity userIdentity, boolean changePassword) throws IOException, UserPreferencesException, InterruptedException
    {
        User user = _userManager.getUser(userIdentity);
        if (user == null)
        {
            throw new IllegalStateException("Cannot update user in Rocket.Chat for unexisting user " + UserIdentity.userIdentityToString(userIdentity));
        }
        
        Map<String, String> ametysUserInfo = _getAmetysUserInfo(user);
        String userName = ametysUserInfo.get("userName");
        String userEmail = ametysUserInfo.get("userEmail");
        
        Map<String, String> data = new HashMap<>();
        data.put("email", userEmail);
        data.put("name", userName);
        if (changePassword)
        {
            data.put("password", _getUserPassword(userIdentity));
        }
        
        Map<String, Object> updateInfos = _doPOST("v1/users.update", Map.of("userId", _getUserId(userIdentity),
                                                                            "data", data));
        if (!_isOperationSuccessful(updateInfos))
        {
            throw new IOException("Cannot update user " + UserIdentity.userIdentityToString(userIdentity) + " on chat server: " + _getError(updateInfos));
        }
        
        if (changePassword)
        {
            // When changing password, it unlogs people and this takes time
            Thread.sleep(1000);
        }
        
        _updateAvatar(userIdentity);
    }
    
    private void _updateAvatar(UserIdentity user)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute(AuthenticateAction.REQUEST_ATTRIBUTE_INTERNAL_ALLOWED, true);
        
        String oldPluginName = (String) request.getAttribute("pluginName");
        
        Source src = null;
        try
        {
            src = _sourceResolver.resolveURI("cocoon://_plugins/user-directory/user/" + user.getPopulationId() + "/" + user.getLogin() + "/image_64");
            try (InputStream is = src.getInputStream())
            {
                Map<String, Object> avatarInfo = _doMultipartPOST("v1/users.setAvatar", Map.of("username", userIdentitytoUserName(user),
                                                                                                "image", is));
                if (!_isOperationSuccessful(avatarInfo))
                {
                    getLogger().warn("Fail to update avatar for user " + UserIdentity.userIdentityToString(user) + ": " + _getError(avatarInfo));
                }
            }
        }
        catch (Exception e)
        {
            getLogger().warn("Fail to update avatar for user " + UserIdentity.userIdentityToString(user), e);
        }
        finally
        {
            _sourceResolver.release(src);
            request.setAttribute("pluginName", oldPluginName);
        }
        
    }
    
    private boolean _isUserInGroup(UserIdentity userIdentity, String roomName) throws IOException
    {
        Map<String, Object> getMembers = _doGet("v1/groups.members", Map.of("roomName", roomName));
        
        if (_isOperationSuccessful(getMembers))
        {
            @SuppressWarnings("unchecked")
            List<Map<String, String>> membersObject = (List<Map<String, String>>) getMembers.get("members");
            for (Map<String, String> map : membersObject)
            {
                if (userIdentitytoUserName(userIdentity).equalsIgnoreCase(map.get("username")))
                {
                    return true;
                }
            }
        }
        
        return false;
    }
    
    /**
     * Adds a user to the private group.
     * @param userIdentity user to invite
     * @param roomName name of the group where to invite the user
     * @throws IOException something went wrong
     * @throws UserPreferencesException If the user need to be created and its password cannot be stored
     */
    public void addUserToRoom(UserIdentity userIdentity, String roomName) throws IOException, UserPreferencesException
    {
        if (!_isUserInGroup(userIdentity, roomName))
        {
            Map<String, Object> userInfo = getUser(userIdentity, true);
            Map<String, Object> groupInfo = getRoom(roomName, true);
            
            String roomId = (String) groupInfo.get("_id");
            String userId = (String) userInfo.get("_id");
            
            Map<String, Object> inviteInfo = _doPOST("v1/groups.invite", Map.of("roomId", roomId,
                                                                                "userId", userId));
            
            if (!_isOperationSuccessful(inviteInfo))
            {
                throw new IOException("Could not add user " + UserIdentity.userIdentityToString(userIdentity) + " to room " + roomName + ": " + _getError(inviteInfo));
            }
        }
    }
    
    /**
     * Remove all existing user
     * @param roomName The room name
     * @param except Do not remove these users
     * @throws IOException If an error occurred
     */
    public void removeAllUsersFromRoom(String roomName, List<UserIdentity> except) throws IOException
    {
        List<String> exceptUsernames = except.stream().map(ChatHelper::userIdentitytoUserName).collect(Collectors.toList());
        
        Map<String, Object> groupInfo = getRoom(roomName, false);
        String roomId = (String) groupInfo.get("_id");
        
        Map<String, Object> membersInfo = _doGet("v1/groups.members", Map.of("roomId", roomId));
        if (_isOperationSuccessful(membersInfo))
        {
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> members = (List<Map<String, Object>>) membersInfo.get("members");
            for (Map<String, Object> member : members)
            {
                if (!exceptUsernames.contains(member.get("username")))
                {
                    _doPOST("v1/groups.kick", Map.of("roomId", roomId,
                                                 "userId", (String) member.get("_id")));
                }
            }
        }
    }
    
    /**
     * Removes a user from the private group.
     * @param user user to kick
     * @param roomName name of the group from where to kick the user
     * @return true if success
     * @throws IOException something went wrong
     * @throws UserPreferencesException If an error occurred with user password
     */
    public boolean removeUserFromRoom(UserIdentity user, String roomName) throws IOException, UserPreferencesException
    {
        if (_isUserInGroup(user, roomName))
        {
            Map<String, Object> userInfo = getUser(user, false);
            Map<String, Object> groupInfo = getRoom(roomName, false);
            
            if (userInfo != null && groupInfo != null)
            {
                String roomId = (String) groupInfo.get("_id");
                String userId = (String) userInfo.get("_id");
                
                return _isPOSTSucessful("v1/groups.kick", Map.of("roomId", roomId,
                                                                "userId", userId));
            }
            else
            {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Convert a useridentity to a chat server username
     * @param userIdentity The user to convert
     * @return the chat username
     */
    public static String userIdentitytoUserName(UserIdentity userIdentity)
    {
        return UserIdentity.userIdentityToString(userIdentity).replaceAll("[@# ]", "_");
    }
    
    private String _getUserPassword(UserIdentity userIdentity) throws UserPreferencesException
    {
        String cryptedPassword = _userPreferencesManager.getUserPreferenceAsString(userIdentity, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_PASSWORD);
        if (!StringUtils.isBlank(cryptedPassword))
        {
            try
            {
                return _cryptoHelper.decrypt(cryptedPassword);
            }
            catch (CryptoHelper.WrongKeyException e)
            {
                getLogger().warn("Password of user {} cannot be decrypted, and thus will be reset",  UserIdentity.userIdentityToString(userIdentity), e);
            }
        }
        
        return _generateAndStorePassword(userIdentity);
    }
    
    private String _getUserAuthToken(UserIdentity userIdentity) throws UserPreferencesException, IOException, InterruptedException
    {
        String cryptedToken = _userPreferencesManager.getUserPreferenceAsString(userIdentity, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_TOKEN);
        if (!StringUtils.isBlank(cryptedToken))
        {
            try
            {
                return _cryptoHelper.decrypt(cryptedToken);
            }
            catch (CryptoHelper.WrongKeyException e)
            {
                getLogger().warn("Token of user {} cannot be decrypted, and thus will be reset",  UserIdentity.userIdentityToString(userIdentity), e);
            }
        }
        return _generateAndStoreAuthToken(userIdentity, true);
    }
    
    
    private String _getUserId(UserIdentity userIdentity) throws UserPreferencesException
    {
        String userId = _userPreferencesManager.getUserPreferenceAsString(userIdentity, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_ID);
        return userId;
    }

    private String _generateAndStorePassword(UserIdentity user) throws UserPreferencesException
    {
        Double random = Math.random();
        byte[] randoms = {random.byteValue()};
        String randomPassword = Sha2Crypt.sha256Crypt(randoms);
        
        String cryptedPassword = _cryptoHelper.encrypt(randomPassword);
        _userPreferencesManager.addUserPreference(user, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_PASSWORD, cryptedPassword);
        
        return randomPassword;
    }
    
    
    @SuppressWarnings("unchecked")
    private String _generateAndStoreAuthToken(UserIdentity user, boolean tryToChangePassword) throws IOException, UserPreferencesException, InterruptedException
    {
        Map<String, Object> loginInfo = _doPOST("v1/login", Map.of("user", userIdentitytoUserName(user),
                                                                  "password", _getUserPassword(user)));
        if (_isOperationSuccessful(loginInfo))
        {
            String authToken = (String) ((Map<String, Object>) loginInfo.get("data")).get("authToken");
            String cryptedAuthToken = _cryptoHelper.encrypt(authToken);
            _userPreferencesManager.addUserPreference(user, __USERPREF_CONTEXT, Collections.emptyMap(), __USERPREF_PREF_TOKEN, cryptedAuthToken);
            
            return authToken;
        }
        else if (tryToChangePassword)
        {
            this.updateUserInfos(user, true);
            
            return _generateAndStoreAuthToken(user, false);
        }
        else
        {
            throw new IOException("Could not log user " + UserIdentity.userIdentityToString(user) + " into chat " + _getError(loginInfo));
        }
    }
    
    private String _getUserStatus(UserIdentity user, String authToken, String userId) throws IOException
    {
        Map<String, Object> statusInfo = _doGet("v1/users.getStatus", Map.of("user", userIdentitytoUserName(user)), authToken, userId);
        if (_isOperationSuccessful(statusInfo))
        {
            return (String) statusInfo.get("status");
        }
        else
        {
            return null;
        }
    }
    
    /**
     * Read the JSON result to test for success
     * @param result the JSON result of a rest call
     * @return true if success
     */
    protected boolean _isOperationSuccessful(Map<String, Object> result)
    {
        Boolean success = false;
        if (result != null)
        {
            Object successObj =  result.get("success");
            if (successObj instanceof Boolean)
            {
                success = (Boolean) successObj;
            }
            else if (successObj instanceof String)
            {
                success = "true".equalsIgnoreCase((String) successObj);
            }
            else
            {
                Object statusObj =  result.get("status");
                if (statusObj instanceof String)
                {
                    success = "success".equalsIgnoreCase((String) statusObj);
                }
            }
        }
        return success;
    }
    
    /**
     * Login the current user to the given room
     * @param roomName The room to log in
     * @return The info about the user
     * @throws UserPreferencesException If the user password stored in prefs has an issue
     * @throws IOException If an error occurred
     * @throws InterruptedException If an error occurred
     */
    @Callable
    public Map<String, Object> login(String roomName) throws IOException, UserPreferencesException, InterruptedException
    {
        // Get current user in Ametys
        UserIdentity user = _currentUserProvider.getUser();
        
        // Ensure user exists on chat server
        getUser(user, true);
        
        // Ensure the room exists on chat server
        getRoom(roomName, true);
        
        // Ensure the user is part of the room  on chat server
        addUserToRoom(user, roomName);
        
        // Get the login info of the user
        String authToken = _getUserAuthToken(user);
        String userId = _getUserId(user);
        
        // Ensure token validity by getting status on chat server
        String status = _getUserStatus(user, authToken, userId);
        if (status == null)
        {
            // If we cannot get status, this is probably because the auth token has expired or the user was recreated. Try a new one
            authToken = _generateAndStoreAuthToken(user, true);
            
            status = _getUserStatus(user, authToken, userId);
            
            if (status == null)
            {
                throw new IllegalStateException("Cannot get the status of user " + UserIdentity.userIdentityToString(user));
            }
        }
        
        return Map.of("userId", userId,
                      "authToken", authToken,
                      "userName", userIdentitytoUserName(user),
                      "status", status);
    }
}
