/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.chat;

import java.io.IOException;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.logger.AbstractLogEnabled;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;

/**
 * Kick a Chat user from a room when it is removed from a project
 */
public class MemberRemovedObserver extends AbstractLogEnabled implements Observer, Serviceable 
{
    /** Chat helper */
    protected ChatHelper _chatHelper;
    private GroupManager _groupManager;
    private ProjectMemberManager _projectMemberManager;

    public void service(ServiceManager manager) throws ServiceException
    {
        _chatHelper = (ChatHelper) manager.lookup(ChatHelper.ROLE);
        _groupManager = (GroupManager) manager.lookup(GroupManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
    }

    public boolean supports(Event event)
    {
        return ObservationConstants.EVENT_MEMBER_DELETED.equals(event.getId()) && Config.getInstance().<Boolean>getValue("workspaces.chat.active");
    }

    public int getPriority(Event event)
    {
        return MAX_PRIORITY; // Yes, "MAX PRIORITY" is the lowest priority...
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Project project = (Project) event.getArguments().get(ObservationConstants.ARGS_PROJECT);

        String memberIdentity = (String) event.getArguments().get(ObservationConstants.ARGS_MEMBER_IDENTITY);
        MemberType memberType = (MemberType) event.getArguments().get(ObservationConstants.ARGS_MEMBER_IDENTITY_TYPE);

        Set<UserIdentity> users;
        
        if (memberType == MemberType.USER)
        {
            users = Set.of(UserIdentity.stringToUserIdentity(memberIdentity));
        }
        else
        {
            GroupIdentity groupIdentity = GroupIdentity.stringToGroupIdentity(memberIdentity);
            Group group = _groupManager.getGroup(groupIdentity);
            users = group.getUsers();
        }
        
        Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, true);
        Set<UserIdentity> stillMembers = projectMembers.stream().map(m -> m.getUser().getIdentity()).collect(Collectors.toSet());
        
        for (UserIdentity user : users)
        {
            // A user can be still member via another group or directly...
            if (!stillMembers.contains(user))
            {
                try
                {
                    _chatHelper.removeUserFromRoom(user, project.getName());
                }
                catch (IOException e)
                {
                    getLogger().error("An error occurred that prevented the user " + UserIdentity.userIdentityToString(user) + " to be removed from the room " + project.getName(), e);
                }
            }
        }
    }

}
