/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.dav;

import java.io.InputStream;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.lock.Lock;
import javax.jcr.lock.LockManager;
import javax.servlet.http.HttpServletRequest;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.acting.AbstractAction;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.environment.http.HttpResponse;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.resources.ModifiableResource;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationMode;
import org.ametys.plugins.explorer.resources.actions.AddOrUpdateResourceHelper.ResourceOperationResult;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.jcr.JCRAmetysObject;
import org.ametys.plugins.repository.lock.LockHelper;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * Action for WebDAV PUT method
 *
 */
public class WebdavPutAction extends AbstractAction implements Serviceable
{
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** Observer manager. */
    protected ObservationManager _observationManager;

    private AmetysObjectResolver _resolver;

    private AddOrUpdateResourceHelper _addOrUpdateResourceHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _observationManager = (ObservationManager) smanager.lookup(ObservationManager.ROLE);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _addOrUpdateResourceHelper = (AddOrUpdateResourceHelper) smanager.lookup(AddOrUpdateResourceHelper.ROLE);
    }

    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        HttpServletRequest request = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);

        ModifiableResource resource = (ModifiableResource) request.getAttribute("resource");

        if (resource != null)
        {
            // Resource exists, it is updated
            if (resource.isLocked() && !LockHelper.isLockOwner(resource, _currentUserProvider.getUser()))
            {
                HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
                response.setStatus(423);
                return null;
            }

            if (resource instanceof JCRAmetysObject)
            {
                Node node = ((JCRAmetysObject) resource).getNode();

                if (node.isLocked())
                {
                    LockManager lockManager = node.getSession().getWorkspace().getLockManager();

                    Lock lock = lockManager.getLock(node.getPath());
                    Node lockHolder = lock.getNode();

                    lockManager.addLockToken(lockHolder.getProperty(RepositoryConstants.METADATA_LOCKTOKEN).getString());
                }
            }

            InputStream is = request.getInputStream();

            UserIdentity author = _currentUserProvider.getUser();
            String mimeType = resource.getMimeType();
            mimeType = mimeType == null ? "application/unknown" : mimeType;

            resource.setData(is, mimeType, new Date(), author);

            resource.saveChanges();
            _checkpoint(resource);

            Map<String, Object> eventParams = new HashMap<>();

            eventParams.put(ObservationConstants.ARGS_ID, resource.getId());
            eventParams.put(ObservationConstants.ARGS_NAME, resource.getName());
            eventParams.put(ObservationConstants.ARGS_PATH, resource.getPath());
            eventParams.put(ObservationConstants.ARGS_RESOURCE_PATH, resource.getResourcePath());
            AmetysObject parent = resource.getParent();
            if (parent != null)
            {
                eventParams.put(ObservationConstants.ARGS_PARENT_ID, parent.getId());
                eventParams.put(ObservationConstants.ARGS_PARENT_PATH, parent.getPath());
            }
            _observationManager.notify(new Event(ObservationConstants.EVENT_RESOURCE_UPDATED, _currentUserProvider.getUser(), eventParams));
        }
        else
        {
            // Resource needs to be created.
            String filePath = parameters.getParameter("path");
            int lastSlashPos = filePath.lastIndexOf('/');
            if (lastSlashPos > 0)
            {
                String folderPath = filePath.substring(0, lastSlashPos);
                String fileName = filePath.substring(lastSlashPos + 1);
                Project project = (Project) request.getAttribute("project");
                AmetysObject folderAmetysObject = _resolver.resolveByPath(project.getPath() + "/ametys-internal:resources/" + folderPath);
                if (folderAmetysObject instanceof ModifiableResourceCollection)
                {
                    ModifiableResourceCollection folder = (ModifiableResourceCollection) folderAmetysObject;
                    _addOrUpdateResourceHelper.checkAddResourceRight(folder);
                    
                    InputStream is = request.getInputStream();
                    ResourceOperationResult operationResult = _addOrUpdateResourceHelper.performResourceOperation(is, fileName, folder, ResourceOperationMode.ADD);
                    if (operationResult.isSuccess())
                    {
                        resource = (ModifiableResource) operationResult.getResource();
                    }
                    else
                    {
                        getLogger().error("User '" + _currentUserProvider.getUser() + "' try create a ressource named '" + fileName + "' in parent '" + folder.getId() + "' but it failed with message : " + operationResult.getErrorMessage());
                        HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
                        int status = 500;
                        switch (operationResult.getErrorMessage())
                        {
                            case "locked":
                            case "locked-file":
                                status = 423; // locked
                                break;
                            case "already-exist":
                                status = 409; // conflict
                                break;
                            case "error":
                            default:
                                status = 500;
                        }
                        response.setStatus(status);
                        return EMPTY_MAP;
                    }
                }
                else
                {
                    getLogger().error("User '" + _currentUserProvider.getUser() + "' try create a ressource named '" + fileName + "' in parent '" + folderAmetysObject.getId() + "' but the parent is not a ModifiableResourceCollection");
                    HttpResponse response = (HttpResponse) ObjectModelHelper.getResponse(objectModel);
                    response.setStatus(409);
                    return EMPTY_MAP;
                }
            }
        }


        return EMPTY_MAP;
    }

    private void _checkpoint(ModifiableResource resource)
    {
        if (resource instanceof VersionableAmetysObject)
        {
            // Create first version
            ((VersionableAmetysObject) resource).checkpoint();
        }
    }
}
